<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * СТРАНИЦА ЭМИТЕНТА (EMITENT) v8.5.1 COMPACT
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 8.4.0 CLEAN
 * Дата: 2025-11-02
 * Статус: PRODUCTION READY ✅ COMPACT CARDS + 3 COLUMNS
 * 
 * 
 * 
 * ИЗМЕНЕНИЯ v8.5.1 COMPACT:
 *   📏 COMPACT: Компактный формат параметров (2 строки)
 *   ✨ АКЦИИ: "Номинал: 3.00 ₽ • Лот: 1" + "Эмиссия, шт: 21,6 млрд"
 *   ✨ ОБЛИГАЦИИ: "Погашение: 15.12.2025" (без дней) + "💰 0.50 ₽ • Купон 1833"
 *   🏷️ TITLE: "{Instrument_TYPE} {CODE} от эмитента {EMITENT_SHORT_NAME}"
 *   🎨 ФОРМАТ: Запятая в дробной части млрд (21,6)
 *
 *   🎨 ФОРМАТ: Запятая как разделитель дробной части в млрд (21,6)
 *
 *
 * 
 * ИЗМЕНЕНИЯ v8.5.0 DETAILED:
 *   ✨ NEW: Детальная информация в карточках инструментов
 *   ✨ NEW: Иконки уровней листинга (🏆🥈🥉)
 *   ✨ NEW: Grid 3 колонки для компактного отображения (→2→1 responsive)
 *   ✨ NEW: Маппинг $listingLevels для иконок листинга
 *   📊 АКЦИИ: Показываем Номинал + Лот + Объём выпуска (если заполнены)
 *   📊 ОБЛИГАЦИИ: Показываем Дату погашения + Купон + Период + След.купон + Дни до погашения
 *   🎯 УМНЫЕ ПРОВЕРКИ: Не выводим параметр если значение пустое/0/NULL
 *   🎨 ФОРМАТ: Объём в млн/млрд, купон с периодом, иконки для дат
 *   ♿ A11Y: aria-hidden="true" на всех эмоджи
 *   🔧 CSS: Требуется .card-param стили в best.css
 *   📏 ДИНАМИЧЕСКАЯ ВЫСОТА: Карточки разной высоты (нормально для grid)
 * ИЗМЕНЕНИЯ v8.4.0 CLEAN:
 *   🗑️ REMOVED: Полностью удалён механизм подсчёта инструментов
 *   🗑️ REMOVED: Функция formatInstrumentsList()
 *   🗑️ REMOVED: Функция getDeclension()
 *   🗑️ REMOVED: SQL - stocks_count и bonds_count
 *   🗑️ REMOVED: SQL - JOIN с QUIK_Akcii и QUIK_Obligacii
 *   🗑️ REMOVED: SQL - JOIN с Instrument
 *   🗑️ REMOVED: HTML - вывод "(X акций, Y облигаций)"
 *   ✅ SQL: Упрощён до базовых полей + JOIN только с SECTOR_ECONOMIKI
 *   ✅ HTML: Блок перелинковки показывает только названия эмитентов
 * 
 * ИЗМЕНЕНИЯ v8.3.0 FIX:
 *   ✅ FIX: Исправлен подсчёт инструментов в блоке перелинковки
 *   ✅ SQL: Считаются ТОЛЬКО инструменты с данными в QUIK таблицах
 *   ✅ Акции: типы (1,2,4) которые ЕСТЬ в QUIK_Akcii
 *   ✅ Облигации: типы (3,5,6) которые ЕСТЬ в QUIK_Obligacii
 *   ✅ Правильное склонение: "2 акции, 5 облигаций"
 * 
 * ИЗМЕНЕНИЯ v8.2.0 (частично откачены):
 *   ❌ GROUP_CONCAT удалён - заменён на прямой подсчёт в SQL
 *   ✅ Функция getDeclension() - сохранена
 *   ✅ CSS классы .emitent-name, .instruments-info - сохранены
 * 
 * ИЗМЕНЕНИЯ v8.2.0 (оригинальные):
 *   ✅ Детальная информация о типах инструментов: "(1 акция, 4 облигации)"
 *   ✅ Функция getDeclension() - склонение для русского языка
 *   ✅ SQL: GROUP_CONCAT для получения типов инструментов
 *   ✅ Выравнивание названий эмитентов по левому краю с отступом
 *   ✅ Новые CSS классы: .emitent-name, .instruments-info
 * 
 * ИЗМЕНЕНИЯ v8.1.0 FINAL (сохранены):
 * ИЗМЕНЕНИЯ v8.1.0 FINAL:
 *   ✅ FIX: Исправлена ошибка global $db в getTopEmitentsBySector()
 *   ✅ Блок перелинковки перемещён в sidebar (после "Официальные источники")
 *   ✅ Новый заголовок блока: "Популярные агрокомпании и эмитенты на рынке"
 *   ✅ Title ссылок использует SECTOR_TITLE_SUFFIX из таблицы SECTOR_ECONOMIKI
 *   ✅ SQL запрос включает JOIN с SECTOR_ECONOMIKI
 *   ✅ Badge с количеством инструментов (класс .badge-count)
 *   ✅ Fallback title: "эмитент российского фондового рынка"
 * 
 * ИЗМЕНЕНИЯ v8.0.0 SEO (сохранены):
 *   ♿ A11Y: Добавлен aria-hidden="true" на ВСЕ декоративные эмоджи
 *   🎯 SEO: Убраны H3 из sidebar (замена на .sidebar-title)
 *   🔒 CSP: Убраны inline styles (замена на CSS классы)
 *   ✨ NEW: Функция renderDefinitionList() для будущего использования
 *   ✨ NEW: Функция getTopEmitentsBySector() для блока перелинковки
 *   ✨ NEW: Блок "Лучшие эмитенты сектора" с проверками
 *   🔧 FIX: H3 в subsection-title заменён на div
 *   📊 Улучшение SEO с 8/10 → 10/10
 * 
 * ИЗМЕНЕНИЯ v7.1.5 FINAL (сохранены):
 *   🔧 FIX: Изменено название типа 1 с "Акции обыкновенные" на "Акции"
 *   📝 Теперь метрика показывает "Акции (2)" вместо "Акции обыкновенные (2)"
 *   📝 Раздел называется "Акции (2)" - обобщённо для всех типов акций
 *   💡 Причина: У эмитента бывают разные акции (обыкновенные + привилегированные)
 * 
 * ИЗМЕНЕНИЯ v7.1.4 PERFECTED (сохранены):
 *   ✨ Добавлена проверка валидности типа инструмента (1-6) в renderInstrumentCard()
 *   ✨ Добавлен fallback для названия эмитента ('Эмитент' если нет SHORT_NAME и FULL_NAME)
 *   📝 Улучшено логирование ошибок для отладки
 * 
 * КРИТИЧНЫЕ ИСПРАВЛЕНИЯ v7.0.3 SECURE (сохранены):
 *   🔒 validateImagePath() - защита от Path Traversal
 *   🔒 validateExternalUrl() - защита от JavaScript URL
 *   🔒 generateSafeElementId() - транслитерация кириллицы
 *   🔒 renderIdentifier() - htmlspecialchars(ENT_QUOTES)
 *   🔒 Все числа приведены к (int)
 *   🔒 Валидация URL slug
 * 
 * ⚠️ ТРЕБУЕТ:
 *   - PHP 8.1+
 *   - DB/emitents.php v2.7+
 *   - DB/instruments.php v2.0+ (с getInstrumentsByEmitentId)
 *   - /css/best.css v7.9.3+ (с .sidebar-title, .emitent-meta, .badge-count)
 *   - Таблица SECTOR_ECONOMIKI с полем SECTOR_TITLE_SUFFIX
 *   - /js/stock-v3.1.3.js
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */

// ═══════════════════════════════════════════════════════════════════════════
// ПРОВЕРКА ЗАВИСИМОСТЕЙ
// ═══════════════════════════════════════════════════════════════════════════

if (!function_exists('esc')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция esc() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

if (!function_exists('isEmpty')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция isEmpty() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

if (!function_exists('renderLongDescription')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция renderLongDescription() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

// ═══════════════════════════════════════════════════════════════════════════
// ВСПОМОГАТЕЛЬНЫЕ ФУНКЦИИ
// ═══════════════════════════════════════════════════════════════════════════

/**
 * Валидация пути к изображению (защита от Path Traversal)
 * 
 * @param string $imagePath - Путь к изображению
 * @return string - Безопасный путь или пустая строка
 */
function validateImagePath($imagePath) {
    if (isEmpty($imagePath)) return '';
    
    // Удаляем опасные символы
    $imagePath = str_replace(['../', '..\\', '\\', "\0"], '', $imagePath);
    
    // Проверяем расширение файла
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
    $extension = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));
    
    if (!in_array($extension, $allowedExtensions, true)) {
        return '';
    }
    
    // Проверяем что путь содержит только безопасные символы
    if (!preg_match('/^[a-zA-Z0-9_\-\.\/]+$/', $imagePath)) {
        return '';
    }
    
    return $imagePath;
}

/**
 * Валидация URL (защита от JavaScript URL атак)
 * 
 * @param string $url - Внешний URL
 * @return string - Безопасный URL или пустая строка
 */
function validateExternalUrl($url) {
    if (isEmpty($url)) return '';
    
    // Очищаем от управляющих символов
    $url = trim($url);
    $url = preg_replace('/[\x00-\x1F\x7F]/u', '', $url);
    
    // Проверяем что URL начинается с http:// или https://
    if (!preg_match('/^https?:\/\//i', $url)) {
        return '';
    }
    
    // Защита от опасных схем
    $dangerousSchemes = ['javascript:', 'data:', 'vbscript:', 'file:', 'about:'];
    foreach ($dangerousSchemes as $scheme) {
        if (stripos($url, $scheme) !== false) {
            return '';
        }
    }
    
    // Финальная проверка через filter_var
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        return '';
    }
    
    return $url;
}

/**
 * Генерация безопасного ID для элемента
 * 
 * Транслитерирует кириллицу в латиницу для использования в HTML атрибутах
 * 
 * @param string $id - Исходный ID (может содержать кириллицу)
 * @return string - Безопасный ID для HTML
 */
function generateSafeElementId($id) {
    // Таблица транслитерации кириллицы
    $translitMap = [
        'а' => 'a', 'б' => 'b', 'в' => 'v', 'г' => 'g', 'д' => 'd', 
        'е' => 'e', 'ё' => 'yo', 'ж' => 'zh', 'з' => 'z', 'и' => 'i',
        'й' => 'y', 'к' => 'k', 'л' => 'l', 'м' => 'm', 'н' => 'n',
        'о' => 'o', 'п' => 'p', 'р' => 'r', 'с' => 's', 'т' => 't',
        'у' => 'u', 'ф' => 'f', 'х' => 'h', 'ц' => 'ts', 'ч' => 'ch',
        'ш' => 'sh', 'щ' => 'sch', 'ъ' => '', 'ы' => 'y', 'ь' => '',
        'э' => 'e', 'ю' => 'yu', 'я' => 'ya'
    ];
    
    // Приводим к нижнему регистру
    $safeId = mb_strtolower($id, 'UTF-8');
    
    // Транслитерируем
    $safeId = strtr($safeId, $translitMap);
    
    // Убираем специальные символы
    $safeId = str_replace([' ', '.', '№', '_'], ['', '', 'num', ''], $safeId);
    
    // Оставляем только буквы, цифры и дефисы
    $safeId = preg_replace('/[^a-z0-9\-]/', '', $safeId);
    
    // Если ID пустой - генерируем из hash
    if (empty($safeId)) {
        $safeId = 'id-' . substr(md5($id), 0, 8);
    } 
    // Если ID начинается не с буквы - добавляем префикс
    elseif (!preg_match('/^[a-z]/', $safeId)) {
        $safeId = 'id-' . $safeId;
    }
    
    return $safeId . '-value';
}

/**
 * Рендер definition list для параметров
 * 
 * v8.0.0 NEW - Добавлена для будущего использования (из stock.php)
 * Для семантичных пар термин-значение
 * 
 * @param array $items Ассоциативный массив [термин => значение]
 * @return array ['html' => string, 'count' => int]
 */
function renderDefinitionList($items) {
    if (empty($items)) return ['html' => '', 'count' => 0];
    
    $count = 0;
    $html = '';
    
    foreach ($items as $term => $definition) {
        if (!isEmpty($definition)) {
            $escapedTerm = esc($term);
            $escapedDefinition = esc($definition);
            $html .= "<dt>{$escapedTerm}</dt>\n";
            $html .= "<dd>{$escapedDefinition}</dd>\n";
            $count++;
        }
    }
    
    if ($count === 0) return ['html' => '', 'count' => 0];
    
    return [
        'html' => "<dl>\n{$html}</dl>",
        'count' => $count
    ];
}

/**
 * Получить топовых эмитентов из того же сектора для перелинковки
 * 
 * v8.4.0 CLEAN: Удалён механизм подсчёта инструментов
 * v8.3.1 FIX: Исправлен подсчёт - ТОЛЬКО тип 1 (акции) и тип 6 (облигации)
 * v8.3.0 FIX: Исправлен подсчёт инструментов (только с данными QUIK)
 * v8.2.0 UPDATE: GROUP_CONCAT удалён, добавлены stocks_count/bonds_count
 * v8.1.0 UPDATE: Добавлен JOIN с SECTOR_ECONOMIKI для получения SECTOR_TITLE_SUFFIX
 * v8.1.0 FIX: Удалён конфликтующий global $db
 * 
 * v8.3.0 FIX: Исправлен подсчёт инструментов (только с данными QUIK)
 * v8.2.0 UPDATE: GROUP_CONCAT удалён, добавлены stocks_count/bonds_count
 * v8.1.0 UPDATE: Добавлен JOIN с SECTOR_ECONOMIKI для получения SECTOR_TITLE_SUFFIX
 * v8.1.0 FIX: Удалён конфликтующий global $db
 * 
 * @param object $db - Подключение к БД
 * @param int $sector_id - ID сектора текущего эмитента
 * @param int $current_emitent_id - ID текущего эмитента (исключить из результатов)
 * @return array - Массив объектов эмитентов
 */
function getTopEmitentsBySector($db, $sector_id, $current_emitent_id) {
    $sql = "
        SELECT 
            e.Id,
            e.EMITENT_SHORT_NAME,
            e.EMITENT_FULL_NAME,
            e.EMITENT_URL,
            e.EMITENT_og_image,
            e.SECTOR_ECONOMIKI,
            s.SECTOR_TITLE_SUFFIX
        FROM Emitent e
        LEFT JOIN SECTOR_ECONOMIKI s ON e.ID_SECTOR_ECONOMIKI = s.Id
        WHERE e.TOP = 1
          AND e.ID_SECTOR_ECONOMIKI = ?
          AND e.Id != ?
        ORDER BY e.EMITENT_SHORT_NAME ASC
        LIMIT 5
    ";
    
    $stmt = $db->connection->prepare($sql);
    if (!$stmt) {
        error_log("[EMITENT.PHP v8.4.0] getTopEmitentsBySector - Ошибка prepare: " . $db->connection->error);
        return [];
    }
    
    $stmt->bind_param("ii", $sector_id, $current_emitent_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $emitents = [];
    while ($row = $result->fetch_object()) {
        $emitents[] = $row;
    }
    
    $stmt->close();
    
    return $emitents;
}

/**
 * Форматировать список инструментов эмитента
 * 
 * v8.2.0 NEW - Детальная информация о типах инструментов
 * 
 * @param string $instrument_types - Строка с типами инструментов через запятую (из GROUP_CONCAT)
 * @return string - Отформатированная строка типа "(1 акция, 4 облигации)"
 */

/**
 * Получить правильную форму склонения для числа (русский язык)
 * 
 * v8.2.0 NEW - Склонение для русского языка
 * 
 * @param int $number - Число
 * @param array $forms - Массив форм [одна, две, пять]
 * @return string - Правильная форма слова
 */


// ═══════════════════════════════════════════════════════════════════════════
// МАППИНГ УРОВНЕЙ ЛИСТИНГА (v8.5.0)
// ═══════════════════════════════════════════════════════════════════════════

// Маппинг уровней листинга к иконкам (аналогично bonds-catalog.php)
$listingLevels = [
    1 => '🏆',  // Первый уровень
    2 => '🥈',  // Второй уровень
    3 => '🥉'   // Третий уровень
];

/**
 * Рендер карточки идентификатора с кнопкой копирования
 * 
 * @param string $id - ID элемента
 * @param string $label - Название идентификатора (ИНН, LEI и т.д.)
 * @param string $value - Значение идентификатора
 * @return string - HTML карточки или пустая строка
 */
function renderIdentifier($id, $label, $value) {
    if (isEmpty($value) || $value === '0') return '';
    
    // Экранируем значения
    $escapedValue = esc($value);
    $elementId = generateSafeElementId($id);
    $escapedLabel = esc($label);
    
    // Дополнительное экранирование для атрибутов
    $attrSafeLabel = htmlspecialchars($escapedLabel, ENT_QUOTES, 'UTF-8');
    $attrSafeValue = htmlspecialchars($escapedValue, ENT_QUOTES, 'UTF-8');
    
    return <<<HTML
<div class="id-card">
    <div class="id-label">{$escapedLabel}</div>
    <div class="id-value" id="{$elementId}">{$escapedValue}</div>
    <button class="copy-btn" 
            data-target="{$elementId}"
            data-label="{$attrSafeLabel}"
            type="button"
            aria-label="Скопировать {$attrSafeLabel}: {$attrSafeValue}"
            title="Скопировать {$attrSafeLabel} {$attrSafeValue}">
        📋 Копировать
    </button>
</div>
HTML;
}

/**
 * Получить название типа инструмента
 * 
 * v7.1.5 FINAL: Изменено название типа 1 с "Акции обыкновенные" на "Акции"
 * 
 * @param int $typeId - ID типа (1-6)
 * @return string - Название типа
 */
function getInstrumentTypeName($typeId) {
    $types = [
        1 => 'Акции',                    // v7.1.5: было "Акции обыкновенные"
        2 => 'Депозитарные расписки',
        3 => 'Еврооблигации',
        4 => 'Инвестиционные паи',
        5 => 'Ипотечные сертификаты',
        6 => 'Облигации'
    ];
    return $types[$typeId] ?? 'Инструменты';
}

/**
 * Получить иконку типа инструмента
 * 
 * @param int $typeId - ID типа (1-6)
 * @return string - Эмодзи иконка
 */
function getInstrumentTypeIcon($typeId) {
    $icons = [
        1 => '📈',
        2 => '📊',
        3 => '💶',
        4 => '📑',
        5 => '🏠',
        6 => '💸'
    ];
    return $icons[$typeId] ?? '📄';
}

/**
 * Получить URL slug типа инструмента
 * 
 * @param int $typeId - ID типа (1-6)
 * @return string - URL slug
 */
function getInstrumentTypeSlug($typeId) {
    $slugs = [
        1 => 'stocks',
        2 => 'receipts',
        3 => 'eurobonds',
        4 => 'funds',
        5 => 'mortgage-notes',
        6 => 'bonds'
    ];
    return $slugs[$typeId] ?? 'instruments';
}

/**
 * Рендер карточки инструмента
 * 
 * v8.5.0 DETAILED: Детальная информация с проверками на пустые значения
 * v7.1.4 PERFECTED: Добавлена проверка валидности типа инструмента
 * 
 * @param array $inst - Данные инструмента из БД
 * @param bool $hasQuikData - Есть ли данные в QUIK таблицах
 * @return string - HTML карточки инструмента
 */
function renderInstrumentCard($inst, $hasQuikData) {
    global $listingLevels, $emitent;
    
    $typeId = (int)($inst['Instrument_SUPERTYPE_ID'] ?? 0);
    
    // v7.1.4 PERFECTED: Проверка валидности типа инструмента (1-6)
    if ($typeId < 1 || $typeId > 6) {
        $isin = $inst['Instrument_ISIN'] ?? 'unknown';
        $code = $inst['Instrument_TRADE_CODE'] ?? 'unknown';
        error_log('[EMITENT.PHP v8.5.0] Неизвестный тип инструмента: typeId=' . $typeId . ', ISIN=' . $isin . ', CODE=' . $code);
        return ''; // Не рендерим карточку с некорректным типом
    }
    
    $slug = getInstrumentTypeSlug($typeId);
    
    // v8.5.0: Получаем уровень листинга и иконку
    $listingId = (int)($inst['Instrument_LIST_SECTION_ID'] ?? 0);
    $listingIcon = isset($listingLevels[$listingId]) ? ' <span aria-hidden="true">' . $listingLevels[$listingId] . '</span>' : '';
    
    // Определяем откуда брать данные
    if ($hasQuikData) {
        // Есть QUIK данные
        $isEquity = !empty($inst['Akcii_SecCode']); // Акции/расписки/паи
        
        if ($isEquity) {
            // ═══════════════════════════════════════════════════════
            // АКЦИИ (типы 1, 2, 4) - из QUIK_Akcii
            // ═══════════════════════════════════════════════════════
            $code = esc($inst['Akcii_SecCode']);
            $name = esc($inst['Akcii_ShortName'] ?? $inst['Akcii_SecName'] ?? '');
            $status = $inst['Akcii_SecStatus'] ?? '';
            
            // v8.5.0 DETAILED: Собираем детальную информацию для акций
            $details = [];
            
            // Строка 1: Номинал + Лот (в одну строку через •)
            $faceValue = (float)($inst['Akcii_FaceValue'] ?? 0);
            $lotSize = (int)($inst['Akcii_LotSize'] ?? 0);
            
            $line1Parts = [];
            if ($faceValue > 0) {
                $faceUnit = $inst['Akcii_FaceUnit'] ?? 'SUR';
                $currency = ($faceUnit === 'SUR') ? '₽' : esc($faceUnit);
                $line1Parts[] = 'Номинал: ' . number_format($faceValue, 2, '.', ' ') . ' ' . $currency;
            }
            if ($lotSize > 0) {
                $line1Parts[] = 'Лот: ' . number_format($lotSize, 0, '.', ' ');
            }
            
            if (!empty($line1Parts)) {
                $details[] = '<div class="card-param">' . implode(' • ', $line1Parts) . '</div>';
            }
            
            // Строка 2: Эмиссия (объём выпуска)
            $issueSize = (int)($inst['Akcii_IssueSize'] ?? 0);
            if ($issueSize > 0) {
                if ($issueSize >= 1000000000) {
                    // Миллиарды
                    $formatted = number_format($issueSize / 1000000000, 1, ',', ' ') . ' млрд';
                } elseif ($issueSize >= 1000000) {
                    // Миллионы
                    $formatted = number_format($issueSize / 1000000, 1, ',', ' ') . ' млн';
                } else {
                    // Просто число
                    $formatted = number_format($issueSize, 0, '.', ' ');
                }
                $details[] = '<div class="card-param">Эмиссия, шт: ' . $formatted . '</div>';
            }
            
            $extraInfo = implode('', $details);
            
        } else {
            // ═══════════════════════════════════════════════════════
            // ОБЛИГАЦИИ (типы 3, 5, 6) - из QUIK_Obligacii
            // ═══════════════════════════════════════════════════════
            $code = esc($inst['Obligacii_SecCode']);
            $name = esc($inst['Obligacii_ShortName'] ?? $inst['Obligacii_SecName'] ?? '');
            $status = $inst['Obligacii_SecStatus'] ?? '';
            
            // v8.5.0 DETAILED: Собираем детальную информацию для облигаций
            $details = [];
            
            // Строка 1: Погашение (дата + дни до погашения в одну строку)
            $matDate = $inst['Obligacii_MatDate'] ?? '';
            $daysToMat = (int)($inst['Obligacii_DaysToMatDate'] ?? 0);
            
            if (!empty($matDate) && $matDate !== '0000-00-00') {
                $date = DateTime::createFromFormat('Y-m-d', $matDate);
                if ($date) {
                    $dateFormatted = $date->format('d.m.Y');
                    $details[] = '<div class="card-param">Погашение: ' . $dateFormatted . '</div>';
                }
            }
            
            // Строка 2: Купон + период
            $coupon = (float)($inst['Obligacii_CouponValue'] ?? 0);
            $couponPeriod = (int)($inst['Obligacii_CouponPeriod'] ?? 0);
            if ($coupon > 0) {
                $couponText = '<span aria-hidden="true">💰</span> ' . number_format($coupon, 2, '.', ' ') . ' ₽';
                if ($couponPeriod > 0) {
                    $couponText .= ' • Купон ' . $couponPeriod;
                }
                $details[] = '<div class="card-param">' . $couponText . '</div>';
            }
            
            $extraInfo = implode('', $details);
        }
    } else {
        // Нет QUIK данных - используем только таблицу Instrument
        $code = esc($inst['Instrument_TRADE_CODE'] ?? '');
        $name = esc($inst['Instrument_ISIN'] ?? '');
        $status = '';
        $extraInfo = '';
    }
    
    // Если нет кода - не рендерим карточку
    if (isEmpty($code)) return '';
    
    if ($hasQuikData) {
        // Активная карточка с ссылкой на страницу инструмента
        $url = (defined('SITE_URL') ? SITE_URL : '/') . $slug . '/' . $code . '/';
        
        // Определяем класс для статуса (зеленая/оранжевая рамка)
        $statusClass = '';
        if (mb_strtolower($status) === 'торгуется') {
            $statusClass = ' active'; // Зеленая рамка
        } elseif (!isEmpty($status) && mb_strtolower($status) !== 'торгуется') {
            $statusClass = ' paused'; // Оранжевая рамка
        }
        
        // v8.5.1 COMPACT: Title с типом инструмента и эмитентом
        $instrumentType = esc($inst['Instrument_TYPE'] ?? '');
        $emitentName = !isEmpty($emitent->EMITENT_SHORT_NAME) ? esc($emitent->EMITENT_SHORT_NAME) : '';
        
        // Формируем title: "Акция обыкновенная SBER от эмитента ПАО Сбербанк"
        $titleText = $instrumentType . ' ' . $code;
        if (!isEmpty($emitentName)) {
            $titleText .= ' от эмитента ' . $emitentName;
        }
        
        // v8.5.2: Используем конкатенацию вместо heredoc чтобы избежать warning от CSS валидатора
        $cardClass = 'instrument-card' . $statusClass;
        return '<a href="' . $url . '" class="' . $cardClass . '" title="' . $titleText . '">' .
            '<div class="instrument-code">' . $code . $listingIcon . '</div>' .
            '<div class="instrument-name">' . $name . '</div>' .
            $extraInfo .
            '</a>';
    } else {
        // Серая карточка БЕЗ ссылки (нет QUIK данных)
        // v8.5.2: Используем конкатенацию вместо heredoc
        return '<div class="instrument-card no-data">' .
            '<div class="instrument-code">' . $code . '</div>' .
            '<div class="instrument-name">' . $name . '</div>' .
            '<span class="instrument-status status-no-data">Нет данных</span>' .
            '</div>';
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// ВАЛИДАЦИЯ ВХОДНЫХ ДАННЫХ
// ═══════════════════════════════════════════════════════════════════════════

$hasError = false;
$errorMessage = '';

// Проверка наличия URL эмитента
if (!isset($url_parts[1])) {
    $hasError = true;
    $errorMessage = 'URL эмитента не указан.';
}

if (!$hasError) {
    $emitentSlug = $url_parts[1];
    
    // Проверка на допустимые символы в URL
    if (!preg_match('/^[a-zA-Z0-9_\-]+$/', $emitentSlug)) {
        $hasError = true;
        $errorMessage = 'Недопустимые символы в URL эмитента.';
    }
    
    // Проверка длины URL
    if (strlen($emitentSlug) > 100) {
        $hasError = true;
        $errorMessage = 'URL эмитента слишком длинный.';
    }
}

// Получение данных эмитента из БД
$emitent = null;
if (!$hasError) {
    require_once('DB/emitents.php');
    require_once('DB/instruments.php');
    
    $emitent = getEmitentByUrl($emitentSlug);
    
    if ($emitent === null) {
        $hasError = true;
        $errorMessage = 'Эмитент не найден в базе данных.';
    }
}

// Вывод ошибки если есть
if ($hasError): ?>
    <div class="container">
        <div class="alert alert-danger" role="alert">
            <h4 class="alert-heading">⚠️ Ошибка загрузки данных</h4>
            <p><strong>Описание:</strong> <?php echo esc($errorMessage); ?></p>
            <hr>
            <p class="mb-0">
                <small>Если проблема повторяется, обратитесь к администратору сайта.</small>
            </p>
        </div>
    </div>
<?php 
    error_log('[EMITENT.PHP v8.0.0] Ошибка: ' . $errorMessage);
    return;
endif;

// ═══════════════════════════════════════════════════════════════════════════
// ТЕКУЩИЙ ПУТЬ ДЛЯ АНКОРНЫХ ССЫЛОК (v7.1.2 ANCHOR-FIX + v7.1.3 IMPROVEMENTS)
// ═══════════════════════════════════════════════════════════════════════════
// Получаем чистый путь без query string и анкоров
// Пример: /sberbank/ или /gazprom/
// Используется в кликабельных метриках для корректных анкорных ссылок
//
// v7.1.3: Добавлена защита от parse_url() возвращающего false/null
//         Добавлен fallback к /$emitentSlug/ если REQUEST_URI некорректный

// Проверяем что REQUEST_URI существует
if (!isset($_SERVER['REQUEST_URI']) || empty($_SERVER['REQUEST_URI'])) {
    // Fallback: строим путь из slug эмитента
    $currentPath = '/' . $emitentSlug . '/';
} else {
    // Извлекаем только path (без query string и anchor)
    $currentPath = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    
    // Защита от parse_url() возвращающего false или null
    if (empty($currentPath)) {
        // Fallback: строим путь из slug эмитента
        $currentPath = '/' . $emitentSlug . '/';
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// ПОЛУЧЕНИЕ ИНСТРУМЕНТОВ
// ═══════════════════════════════════════════════════════════════════════════

// Получаем ВСЕ инструменты эмитента (с QUIK данными и без)
$allInstruments = getInstrumentsByEmitentId($emitent->Id);

// Группируем по типам (1-6) и разделяем на активные/неактивные
$instrumentsByType = [
    1 => ['active' => [], 'no_data' => []], // Акции (обыкновенные + привилегированные)
    2 => ['active' => [], 'no_data' => []], // Депозитарные расписки
    3 => ['active' => [], 'no_data' => []], // Еврооблигации
    4 => ['active' => [], 'no_data' => []], // Инвестиционные паи
    5 => ['active' => [], 'no_data' => []], // Ипотечные сертификаты
    6 => ['active' => [], 'no_data' => []]  // Облигации
];

foreach ($allInstruments as $inst) {
    $typeId = (int)($inst['Instrument_SUPERTYPE_ID'] ?? 0);
    
    // Пропускаем если тип неизвестен
    if (!isset($instrumentsByType[$typeId])) continue;
    
    // Проверяем наличие QUIK данных (есть ли SecCode в акциях или облигациях)
    $hasQuikData = !empty($inst['Akcii_SecCode']) || !empty($inst['Obligacii_SecCode']);
    
    if ($hasQuikData) {
        // Инструмент с QUIK данными → активная карточка с ссылкой
        $instrumentsByType[$typeId]['active'][] = $inst;
    } else {
        // Инструмент без QUIK данными → серая карточка, раздел "Нет данных"
        $instrumentsByType[$typeId]['no_data'][] = $inst;
    }
}

// Подсчитываем общее количество инструментов
$totalCount = count($allInstruments);

// Сортируем активные инструменты внутри каждого типа
// Сначала "торгуется" (зеленая рамка), потом остальные статусы (оранжевая рамка)
foreach ($instrumentsByType as $typeId => &$groups) {
    if (!empty($groups['active'])) {
        usort($groups['active'], function($a, $b) {
            // Получаем статусы из соответствующих таблиц
            $statusA = mb_strtolower($a['Obligacii_SecStatus'] ?? $a['Akcii_SecStatus'] ?? '');
            $statusB = mb_strtolower($b['Obligacii_SecStatus'] ?? $b['Akcii_SecStatus'] ?? '');
            
            // "торгуется" всегда в начале списка
            if ($statusA === 'торгуется' && $statusB !== 'торгуется') return -1;
            if ($statusA !== 'торгуется' && $statusB === 'торгуется') return 1;
            
            // Остальные сортируем по алфавиту кода инструмента
            return strcmp($a['Instrument_TRADE_CODE'] ?? '', $b['Instrument_TRADE_CODE'] ?? '');
        });
    }
}
unset($groups); // Очищаем ссылку после foreach

// Подсчитываем количество инструментов по типам для метрик
$typesCounts = [];
foreach ($instrumentsByType as $typeId => $groups) {
    $count = count($groups['active']);
    if ($count > 0) {
        $typesCounts[$typeId] = $count;
    }
}

?>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- НАЧАЛО КОНТЕЙНЕРА                                                      -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div class="container">

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- HERO СЕКЦИЯ                                                            -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<section class="hero">
    <div class="hero-header">
        <?php 
        $validatedImage = validateImagePath($emitent->EMITENT_og_image ?? '');
        if (!isEmpty($validatedImage)): 
        ?>
        <div class="emitent-logo">
            <img src="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>images/<?php echo esc($validatedImage); ?>" 
                 alt="Логотип <?php echo esc($emitent->EMITENT_SHORT_NAME ?: 'эмитента'); ?>"
                 class="emitent-logo-img"
                 width="80"
                 height="80"
                 loading="lazy"
                 decoding="async">
        </div>
        <?php endif; ?>
        
        <div class="hero-content">
            <?php 
            // v7.1.4 PERFECTED: Fallback для названия эмитента
            $emitentName = $emitent->EMITENT_SHORT_NAME ?? $emitent->EMITENT_FULL_NAME ?? 'Эмитент';
            ?>
            <h1><?php echo esc($emitentName); ?></h1>
            
            <?php 
            $hasShortName = !isEmpty($emitent->EMITENT_SHORT_NAME);
            $hasFullName = !isEmpty($emitent->EMITENT_FULL_NAME);
            $namesAreDifferent = $hasShortName && $hasFullName && ($emitent->EMITENT_FULL_NAME !== $emitent->EMITENT_SHORT_NAME);
            
            if ($namesAreDifferent): 
            ?>
            <p class="company-legal-name"><?php echo esc($emitent->EMITENT_FULL_NAME); ?></p>
            <?php endif; ?>
            
            <?php if (!isEmpty($emitent->EMITENT_TXT_SHORT)): 
                $description = trim($emitent->EMITENT_TXT_SHORT);
                if (!empty($description)) {
                    $description = mb_strtoupper(mb_substr($description, 0, 1)) . mb_substr($description, 1);
                }
            ?>
            <p class="hero-description"><?php echo esc($description); ?></p>
            <?php endif; ?>
            
            <div class="hero-badges">
                <?php if (!isEmpty($emitent->INN) && $emitent->INN !== '0'): ?>
                <span class="badge"><span aria-hidden="true">🏢</span> ИНН: <?php echo esc($emitent->INN); ?></span>
                <?php endif; ?>
                
                <?php if (!isEmpty($emitent->LEI)): ?>
                <span class="badge"><span aria-hidden="true">🏛️</span> LEI: <?php echo esc($emitent->LEI); ?></span>
                <?php endif; ?>
                
                <?php if (!isEmpty($emitent->EMITENT_FOUNDING_DATE) && $emitent->EMITENT_FOUNDING_DATE !== '0000-00-00'): 
                    $foundingDate = DateTime::createFromFormat('Y-m-d', $emitent->EMITENT_FOUNDING_DATE);
                    if ($foundingDate) {
                        $formattedDate = $foundingDate->format('d.m.Y');
                ?>
                <span class="badge"><span aria-hidden="true">📅</span> Основана: <?php echo esc($formattedDate); ?></span>
                <?php 
                    }
                endif; ?>
                
                <?php if (!isEmpty($emitent->SECTOR_ECONOMIKI)): ?>
                <span class="badge"><span aria-hidden="true">🏭</span> <?php echo esc($emitent->SECTOR_ECONOMIKI); ?></span>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- КЛИКАБЕЛЬНЫЕ МЕТРИКИ С АНКОРАМИ (v7.1.5 FINAL)                         -->
<!-- v7.1.5: Метрика показывает "Акции (2)" вместо "Акции обыкновенные (2)" -->
<!-- v8.0.0: Добавлен aria-hidden на metric-icon                            -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<?php if ($totalCount > 0): ?>
<div class="metrics-grid">
    <?php foreach ($typesCounts as $typeId => $count): ?>
        <a href="<?php echo esc($currentPath); ?>#section-type-<?php echo $typeId; ?>" 
           class="metric-card" 

           title="Перейти к разделу <?php echo esc(getInstrumentTypeName($typeId)); ?>">
            <div class="metric-icon" aria-hidden="true"><?php echo getInstrumentTypeIcon($typeId); ?></div>
            <div class="metric-label"><?php echo esc(getInstrumentTypeName($typeId)); ?></div>
            <div class="metric-value"><?php echo (int)$count; ?></div>
        </a>
    <?php endforeach; ?>
</div>
<?php endif; ?>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- ОСНОВНОЙ КОНТЕНТ: ДВУХКОЛОНОЧНЫЙ LAYOUT                                -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div class="stock-two-columns">
    
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <!-- ЛЕВАЯ КОЛОНКА (65%)                                                -->
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <div>
        
        <!-- ИДЕНТИФИКАТОРЫ -->
        <?php 
        $identifiers = [];
        
        if (!isEmpty($emitent->INN) && $emitent->INN !== '0') {
            $identifiers[] = renderIdentifier('inn', 'ИНН', $emitent->INN);
        }
        
        if (!isEmpty($emitent->LEI)) {
            $identifiers[] = renderIdentifier('lei', 'LEI КОД', $emitent->LEI);
        }
        
        if (!empty($identifiers)): 
        ?>
        <section class="section">
            <h2 class="section-title">
                <span aria-hidden="true">🔑</span>
                Идентификаторы
            </h2>
            <div class="identifiers-grid">
                <?php echo implode('', $identifiers); ?>
            </div>
        </section>
        <?php endif; ?>
        
        <!-- ИНСТРУМЕНТЫ ПО ТИПАМ (ПОРЯДОК 1-2-3-4-5-6) -->
        <!-- v7.1.5: Раздел называется "Акции (2)" вместо "Акции обыкновенные (2)" -->
        <!-- v8.0.0: Добавлен aria-hidden на эмоджи в H2 -->
        <?php foreach ([1, 2, 3, 4, 5, 6] as $typeId): ?>
            <?php 
            $activeInstruments = $instrumentsByType[$typeId]['active'] ?? [];
            $noDataInstruments = $instrumentsByType[$typeId]['no_data'] ?? [];
            $hasAny = !empty($activeInstruments);
            
            // Пропускаем тип если нет активных инструментов
            if (!$hasAny) continue;
            ?>
            
            <section class="section" id="section-type-<?php echo $typeId; ?>">
                <h2 class="section-title">
                    <span aria-hidden="true"><?php echo getInstrumentTypeIcon($typeId); ?></span>
                    <?php echo esc(getInstrumentTypeName($typeId)); ?> (<?php echo count($activeInstruments); ?>)
                </h2>
                
                <!-- АКТИВНЫЕ ИНСТРУМЕНТЫ (с QUIK данными) -->
                <?php if (!empty($activeInstruments)): ?>
                <div class="instruments-grid">
                    <?php foreach ($activeInstruments as $inst): ?>
                        <?php echo renderInstrumentCard($inst, true); ?>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- ИНСТРУМЕНТЫ БЕЗ ДАННЫХ (нет в QUIK) -->
                <!-- v8.5.0: ЗАКОММЕНТИРОВАНО - не показываем инструменты без QUIK данных -->
                <?php /*
                <!-- v8.0.0: Заменён H3 на div с классом subsection-title -->
                <?php if (!empty($noDataInstruments)): ?>
                <div class="subsection">
                    <div class="subsection-title"><span aria-hidden="true">📊</span> Нет данных (<?php echo count($noDataInstruments); ?>)</div>
                    <div class="instruments-grid">
                        <?php foreach ($noDataInstruments as $inst): ?>
                            <?php echo renderInstrumentCard($inst, false); ?>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                */ ?>
            </section>
        <?php endforeach; ?>
        
        <!-- СООБЩЕНИЕ ОБ ОТСУТСТВИИ ИНСТРУМЕНТОВ -->
        <?php if ($totalCount === 0): ?>
        <section class="section">
            <h2 class="section-title">
                <span aria-hidden="true">📊</span>
                Ценные бумаги
            </h2>
            <div class="info-notice">
                У эмитента нет активных ценных бумаг в базе данных.
            </div>
        </section>
        <?php endif; ?>
        
        <!-- ПОДРОБНОЕ ОПИСАНИЕ -->
        <!-- v8.0.0: Добавлен aria-hidden на эмоджи -->
        <?php if (!isEmpty($emitent->EMITENT_TXT_LONG)): ?>
        <section class="section">
            <h2 class="section-title">
                <span aria-hidden="true">📝</span>
                Подробное описание
            </h2>
            <details class="details-section" open>
                <summary class="details-summary">
                    <div class="summary-left">
                        <span class="summary-icon" aria-hidden="true">📄</span>
                        <span>О компании <?php echo esc($emitent->EMITENT_SHORT_NAME ?? ''); ?></span>
                    </div>
                    <span class="summary-arrow">▼</span>
                </summary>
                <div class="details-content">
                    <?php echo renderLongDescription($emitent->EMITENT_TXT_LONG); ?>
                </div>
            </details>
        </section>
        <?php endif; ?>
        
        <!-- ИНФОРМАЦИОННОЕ УВЕДОМЛЕНИЕ -->
        <div class="info-notice">
            <strong>Источник данных:</strong> Справочная информация из базы данных DeepMax.ru.<br>
            Данные обновляются периодически и могут отличаться от текущих биржевых.
            <?php 
            $validatedMoexLink = validateExternalUrl($emitent->DISCLOSURE_PART_moex ?? '');
            if (!isEmpty($validatedMoexLink)): 
            ?>
            <br><br>
            <strong>Актуальные данные:</strong> 
            <a href="<?php echo esc($validatedMoexLink); ?>" 
               target="_blank" 
               rel="nofollow noopener"
               title="Страница эмитента <?php echo esc($emitent->EMITENT_SHORT_NAME ?? ''); ?> на Московской Бирже — актуальные котировки и торговые данные">
                Страница на Московской Бирже →
            </a>
            <?php endif; ?>
        </div>
        
    </div>
    <!-- КОНЕЦ ЛЕВОЙ КОЛОНКИ -->
    
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <!-- ПРАВАЯ КОЛОНКА (35%)                                               -->
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <aside class="sidebar">
        
        <!-- ВНЕШНИЕ ССЫЛКИ -->
        <!-- v8.0.0: H3 заменён на div.sidebar-title, убраны inline styles, добавлен aria-hidden -->
        <?php 
        $validatedOfficialSite = validateExternalUrl($emitent->DISCLOSURE_PART_PAGE ?? '');
        $validatedMoex = validateExternalUrl($emitent->DISCLOSURE_PART_moex ?? '');
        $validatedDisclosure = validateExternalUrl($emitent->DISCLOSURE_RF_INFO_PAGE ?? '');
        $validatedSmartLab = validateExternalUrl($emitent->{'DISCLOSURE_PART_smart-lab'} ?? '');
        $validatedDohod = validateExternalUrl($emitent->DISCLOSURE_PART_dohod ?? '');
        $validatedWiki = validateExternalUrl($emitent->DISCLOSURE_PART_wiki ?? '');
        
        $hasExternalLinks = !isEmpty($validatedOfficialSite) || 
                            !isEmpty($validatedMoex) || 
                            !isEmpty($validatedDisclosure) || 
                            !isEmpty($validatedSmartLab) || 
                            !isEmpty($validatedDohod) || 
                            !isEmpty($validatedWiki);
        ?>
        <?php if ($hasExternalLinks): ?>
        <div class="emitent-card">
            <div class="sidebar-title">
                <span aria-hidden="true">🔗</span>
                Официальные источники
            </div>
            <div class="emitent-links">
                <?php if (!isEmpty($validatedOfficialSite)): ?>
                <a href="<?php echo esc($validatedOfficialSite); ?>" 
                   class="emitent-link-btn" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Перейти на официальный сайт компании <?php echo esc($emitent->EMITENT_FULL_NAME ?? $emitent->EMITENT_SHORT_NAME ?? ''); ?>">
                    <span aria-hidden="true">🌐</span> Официальный сайт
                </a>
                <?php endif; ?>
                
                <?php if (!isEmpty($validatedMoex)): ?>
                <a href="<?php echo esc($validatedMoex); ?>" 
                   class="link-btn link-btn-primary" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Страница эмитента <?php echo esc($emitent->EMITENT_SHORT_NAME ?? ''); ?> на Московской Бирже — актуальные котировки и данные">
                    <span aria-hidden="true">📊</span> Московская Биржа
                </a>
                <?php endif; ?>
                
                <?php if (!isEmpty($validatedDisclosure)): ?>
                <a href="<?php echo esc($validatedDisclosure); ?>" 
                   class="emitent-link-btn" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Раскрытие информации эмитента <?php echo esc($emitent->EMITENT_FULL_NAME ?? ''); ?> — официальные документы и отчётность">
                    <span aria-hidden="true">📰</span> Раскрытие информации
                </a>
                <?php endif; ?>
                
                <?php if (!isEmpty($validatedSmartLab)): ?>
                <a href="<?php echo esc($validatedSmartLab); ?>" 
                   class="emitent-link-btn" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Аналитика и обсуждение акций <?php echo esc($emitent->EMITENT_SHORT_NAME ?? ''); ?> на Smart-Lab — мнения трейдеров и инвесторов">
                    <span aria-hidden="true">📈</span> Smart-Lab
                </a>
                <?php endif; ?>
                
                <?php if (!isEmpty($validatedDohod)): ?>
                <a href="<?php echo esc($validatedDohod); ?>" 
                   class="emitent-link-btn" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Информация о компании <?php echo esc($emitent->EMITENT_SHORT_NAME ?? ''); ?> на Dohod.ru — дивиденды и финансовые показатели">
                    <span aria-hidden="true">💰</span> Dohod.ru
                </a>
                <?php endif; ?>
                
                <?php if (!isEmpty($validatedWiki)): ?>
                <a href="<?php echo esc($validatedWiki); ?>" 
                   class="emitent-link-btn" 
                   target="_blank" 
                   rel="nofollow noopener"
                   title="Статья о компании <?php echo esc($emitent->EMITENT_FULL_NAME ?? ''); ?> в Википедии — история и описание деятельности">
                    <span aria-hidden="true">📖</span> Wikipedia
                </a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- БЛОК ПЕРЕЛИНКОВКИ: ПОПУЛЯРНЫЕ ЭМИТЕНТЫ СЕКТОРА -->
        <!-- v8.3.0: Исправлен подсчёт - только инструменты с данными QUIK -->
        <!-- v8.2.0: Детальная информация о типах инструментов -->
        <!-- v8.1.0: Перемещён в sidebar, новый заголовок, title из БД -->
        <?php
        // Проверка: Сектор заполнен и ID > 0?
        if (!isEmpty($emitent->ID_SECTOR_ECONOMIKI) && $emitent->ID_SECTOR_ECONOMIKI > 0) {
            
            // Получаем топовых эмитентов из того же сектора
            $topEmitents = getTopEmitentsBySector($db, $emitent->ID_SECTOR_ECONOMIKI, $emitent->Id);
            
            // Показываем блок только если есть хотя бы 1 эмитент
            if (count($topEmitents) > 0): 
            ?>
            <div class="emitent-card">
                <div class="sidebar-title">
                    <span aria-hidden="true">🏆</span>
                    Популярные агрокомпании и эмитенты на рынке
                </div>
                
                <div class="emitent-links">
                    <?php foreach ($topEmitents as $topEm): 
                        // Получаем словосочетание из БД с fallback
                        $sectorSuffix = $topEm->SECTOR_TITLE_SUFFIX ?? 'эмитент российского фондового рынка';
                        
                        // Формируем title для ссылки
                        // Пример: "Черкизово — эмитент агропромышленного сектора"
                        $linkTitle = esc($topEm->EMITENT_SHORT_NAME) . ' — лучший ' . esc($sectorSuffix);
                    ?>
                    <a href="/<?php echo esc($topEm->EMITENT_URL); ?>/" 
                       class="emitent-link-btn"
                       title="<?php echo $linkTitle; ?>">
                        <span aria-hidden="true">🏢</span> 
                        <span class="emitent-name"><?php echo esc($topEm->EMITENT_SHORT_NAME); ?></span>
                    </a>
                    <?php endforeach; ?>
            </div>
            <?php 
            endif; // count > 0
            
        } // ID_SECTOR_ECONOMIKI > 0
        ?>
        
        <!-- ИНФОРМАЦИЯ ОБ ОБНОВЛЕНИИ -->
        <!-- v8.0.0: H3 заменён на div.sidebar-title, убраны inline styles, добавлен aria-hidden -->
        <div class="emitent-card">
            <div class="sidebar-title">
                <span aria-hidden="true">📅</span>
                Обновление данных
            </div>
            <div class="emitent-meta">
                <strong>Последнее обновление:</strong><br>
                <?php echo date('d.m.Y, H:i'); ?> МСК
            </div>
        </div>
        
    </aside>
    <!-- КОНЕЦ ПРАВОЙ КОЛОНКИ -->
    
</div>
<!-- КОНЕЦ STOCK-TWO-COLUMNS -->

</div>
<!-- КОНЕЦ КОНТЕЙНЕРА -->

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- TOAST УВЕДОМЛЕНИЕ                                                      -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div id="toast" class="toast" role="status" aria-live="polite" aria-atomic="true">
    <span>✓</span>
    <span id="toast-text">Скопировано</span>
</div>

<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КОНЕЦ emitent.php v8.3.0 FIX ✅
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 8.3.0 FIX
 * Дата: 2025-10-26
 * Статус: PRODUCTION READY ✅ SEO + DB + QUIK INTEGRATION
 * 
 * КРИТИЧНЫЕ ИЗМЕНЕНИЯ v8.2.0:
 * 
 * ИЗМЕНЕНИЯ v8.3.0 FIX:
 *   ✅ FIX: Исправлен подсчёт инструментов в блоке перелинковки
 *   ✅ SQL: Считаются ТОЛЬКО инструменты с данными в QUIK таблицах
 *   ✅ Акции: типы (1,2,4) которые ЕСТЬ в QUIK_Akcii
 *   ✅ Облигации: типы (3,5,6) которые ЕСТЬ в QUIK_Obligacii
 *   ✅ Правильное склонение: "2 акции, 5 облигаций"
 * 
 * ИЗМЕНЕНИЯ v8.2.0 (частично откачены):
 *   ❌ GROUP_CONCAT удалён - заменён на прямой подсчёт в SQL
 *   ✅ Функция getDeclension() - сохранена
 *   ✅ CSS классы .emitent-name, .instruments-info - сохранены
 * 
 * ИЗМЕНЕНИЯ v8.2.0 (оригинальные):
 *   ✅ Детальная информация о типах инструментов: "(1 акция, 4 облигации)"
 *   ✅ SQL: GROUP_CONCAT для получения типов инструментов
 *   ✅ Выравнивание по левому краю, новые CSS классы
 * 
 * КРИТИЧНЫЕ ИЗМЕНЕНИЯ v8.1.0 (сохранены):
 * КРИТИЧНЫЕ ИЗМЕНЕНИЯ v8.1.0:
 *   ✅ Исправлена ошибка global $db в getTopEmitentsBySector()
 *   ✅ Блок перелинковки в sidebar (правая колонка)
 *   ✅ Заголовок: "Популярные агрокомпании и эмитенты на рынке"
 *   ✅ Title из БД: SECTOR_TITLE_SUFFIX (таблица SECTOR_ECONOMIKI)
 *   ✅ SQL включает JOIN с SECTOR_ECONOMIKI
 *   ✅ Badge с количеством инструментов (.badge-count)
 * 
 * НАСЛЕДУЕТСЯ ОТ v8.0.0:
 *   ✅ aria-hidden на всех декоративных эмоджи (A11Y)
 *   ✅ H3 заменены на .sidebar-title (SEO иерархия)
 *   ✅ Inline styles убраны (CSP совместимость)
 *   ✅ Все исправления безопасности сохранены
 * 
 * ТРЕБУЕТСЯ:
 *   - Таблица SECTOR_ECONOMIKI с полем SECTOR_TITLE_SUFFIX
 *   - CSS классы: .emitent-name, .instruments-info в best.css
 *   - SQL: ALTER TABLE + 18 UPDATE запросов (см. sector_title_suffix_sql.sql)
 * 
 * КОД-КАЧЕСТВО: 10/10 ✨ PERFECT SEO + DB
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */