<?php
/**
 * Каталог облигаций Московской биржи
 * Версия: 4.0.0 EMITENTS-GRID
 * Дата: 2025-10-29
 * Основа: bonds-catalog.php v3.0.0 SECTORS-GRID
 * 
 * CHANGELOG:
 * 
 * v4.0.0 EMITENTS-GRID (2025-10-29):
 * - BREAKING: Изменена структура отображения секторов и эмитентов
 * - CHANGE: Секторы отображаются друг под другом (полная ширина) вместо Grid
 * - CHANGE: Эмитенты внутри сектора отображаются в Grid 3 колонки
 * - ADD: .sector-block - блок сектора (полная ширина)
 * - ADD: .emitents-grid - Grid для эмитентов (3 → 2 → 1 колонка)
 * - ADD: .emitent-column - колонка эмитента
 * - REMOVE: .sectors-grid (заменён на последовательные .sector-block)
 * - REMOVE: .sector-column (заменён на .emitent-column)
 * - REMOVE: .sector-content (заменён на .emitents-grid)
 * - PRESERVED: 100% функциональности v3.0.0 (навигация, типы, collapse)
 * - PRESERVED: Все SQL поля, SECTOR_ICONS, DEBUG, метрики, легенда
 * - REASON: Лучшая читаемость - секторы по ширине, эмитенты в колонках
 * - FOR: best.css v4.0.0 + bond-countdown.js v1.5.0 SMOOTH-OFFSET
 * 
 * v3.0.0 SECTORS-GRID (2025-10-29):
 * - BREAKING: Изменена структура отображения типов и секторов
 * - CHANGE: Типы отображаются друг под другом (полная ширина) вместо колонок
 * - CHANGE: Секторы внутри типа отображаются в Grid 3 колонки
 * - CHANGE: H3 секторов теперь обычные заголовки (не в <details>)
 * - CHANGE: Секторы всегда видимы (убраны details/summary)
 * - ADD: .bonds-catalog-container - контейнер каталога
 * - ADD: .bond-type-section - секция типа (полная ширина)
 * - ADD: .sectors-grid - Grid для секторов (3 → 2 → 1 колонка)
 * - ADD: .sector-column - колонка сектора
 * - ADD: .sector-title - H3 заголовок сектора с градиентом
 * - ADD: .sector-content - контент сектора
 * - REMOVE: .bonds-type-grid (заменён на .bonds-catalog-container)
 * - REMOVE: .bond-type-column (заменён на .bond-type-section)
 * - REMOVE: details.v6-sector (заменён на .sector-column)
 * - PRESERVED: 100% функциональности v2.2.0 (навигация, collapse, эмитенты)
 * - PRESERVED: Все SQL поля, SECTOR_ICONS, DEBUG, метрики, легенда
 * - REASON: Лучшая читаемость - типы друг под другом, секторы в колонках
 * - REASON: H3 секторов как кнопки типов (единообразный стиль)
 * - FOR: best.css v3.0.0 с новыми классами для секторов
 * 
 * v2.2.0 NAVIGATION (2025-10-29):
 * - ADD: Навигация по типам облигаций сверху (Grid 3×2, кликабельные якоря)
 * - ADD: Массив иконок типов облигаций (🏛️ 🌍 🏢 📊)
 * - ADD: Якоря id="type-X" для H2 заголовков типов
 * - ADD: Плавный скролл к якорям (CSS scroll-behavior: smooth)
 * - ADD: Стиль кнопок навигации как .listing-column-title из stocks-catalog
 * - PRESERVED: 100% функциональности v2.1.0 (prepared statement, проверки, A11Y)
 * - PRESERVED: Все SQL поля, SECTOR_ICONS, DEBUG, wrapper'ы, легенда
 * - REASON: Удобная навигация по типам без потери функциональности
 * - FOR: best.css с классами .bonds-type-navigation
 * 
 * v2.1.0 LIMIT-4 (2025-10-29):
 * - CHANGE: Логика отображения - первые 4 облигации видны, остальные в collapse
 * - CHANGE: Текст кнопки: "Ещё X облигаций" (без указания уровней)
 * - ADD: Склонение "облигация/облигации/облигаций" в тексте кнопки
 * - ADD: Иконки уровней (🏆🥈🥉) для ВСЕХ облигаций (не только первого уровня)
 * - REASON: Простая и понятная логика для всех эмитентов
 * - REASON: Избегаем "каши" из 5-10 облигаций подряд
 * - REASON: Единообразное отображение независимо от уровней листинга
 * - FOR: best.css с padding: 8px и gap: 8px
 * 
 * v2.0.0 TYPE-STRUCTURE (2025-10-28):
 * - BREAKING: Полная смена структуры группировки
 * - CHANGE: TYPE → SECTOR → EMITENT → bonds (было: LISTING → SECTOR → EMITENT)
 * - CHANGE: 6 колонок типов облигаций вместо 3 колонок листингов
 * - CHANGE: H2 = тип облигации (Биржевая, Корпоративная, ОФЗ и т.д.)
 * - CHANGE: H3 = сектор экономики (в <details>)
 * - CHANGE: Нативный <details> для 2-3 уровня листинга (вместо кнопки)
 * - ADD: Фильтр INSTRUMENT_TYPE_ID IN (7,8,9,10,11,16)
 * - ADD: Фильтр qo.SecCode IS NOT NULL в SQL
 * - ADD: Динамические названия типов из БД (Instrument_TYPE)
 * - ADD: Instrument_LIST_SECTION в SELECT для отображения названия уровня
 * - ADD: Якоря для эмитентов (id="emitent-{id}")
 * - ADD: Метрика "Типов" вместо метрик по листингам
 * - REMOVE: Кнопка "Показать ещё" (.v6-show-more)
 * - REMOVE: Скрытый span (.v6-bonds-hidden)
 * - REMOVE: JavaScript зависимость (bonds-v1.0.0.js больше не нужен)
 * - IMPROVE: Лучшая SEO структура (TYPE как основная группировка)
 * - IMPROVE: Адаптивность (6 → 3 → 1 колонка)
 * - REASON: Архитектура согласована с ChatGPT как эталонная
 * - REASON: H1→H2(type)→H3(sector)→DT/DD = идеальная иерархия
 * - PRESERVED: Вся логика безопасности, A11Y, debug логирование
 * - PRESERVED: Все поля данных (inn, name, status)
 * - PRESERVED: Все проверки и валидации
 * - PRESERVED: Hero секция с правильными классами
 * - PRESERVED: Metrics grid с двойным классом
 * - PRESERVED: Legend блок
 * 
 * v1.8.0 DL-COLLAPSE (2025-10-28):
 * - CHANGE: H4/H5 заменены на <dl><dt><dd> (SEO оптимизация)
 * - CHANGE: Title улучшен: {Instrument_TYPE} {ISIN} — {Эмитент}
 * - ADD: Collapse для облигаций (первые 4 видны, остальные свёрнуты)
 * - ADD: Кнопка "Показать ещё N" / "Скрыть" для эмитентов с > 4 облигациями
 * - ADD: Счётчик облигаций в <dt> для эмитента
 * 
 * СТРУКТУРА v4.0.0:
 * - Hero секция с описанием и иконкой 📜
 * - 5 метрик (типов + секторов + эмитентов + торгуется + всего)
 * - Навигация по типам (Grid 3×2, якоря с плавным скроллом + offset 150px)
 * - Типы друг под другом, полная ширина (bonds-catalog-container)
 * - Секторы друг под другом, полная ширина (sector-block)
 * - Эмитенты в Grid 3 колонки внутри сектора (emitents-grid → 3→2→1)
 * - Иерархия: H1 → NAV (типы) → H2 (тип) → H3 (сектор) → DT (эмитент) → DD (облигации)
 * - Группировка: Тип (full width) → Сектор (full width) → Эмитент (3 колонки) → Облигации (первые 4 + collapse)
 * - Легенда с 4 обозначениями (🟢 🏆 🥈 🥉)
 * - Полная A11Y поддержка (aria-hidden на всех эмоджи)
 * - Нативный <details> для скрытых облигаций (без JavaScript)
 * - Лимит 4 видимых облигации (остальные в collapse независимо от уровня)
 * 
 * ОСОБЕННОСТИ:
 * - Каталог ТОЛЬКО облигаций (Instrument_SUPERTYPE_ID = 6)
 * - Фильтр по типам: 7,8,9,10,11,16 (биржевая, иностранная, корпоративная, муниципальная, субфедеральная, ОФЗ)
 * - Показываются ТОЛЬКО торгующиеся облигации (qo.SecCode IS NOT NULL)
 * - Названия типов динамически из БД (Instrument_TYPE)
 * - DEBUG: Логирование первых 5 записей с TYPE_ID полями
 * - Интеграция с QUIK_Obligacii для торговых данных
 * - URL формат: /bonds/{Instrument_ISIN}/
 * - Wrapper'ы flexbox для mobile order: метрики → каталог → легенда
 * - CSP compatible (нет inline JS/CSS, кроме плавного скролла)
 * - XSS защита через esc()
 */

if (!function_exists('esc')) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА:</strong> Функция esc() не найдена.</div>');
}

global $db;
global $page_data;

if (!isset($db)) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА:</strong> База данных не подключена.</div>');
}

// Маппинг секторов к иконкам
if (!defined('SECTOR_ICONS')) {
    define('SECTOR_ICONS', [
        1  => '💻', 2  => '🌾', 3  => '🏦', 4  => '🏛️', 5  => '🏥',
        6  => '💳', 7  => '⚙️', 8  => '🔩', 9  => '🏗️', 10 => '🛢️',
        11 => '🏭', 12 => '📡', 13 => '🔬', 14 => '🛒', 15 => '🚚',
        16 => '💰', 18 => '🧪', 19 => '⚡'
    ]);
}

// Маппинг уровней листинга (для иконок в облигациях)
$listingLevels = [
    1 => ['name' => 'Первый уровень', 'icon' => '🏆'],
    2 => ['name' => 'Второй уровень', 'icon' => '🥈'],
    3 => ['name' => 'Третий уровень', 'icon' => '🥉']
];

// Маппинг типов облигаций к иконкам (НОВОЕ в v2.2.0)
$typeIcons = [
    7  => '🏛️',  // Облигация биржевая
    8  => '🌍',  // Облигация иностранного эмитента
    9  => '🏢',  // Облигация корпоративная
    10 => '🏛️',  // Облигация муниципальная
    11 => '🏛️',  // Облигация субфедеральная
    16 => '📊',  // ОФЗ
];

// ============================================================================
// ПОЛУЧЕНИЕ ДАННЫХ ИЗ БД
// ============================================================================

$stmt = $db->connection->prepare("SELECT 
    i.Id,
    i.Instrument_TRADE_CODE,
    i.Instrument_ISIN,
    i.Instrument_TYPE,
    i.INSTRUMENT_TYPE_ID,
    i.Instrument_SUPERTYPE_ID,
    i.Instrument_LIST_SECTION,
    i.Instrument_LIST_SECTION_ID,
    i.Emitent_Id,
    i.Instrument_EMITENT_FULL_NAME,
    e.EMITENT_SHORT_NAME,
    e.EMITENT_FULL_NAME,
    e.INN,
    e.EMITENT_URL,
    e.ID_SECTOR_ECONOMIKI,
    e.SECTOR_ECONOMIKI,
    qo.SecCode as Obligacii_SecCode
FROM Instrument i
LEFT JOIN Emitent e ON i.Emitent_Id = e.Id
LEFT JOIN QUIK_Obligacii qo ON i.Instrument_ISIN = qo.ISIN
WHERE i.Instrument_SUPERTYPE_ID = ?
  AND i.INSTRUMENT_TYPE_ID IN (7, 8, 9, 10, 11, 16)
  AND qo.SecCode IS NOT NULL
ORDER BY i.INSTRUMENT_TYPE_ID, e.ID_SECTOR_ECONOMIKI, e.EMITENT_SHORT_NAME, i.Instrument_LIST_SECTION_ID");

$supertype_id = 6;
$stmt->bind_param("i", $supertype_id);
$stmt->execute();

if ($stmt->errno) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА БД:</strong> ' . esc($stmt->error) . '</div>');
}

$result = $stmt->get_result();

// Группируем данные: Тип -> Сектор -> Эмитент -> Облигации
$data = [];
$totalBonds = 0;
$tradingBonds = 0;

while ($row = $result->fetch_assoc()) {
    // DEBUG: Логируем первые 5 записей для проверки TYPE_ID
    static $debugCounter = 0;
    if ($debugCounter < 5) {
        error_log('[BONDS-CATALOG v2.2.0 NAVIGATION] ISIN=' . $row['Instrument_ISIN'] . 
                  ', Instrument_SUPERTYPE_ID=' . ($row['Instrument_SUPERTYPE_ID'] ?? 'NULL') . 
                  ', INSTRUMENT_TYPE_ID=' . ($row['INSTRUMENT_TYPE_ID'] ?? 'NULL') . 
                  ', TYPE=' . ($row['Instrument_TYPE'] ?? 'NULL') . 
                  ', Obligacii_SecCode=' . ($row['Obligacii_SecCode'] ?? 'NULL'));
        $debugCounter++;
    }
    
    $typeId = (int)$row['INSTRUMENT_TYPE_ID'];
    $typeName = $row['Instrument_TYPE'];
    $sectorId = (int)$row['ID_SECTOR_ECONOMIKI'];
    $sectorName = $row['SECTOR_ECONOMIKI'];
    $emitentId = (int)$row['Emitent_Id'];
    $listingId = (int)$row['Instrument_LIST_SECTION_ID'];
    
    // Пропускаем записи без эмитента, сектора или листинга
    if (!$emitentId || !$sectorId || !$listingId) continue;
    
    // КРИТИЧНО: Пропускаем облигации БЕЗ торговых данных (уже в SQL, но проверяем на всякий случай)
    if (empty($row['Obligacii_SecCode'])) continue;
    
    // Инициализация типа
    if (!isset($data[$typeId])) {
        $data[$typeId] = [
            'name' => $typeName,
            'icon' => $typeIcons[$typeId] ?? '📜',  // НОВОЕ: иконка типа
            'sectors' => []
        ];
    }
    
    // Инициализация сектора
    if (!isset($data[$typeId]['sectors'][$sectorId])) {
        $data[$typeId]['sectors'][$sectorId] = [
            'name' => $sectorName ?: 'Без сектора',
            'emitents' => []
        ];
    }
    
    // Инициализация эмитента
    if (!isset($data[$typeId]['sectors'][$sectorId]['emitents'][$emitentId])) {
        $data[$typeId]['sectors'][$sectorId]['emitents'][$emitentId] = [
            'name' => $row['EMITENT_SHORT_NAME'] ?: $row['EMITENT_FULL_NAME'],
            'inn' => $row['INN'] ?: '',
            'url' => $row['EMITENT_URL'] ?: '',
            'bonds' => []
        ];
    }
    
    // Определяем статус облигации
    $status = !empty($row['Obligacii_SecCode']) ? 'торгуется' : 'нет данных';
    if ($status === 'торгуется') $tradingBonds++;
    
    // Добавляем облигацию
    $data[$typeId]['sectors'][$sectorId]['emitents'][$emitentId]['bonds'][] = [
        'code' => $row['Instrument_ISIN'],
        'name' => $row['Instrument_EMITENT_FULL_NAME'] ?: $row['EMITENT_SHORT_NAME'],
        'type' => $row['Instrument_TYPE'] ?: 'Облигация',
        'listing_id' => $listingId,
        'listing_name' => $row['Instrument_LIST_SECTION'],
        'status' => $status
    ];
    
    $totalBonds++;
}

// Закрываем statement
$stmt->close();

// Подсчёт статистики
$totalTypes = count($data);
$totalEmitents = 0;

// Подсчёт общего количества секторов из константы
$totalSectors = count(SECTOR_ICONS);

// Подсчёт эмитентов
foreach ($data as $type) {
    foreach ($type['sectors'] as $sector) {
        $totalEmitents += count($sector['emitents']);
    }
}

// Подсчёт облигаций по типам
$typeBondsCount = [];
foreach ($data as $typeId => $type) {
    $typeBondsCount[$typeId] = 0;
    foreach ($type['sectors'] as $sector) {
        foreach ($sector['emitents'] as $emitent) {
            $typeBondsCount[$typeId] += count($emitent['bonds']);
        }
    }
}

?>

<style>
/* Плавный скролл к якорям (НОВОЕ в v2.2.0) */
html {
    scroll-behavior: smooth;
}

/* Отступ при скролле к якорю чтобы не перекрывало header */
:target {
    scroll-margin-top: 100px;
}
</style>

<div class="container">

<!-- HERO СЕКЦИЯ -->
<section class="hero">
    <div class="hero-header">
        <span class="sector-icon-large" aria-hidden="true">📜</span>
        <div class="hero-content-sector">
            <h1><?php echo esc($page_data[1]->MENU_ancor); ?></h1>
            <p class="hero-description">
                <?php echo esc($page_data[1]->MENU_description); ?>
            </p>
        </div>
    </div>
</section>

<!-- FLEXBOX WRAPPER ДЛЯ ИЗМЕНЕНИЯ ПОРЯДКА НА MOBILE -->
<div class="page-content-wrapper">

    <!-- МЕТРИКИ WRAPPER -->
    <div class="metrics-wrapper">
        <!-- КОМПАКТНАЯ СТАТИСТИКА -->
        <div class="metrics-grid metrics-grid-compact">
            
            <!-- ВСЕГО ОБЛИГАЦИЙ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-total-desc">
                <div class="metric-icon metric-icon-compact" aria-hidden="true">📜</div>
                <div class="metric-label">Всего</div>
                <div class="metric-value"><?php echo esc($totalBonds); ?></div>
                <span id="metric-total-desc" class="sr-only">
                    Всего облигаций в каталоге: <?php echo esc($totalBonds); ?>
                </span>
            </div>
            
            <!-- ТИПОВ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-types-desc">
                <div class="metric-icon metric-icon-compact" aria-hidden="true">📊</div>
                <div class="metric-label">Типов</div>
                <div class="metric-value"><?php echo esc($totalTypes); ?></div>
                <span id="metric-types-desc" class="sr-only">
                    Количество типов облигаций: <?php echo esc($totalTypes); ?>
                </span>
            </div>
            
            <!-- СЕКТОРОВ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-sectors-desc">
                <div class="metric-icon metric-icon-compact" aria-hidden="true">🗂️</div>
                <div class="metric-label">Секторов</div>
                <div class="metric-value"><?php echo esc($totalSectors); ?></div>
                <span id="metric-sectors-desc" class="sr-only">
                    Количество секторов экономики: <?php echo esc($totalSectors); ?>
                </span>
            </div>
            
            <!-- ЭМИТЕНТОВ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-emitents-desc">
                <div class="metric-icon metric-icon-compact" aria-hidden="true">🏢</div>
                <div class="metric-label">Эмитентов</div>
                <div class="metric-value"><?php echo esc($totalEmitents); ?></div>
                <span id="metric-emitents-desc" class="sr-only">
                    Количество эмитентов: <?php echo esc($totalEmitents); ?>
                </span>
            </div>
            
            <!-- ТОРГУЕТСЯ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-trading-desc">
                <div class="metric-icon metric-icon-compact" aria-hidden="true">🟢</div>
                <div class="metric-label">Торгуется</div>
                <div class="metric-value"><?php echo esc($tradingBonds); ?></div>
                <span id="metric-trading-desc" class="sr-only">
                    Количество активно торгуемых облигаций: <?php echo esc($tradingBonds); ?>
                </span>
            </div>
        </div>
    </div>
    <!-- КОНЕЦ МЕТРИКИ WRAPPER -->

    <!-- КАТАЛОГ WRAPPER -->
    <div class="catalog-wrapper">

        <?php if (empty($data)): ?>
        <div class="alert alert-info" role="alert">
            <strong>Нет данных</strong><br>
            В базе данных не найдено облигаций для отображения.
        </div>
        <?php else: ?>

        <!-- НАВИГАЦИЯ ПО ТИПАМ (Grid 3×2) - НОВОЕ в v2.2.0 -->
        <nav class="bonds-type-navigation" aria-label="Навигация по типам облигаций">
            <?php foreach ($data as $typeId => $type): ?>
            <a href="/bonds/#type-<?php echo esc($typeId); ?>" 
               class="listing-column-title" 
               title="Перейти к разделу: <?php echo esc($type['name']); ?>">
                <span aria-hidden="true"><?php echo $type['icon']; ?></span>
                <?php echo esc($type['name']); ?>
            </a>
            <?php endforeach; ?>
        </nav>

        <!-- ТИПЫ ОБЛИГАЦИЙ (полная ширина, друг под другом) -->
        <div class="bonds-catalog-container">
            <?php foreach ($data as $typeId => $type): ?>
            
            <div class="bond-type-section">
                <h2 class="bond-type-title" id="type-<?php echo esc($typeId); ?>">
                    <?php echo esc($type['name']); ?>
                </h2>
                
                <!-- СЕКТОРЫ ДРУГ ПОД ДРУГОМ (полная ширина) -->
                <?php foreach ($type['sectors'] as $sectorId => $sector): ?>
                
                <div class="sector-block">
                    <h3 class="sector-title">
                        <span class="v6-icon" aria-hidden="true"><?php echo SECTOR_ICONS[$sectorId] ?? '📁'; ?></span>
                        <span><?php echo esc($sector['name']); ?></span>
                        <span class="v1-count">(<?php echo esc(count($sector['emitents'])); ?>)</span>
                    </h3>
                    
                    <!-- ЭМИТЕНТЫ В GRID (3 колонки) -->
                    <div class="emitents-grid">
                        
                        <!-- СПИСОК ЭМИТЕНТОВ С ОБЛИГАЦИЯМИ -->
                        <?php foreach ($sector['emitents'] as $emitentId => $emitent): ?>
                        
                        <div class="emitent-column">
                        
                        <?php
                        // НОВАЯ ЛОГИКА: Показываем первые 4 облигации, остальные в collapse
                        $bondsCount = count($emitent['bonds']);
                        $visibleLimit = 4;
                        $visibleBonds = array_slice($emitent['bonds'], 0, $visibleLimit);
                        $hiddenBonds = array_slice($emitent['bonds'], $visibleLimit);
                        $hiddenCount = count($hiddenBonds);
                        ?>
                        
                        <dl class="v6-bond-emitent">
                            <dt class="v6-bond-emitent-name" id="emitent-<?php echo esc($emitentId); ?>">
                                <?php if (!empty($emitent['url'])): ?>
                                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/') . esc($emitent['url']); ?>/" 
                                   title="<?php echo esc($emitent['name']); ?> — полная информация об эмитенте">
                                    <?php echo esc($emitent['name']); ?>
                                </a>
                                <?php else: ?>
                                <span><?php echo esc($emitent['name']); ?></span>
                                <?php endif; ?>
                                <span class="bonds-count">(<?php echo $bondsCount; ?>)</span>
                            </dt>
                            
                            <dd class="v6-bond-stocks">
                                <!-- ПЕРВЫЕ 4 ОБЛИГАЦИИ (всегда показаны) -->
                                <?php foreach ($visibleBonds as $bond): ?>
                                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>bonds/<?php echo esc($bond['code']); ?>/" 
                                   class="v6-stock<?php echo $bond['status'] !== 'торгуется' ? ' paused' : ''; ?>"
                                   title="<?php echo esc($bond['type']); ?> <?php echo esc($bond['code']); ?> — <?php echo esc($emitent['name']); ?>">
                                    <?php echo esc($bond['code']); ?>
                                    <?php if ($bond['listing_id'] == 1): ?>
                                        <span aria-hidden="true">🏆</span>
                                    <?php elseif ($bond['listing_id'] == 2): ?>
                                        <span aria-hidden="true">🥈</span>
                                    <?php elseif ($bond['listing_id'] == 3): ?>
                                        <span aria-hidden="true">🥉</span>
                                    <?php endif; ?>
                                </a>
                                <?php endforeach; ?>
                                
                                <!-- ОСТАЛЬНЫЕ ОБЛИГАЦИИ (в collapse) -->
                                <?php if ($hiddenCount > 0): ?>
                                <details class="bonds-other-levels">
                                    <summary>Ещё <?php echo $hiddenCount; ?> <?php echo $hiddenCount == 1 ? 'облигация' : ($hiddenCount < 5 ? 'облигации' : 'облигаций'); ?></summary>
                                    
                                    <?php foreach ($hiddenBonds as $bond): ?>
                                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>bonds/<?php echo esc($bond['code']); ?>/" 
                                       class="v6-stock<?php echo $bond['status'] !== 'торгуется' ? ' paused' : ''; ?>"
                                       title="<?php echo esc($bond['type']); ?> <?php echo esc($bond['code']); ?> — <?php echo esc($emitent['name']); ?>">
                                        <?php echo esc($bond['code']); ?>
                                        <?php if ($bond['listing_id'] == 1): ?>
                                            <span aria-hidden="true">🏆</span>
                                        <?php elseif ($bond['listing_id'] == 2): ?>
                                            <span aria-hidden="true">🥈</span>
                                        <?php elseif ($bond['listing_id'] == 3): ?>
                                            <span aria-hidden="true">🥉</span>
                                        <?php endif; ?>
                                    </a>
                                    <?php endforeach; ?>
                                </details>
                                <?php endif; ?>
                            </dd>
                        </dl>
                        
                        </div>
                        <!-- Конец .emitent-column -->
                        
                        <?php endforeach; ?>
                        
                    </div>
                    <!-- Конец .emitents-grid -->
                    
                </div>
                <!-- Конец .sector-block -->
                
                <?php endforeach; ?>
                
            </div>
            
            <?php endforeach; ?>
        </div>

        <?php endif; ?>

    </div>
    <!-- КОНЕЦ КАТАЛОГ WRAPPER -->

    <!-- ЛЕГЕНДА WRAPPER -->
    <div class="legend-wrapper">

        <!-- ОБОЗНАЧЕНИЯ -->
        <div class="legend-block">
            <h3 class="legend-title">
                <span aria-hidden="true">🔍</span>
                Обозначения
            </h3>
            <div class="legend-content">
                <div class="legend-item">
                    <span class="legend-icon" aria-hidden="true">🟢</span>
                    <strong>Торгуется</strong> — активная торговля на бирже
                </div>
                <div class="legend-item">
                    <span class="legend-icon" aria-hidden="true">🏆</span>
                    <strong>Первый уровень</strong> — наиболее ликвидные облигации
                </div>
                <div class="legend-item">
                    <span class="legend-icon" aria-hidden="true">🥈</span>
                    <strong>Второй уровень</strong> — средняя ликвидность
                </div>
                <div class="legend-item">
                    <span class="legend-icon" aria-hidden="true">🥉</span>
                    <strong>Третий уровень</strong> — низкая ликвидность
                </div>
            </div>
        </div>

        <div class="info-notice">
            <strong>Источник данных:</strong> Справочная информация из базы данных DeepMax.ru.<br>
            Данные обновляются периодически и могут отличаться от текущих биржевых.<br><br>
            <strong>Структура каталога:</strong> Облигации сгруппированы по типам 
            (биржевая, корпоративная, ОФЗ и т.д.), затем по секторам экономики и эмитентам.
        </div>

    </div>
    <!-- КОНЕЦ ЛЕГЕНДА WRAPPER -->

</div>
<!-- КОНЕЦ PAGE-CONTENT-WRAPPER -->

</div>