<?php
/**
 * DeepMax.ru Logging System v1.1.0 ENHANCED
 * 
 * СИСТЕМА ЛОГИРОВАНИЯ С АВТОМАТИЧЕСКОЙ РОТАЦИЕЙ И РАСШИРЕННЫМИ ВОЗМОЖНОСТЯМИ
 * 
 * Возможности:
 * - Логи по датам (log_2025-10-25.txt)
 * - Автоматическая очистка старых логов (настраивается в cache.php)
 * - Защита от race conditions (LOCK_EX)
 * - Оптимизация производительности (кэш проверки директории)
 * - Поддержка трёх типов логов: log, input, error
 * - Ротация по размеру файла (NEW в v1.1.0)
 * - Автоматическое сжатие старых логов gzip (NEW в v1.1.0)
 * - Валидация размера сообщения (NEW в v1.1.0)
 * - Расширенная статистика логов (NEW в v1.1.0)
 * 
 * Версия: 1.1.0 ENHANCED
 * Дата: 2025-10-25
 * Кодировка: UTF-8 без BOM
 * 
 * ИЗМЕНЕНИЯ v1.1.0 ENHANCED:
 *   ✨ NEW: Кэш проверки директории (микрооптимизация)
 *   ✨ NEW: Валидация размера сообщения (защита от гигантских логов)
 *   ✨ NEW: Ротация по размеру файла (max_file_size)
 *   ✨ NEW: Автоматическое сжатие старых логов (gzip)
 *   ✨ NEW: Функция getLogStats() для аналитики
 *   ✨ NEW: Функция compressOldLogs() для ручного сжатия
 *   ✨ NEW: Функция rotateLogFile() для ротации по размеру
 *   ✨ NEW: Расширенные настройки через cache.php → $log_config
 *   ✅ SAFE: 100% обратная совместимость с v1.0.0
 *   ✅ SAFE: Все новые функции опциональны
 * 
 * КОНФИГУРАЦИЯ:
 *   Настройки берутся из cache.php → $log_config (v3.6.0 LOGS_ENHANCED)
 *   
 *   БАЗОВЫЕ (v1.0.0):
 *   - keep_days: количество дней хранения логов (по умолчанию 7)
 *   - use_date_format: использовать формат с датой (log_2025-10-25.txt)
 *   - log_dir: директория для хранения логов
 *   - cleanup_interval: интервал очистки (3600 секунд)
 *   
 *   РАСШИРЕННЫЕ (v1.1.0 ENHANCED):
 *   - max_message_size: максимальный размер сообщения (10 KB)
 *   - max_file_size: максимальный размер файла для ротации (10 MB)
 *   - compress_old_logs: автоматическое сжатие старых логов (true)
 *   - compress_age_days: возраст логов для сжатия (3 дня)
 *   - archive_timestamp: добавлять timestamp к архивам (true)
 *   - show_message_size: показывать размер в логе (false)
 * 
 * ИСПОЛЬЗОВАНИЕ:
 *   require_once 'logs.php';
 *   writeLog('Сообщение для лога', 'log');     // log_2025-10-25.txt
 *   writeLog('Входные данные', 'input');       // input_2025-10-25.txt
 *   writeLog('Ошибка в скрипте', 'error');     // error_2025-10-25.txt
 *   
 *   // Получить статистику логов
 *   $stats = getLogStats();
 *   print_r($stats);
 *   
 *   // Сжать старые логи вручную
 *   compressOldLogs(3);  // Сжать логи старше 3 дней
 * 
 * МИГРАЦИЯ СО СТАРОЙ СИСТЕМЫ:
 *   БЫЛО: file_put_contents(LOG_PATH, $message, FILE_APPEND);
 *   СТАЛО: writeLog($message, 'log');
 * 
 * ПРАВА ДОСТУПА:
 *   Владелец: www-root (или www-data в зависимости от PHP-FPM)
 *   Группа: www-data
 *   Права файла: 644 (rw-r--r--)
 *   Права папки /logs/: 775 (rwxrwxr-x)
 */

/**
 * Записать сообщение в лог с автоматической ротацией
 * 
 * УЛУЧШЕНИЯ v1.1.0:
 * - Кэш проверки директории (оптимизация)
 * - Валидация размера сообщения
 * - Ротация по размеру файла
 * - Автоматическое сжатие старых логов
 * 
 * Создаёт файл логов с датой в имени (log_2025-10-25.txt).
 * Автоматически очищает старые логи (настраивается через $log_config['keep_days']).
 * Использует LOCK_EX для предотвращения race conditions при параллельной записи.
 * 
 * @param string $message Сообщение для записи в лог
 * @param string $type Тип лога: 'log', 'input', 'error' (по умолчанию 'log')
 * @return bool True при успешной записи, false при ошибке
 */
function writeLog(string $message, string $type = 'log'): bool
{
    global $log_config;
    
    // Загружаем конфигурацию из cache.php если не загружена
    if (!isset($log_config)) {
        // Пытаемся загрузить cache.php
        $cache_file = __DIR__ . '/cache.php';
        if (file_exists($cache_file)) {
            require_once $cache_file;
        }
        
        // Если всё ещё не загружена - используем значения по умолчанию
        if (!isset($log_config)) {
            $log_config = [
                'rotation_enabled' => true,
                'keep_days' => 7,
                'use_date_format' => true,
                'log_dir' => __DIR__ . '/logs/',
                'cleanup_interval' => 3600,
                'max_message_size' => 10240,
                'max_file_size' => 10485760,
                'compress_old_logs' => true,
                'compress_age_days' => 3,
            ];
        }
    }
    
    // ═════════════════════════════════════════════════════════════════════
    // v1.1.0 ENHANCEMENT: Валидация размера сообщения
    // ═════════════════════════════════════════════════════════════════════
    $max_message_size = $log_config['max_message_size'] ?? 0;
    if ($max_message_size > 0 && strlen($message) > $max_message_size) {
        $original_size = strlen($message);
        $message = substr($message, 0, $max_message_size);
        $message .= "\n... [TRUNCATED: original size {$original_size} bytes, max allowed {$max_message_size} bytes]";
        error_log("[logs.php] writeLog() - WARNING: Message truncated from {$original_size} to {$max_message_size} bytes");
    }
    
    // Проверка типа лога
    $allowed_types = ['log', 'input', 'error'];
    if (!in_array($type, $allowed_types, true)) {
        error_log("[logs.php] writeLog() - WARNING: Unknown log type '{$type}', using 'log'");
        $type = 'log';
    }
    
    // Определяем директорию логов
    $log_dir = $log_config['log_dir'] ?? __DIR__ . '/logs/';
    
    // ═════════════════════════════════════════════════════════════════════
    // v1.1.0 OPTIMIZATION: Кэш проверки директории (микрооптимизация)
    // ═════════════════════════════════════════════════════════════════════
    static $dir_checked = [];
    
    if (!isset($dir_checked[$log_dir])) {
        // Создаём директорию если не существует
        if (!is_dir($log_dir)) {
            if (!@mkdir($log_dir, 0775, true)) {
                error_log("[logs.php] writeLog() - ERROR: Cannot create log directory: {$log_dir}");
                return false;
            }
        }
        $dir_checked[$log_dir] = true;
    }
    
    // Определяем имя файла
    $date = date('Y-m-d');
    $filename = $log_config['use_date_format'] ?? true
        ? "{$type}_{$date}.txt"
        : "{$type}.txt";
    
    $filepath = $log_dir . $filename;
    
    // ═════════════════════════════════════════════════════════════════════
    // v1.1.0 ENHANCEMENT: Ротация по размеру файла
    // ═════════════════════════════════════════════════════════════════════
    $max_file_size = $log_config['max_file_size'] ?? 0;
    if ($max_file_size > 0 && file_exists($filepath)) {
        $current_size = filesize($filepath);
        if ($current_size !== false && $current_size >= $max_file_size) {
            // Ротируем файл
            if (!rotateLogFile($filepath, $log_config)) {
                error_log("[logs.php] writeLog() - WARNING: Failed to rotate log file: {$filepath}");
            }
        }
    }
    
    // Формируем сообщение с временной меткой
    $timestamp = date('Y-m-d H:i:s');
    
    // Опционально: показывать размер сообщения (для отладки)
    $show_message_size = $log_config['show_message_size'] ?? false;
    if ($show_message_size) {
        $size = strlen($message);
        $log_message = "[{$timestamp}] [SIZE: {$size}] {$message}\n";
    } else {
        $log_message = "[{$timestamp}] {$message}\n";
    }
    
    // Записываем в файл с блокировкой (LOCK_EX предотвращает race conditions)
    $result = @file_put_contents($filepath, $log_message, FILE_APPEND | LOCK_EX);
    
    if ($result === false) {
        error_log("[logs.php] writeLog() - ERROR: Cannot write to log file: {$filepath}");
        return false;
    }
    
    // ═════════════════════════════════════════════════════════════════════
    // Автоматическая очистка и сжатие (v1.0.0 + v1.1.0 ENHANCEMENTS)
    // ═════════════════════════════════════════════════════════════════════
    static $last_cleanup = 0;
    $cleanup_interval = $log_config['cleanup_interval'] ?? 3600; // По умолчанию 1 час
    
    if ($log_config['rotation_enabled'] ?? true) {
        if (time() - $last_cleanup > $cleanup_interval) {
            // Очищаем старые логи
            cleanOldLogs($log_dir, $log_config['keep_days'] ?? 7);
            
            // v1.1.0: Сжимаем старые логи
            if ($log_config['compress_old_logs'] ?? true) {
                $compress_age = $log_config['compress_age_days'] ?? 3;
                compressOldLogs($compress_age);
            }
            
            $last_cleanup = time();
        }
    }
    
    return true;
}

/**
 * v1.1.0 NEW: Ротация лог-файла по размеру
 * 
 * Архивирует текущий файл с timestamp и создаёт новый.
 * Опционально сжимает архив в gzip.
 * 
 * @param string $filepath Путь к файлу для ротации
 * @param array $config Конфигурация логирования
 * @return bool True при успешной ротации, false при ошибке
 */
function rotateLogFile(string $filepath, array $config): bool
{
    if (!file_exists($filepath)) {
        return false;
    }
    
    // Формируем имя архива
    $add_timestamp = $config['archive_timestamp'] ?? true;
    
    if ($add_timestamp) {
        $archive_name = $filepath . '_' . date('His') . '.archive';
    } else {
        $archive_name = $filepath . '.archive';
    }
    
    // Переименовываем текущий файл в архив
    if (!@rename($filepath, $archive_name)) {
        error_log("[logs.php] rotateLogFile() - ERROR: Cannot rename {$filepath} to {$archive_name}");
        return false;
    }
    
    // Сжимаем архив если включено
    if ($config['compress_old_logs'] ?? true) {
        if (function_exists('gzencode')) {
            $content = @file_get_contents($archive_name);
            if ($content !== false) {
                $compressed = gzencode($content, 9); // Максимальное сжатие
                if ($compressed !== false) {
                    $gz_name = $archive_name . '.gz';
                    if (@file_put_contents($gz_name, $compressed) !== false) {
                        @unlink($archive_name); // Удаляем несжатый архив
                        error_log("[logs.php] rotateLogFile() - INFO: Rotated and compressed: {$gz_name}");
                        return true;
                    }
                }
            }
        }
    }
    
    error_log("[logs.php] rotateLogFile() - INFO: Rotated: {$archive_name}");
    return true;
}

/**
 * Удалить логи старше указанного количества дней
 * 
 * v1.0.0: Базовая функция
 * v1.1.0: Улучшена обработка сжатых файлов
 * 
 * Автоматически вызывается из writeLog() раз в час (или по настройке).
 * Удаляет только файлы с форматом имени: {type}_YYYY-MM-DD.txt
 * Также удаляет архивы и сжатые файлы.
 * 
 * @param string $dir Директория с логами
 * @param int $days Количество дней для хранения (по умолчанию 7)
 * @return int Количество удалённых файлов
 */
function cleanOldLogs(string $dir, int $days = 7): int
{
    if (!is_dir($dir)) {
        return 0;
    }
    
    // Ищем файлы с датами в имени (формат: {type}_YYYY-MM-DD.txt)
    // v1.1.0: Также ищем архивы и сжатые файлы
    $patterns = [
        $dir . '*_????-??-??.txt',           // Обычные логи
        $dir . '*_????-??-??.txt.archive*',  // Архивы
        $dir . '*_????-??-??.txt*.gz',       // Сжатые логи
    ];
    
    $cutoff_time = time() - ($days * 24 * 60 * 60);
    $deleted_count = 0;
    
    foreach ($patterns as $pattern) {
        $files = glob($pattern);
        
        if ($files === false) {
            error_log("[logs.php] cleanOldLogs() - WARNING: glob() failed for pattern: {$pattern}");
            continue;
        }
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $file_time = filemtime($file);
                
                if ($file_time === false) {
                    error_log("[logs.php] cleanOldLogs() - WARNING: Cannot get mtime for file: {$file}");
                    continue;
                }
                
                if ($file_time < $cutoff_time) {
                    if (@unlink($file)) {
                        $deleted_count++;
                    } else {
                        error_log("[logs.php] cleanOldLogs() - WARNING: Cannot delete file: {$file}");
                    }
                }
            }
        }
    }
    
    if ($deleted_count > 0) {
        error_log("[logs.php] cleanOldLogs() - INFO: Deleted {$deleted_count} old log file(s)");
    }
    
    return $deleted_count;
}

/**
 * v1.1.0 NEW: Сжать старые логи в gzip
 * 
 * Находит все несжатые логи старше указанного возраста и сжимает их.
 * Использует максимальное сжатие (уровень 9).
 * 
 * @param int $age_days Возраст логов для сжатия (в днях)
 * @return array Статистика: ['compressed' => int, 'failed' => int, 'saved_bytes' => int]
 */
function compressOldLogs(int $age_days = 3): array
{
    global $log_config;
    
    if (!function_exists('gzencode')) {
        error_log("[logs.php] compressOldLogs() - WARNING: gzencode() not available, skipping compression");
        return ['compressed' => 0, 'failed' => 0, 'saved_bytes' => 0];
    }
    
    if (!isset($log_config)) {
        $log_config = ['log_dir' => __DIR__ . '/logs/'];
    }
    
    $log_dir = $log_config['log_dir'] ?? __DIR__ . '/logs/';
    
    if (!is_dir($log_dir)) {
        return ['compressed' => 0, 'failed' => 0, 'saved_bytes' => 0];
    }
    
    // Ищем несжатые логи (исключаем уже сжатые .gz файлы)
    $pattern = $log_dir . '*_????-??-??.txt';
    $files = glob($pattern);
    
    if ($files === false) {
        return ['compressed' => 0, 'failed' => 0, 'saved_bytes' => 0];
    }
    
    $cutoff_time = time() - ($age_days * 24 * 60 * 60);
    $compressed_count = 0;
    $failed_count = 0;
    $saved_bytes = 0;
    
    foreach ($files as $file) {
        // Пропускаем файл за сегодня (он активен)
        if (strpos($file, date('Y-m-d')) !== false) {
            continue;
        }
        
        // Проверяем возраст файла
        $file_time = filemtime($file);
        if ($file_time === false || $file_time >= $cutoff_time) {
            continue;
        }
        
        // Читаем содержимое
        $content = @file_get_contents($file);
        if ($content === false) {
            $failed_count++;
            continue;
        }
        
        $original_size = strlen($content);
        
        // Сжимаем (уровень 9 = максимальное сжатие)
        $compressed = gzencode($content, 9);
        if ($compressed === false) {
            error_log("[logs.php] compressOldLogs() - ERROR: Failed to compress: {$file}");
            $failed_count++;
            continue;
        }
        
        $compressed_size = strlen($compressed);
        
        // Сохраняем сжатый файл
        $gz_file = $file . '.gz';
        if (@file_put_contents($gz_file, $compressed) === false) {
            error_log("[logs.php] compressOldLogs() - ERROR: Failed to write: {$gz_file}");
            $failed_count++;
            continue;
        }
        
        // Удаляем оригинал
        if (@unlink($file)) {
            $compressed_count++;
            $saved_bytes += ($original_size - $compressed_size);
        } else {
            error_log("[logs.php] compressOldLogs() - WARNING: Failed to delete original: {$file}");
            $failed_count++;
        }
    }
    
    if ($compressed_count > 0) {
        $saved_kb = round($saved_bytes / 1024, 2);
        error_log("[logs.php] compressOldLogs() - INFO: Compressed {$compressed_count} file(s), saved {$saved_kb} KB");
    }
    
    return [
        'compressed' => $compressed_count,
        'failed' => $failed_count,
        'saved_bytes' => $saved_bytes,
    ];
}

/**
 * Получить список логов за последние N дней
 * 
 * v1.0.0: Базовая функция
 * v1.1.0: Добавлена информация о сжатии
 * 
 * Вспомогательная функция для анализа или отображения списка логов.
 * 
 * @param string $type Тип лога: 'log', 'input', 'error', или '*' для всех типов
 * @param int $days Количество дней для поиска (по умолчанию 7)
 * @return array Массив путей к файлам логов, отсортированный по дате (новые первыми)
 */
function getRecentLogs(string $type = '*', int $days = 7): array
{
    global $log_config;
    
    // Загружаем конфигурацию если не загружена
    if (!isset($log_config)) {
        $cache_file = __DIR__ . '/cache.php';
        if (file_exists($cache_file)) {
            require_once $cache_file;
        }
        
        if (!isset($log_config)) {
            $log_config = ['log_dir' => __DIR__ . '/logs/'];
        }
    }
    
    $log_dir = $log_config['log_dir'] ?? __DIR__ . '/logs/';
    
    if (!is_dir($log_dir)) {
        return [];
    }
    
    // Формируем паттерн для поиска (включая сжатые файлы)
    $patterns = [
        $log_dir . ($type === '*' ? '*' : $type) . '_????-??-??.txt',
        $log_dir . ($type === '*' ? '*' : $type) . '_????-??-??.txt.gz',
    ];
    
    $cutoff_time = time() - ($days * 24 * 60 * 60);
    $recent_files = [];
    
    foreach ($patterns as $pattern) {
        $files = glob($pattern);
        
        if ($files === false) {
            continue;
        }
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $file_time = filemtime($file);
                
                if ($file_time !== false && $file_time >= $cutoff_time) {
                    $is_compressed = substr($file, -3) === '.gz';
                    
                    $recent_files[] = [
                        'path' => $file,
                        'name' => basename($file),
                        'size' => filesize($file),
                        'time' => $file_time,
                        'date' => date('Y-m-d H:i:s', $file_time),
                        'compressed' => $is_compressed, // v1.1.0
                    ];
                }
            }
        }
    }
    
    // Сортируем по дате (новые первыми)
    usort($recent_files, function($a, $b) {
        return $b['time'] - $a['time'];
    });
    
    return $recent_files;
}

/**
 * v1.1.0 NEW: Получить статистику логов
 * 
 * Возвращает детальную информацию о логах:
 * - Общее количество файлов
 * - Общий размер
 * - Количество сжатых файлов
 * - Экономия от сжатия
 * - Старейший и новейший лог
 * 
 * @return array Ассоциативный массив со статистикой
 */
function getLogStats(): array
{
    global $log_config;
    
    if (!isset($log_config)) {
        $cache_file = __DIR__ . '/cache.php';
        if (file_exists($cache_file)) {
            require_once $cache_file;
        }
        
        if (!isset($log_config)) {
            $log_config = ['log_dir' => __DIR__ . '/logs/'];
        }
    }
    
    $log_dir = $log_config['log_dir'] ?? __DIR__ . '/logs/';
    
    if (!is_dir($log_dir)) {
        return [
            'total_files' => 0,
            'total_size' => 0,
            'total_size_mb' => 0,
            'compressed_files' => 0,
            'uncompressed_files' => 0,
            'oldest_log' => null,
            'newest_log' => null,
            'types' => [],
        ];
    }
    
    $patterns = [
        $log_dir . '*_????-??-??.txt',
        $log_dir . '*_????-??-??.txt.gz',
    ];
    
    $total_files = 0;
    $total_size = 0;
    $compressed_files = 0;
    $uncompressed_files = 0;
    $oldest_time = PHP_INT_MAX;
    $newest_time = 0;
    $oldest_file = null;
    $newest_file = null;
    $types = [];
    
    foreach ($patterns as $pattern) {
        $files = glob($pattern);
        
        if ($files === false) {
            continue;
        }
        
        foreach ($files as $file) {
            if (!is_file($file)) {
                continue;
            }
            
            $total_files++;
            $size = filesize($file);
            $total_size += $size;
            
            $is_compressed = substr($file, -3) === '.gz';
            if ($is_compressed) {
                $compressed_files++;
            } else {
                $uncompressed_files++;
            }
            
            // Определяем тип лога
            $basename = basename($file);
            preg_match('/^([a-z]+)_/', $basename, $matches);
            $type = $matches[1] ?? 'unknown';
            
            if (!isset($types[$type])) {
                $types[$type] = ['count' => 0, 'size' => 0];
            }
            $types[$type]['count']++;
            $types[$type]['size'] += $size;
            
            // Находим старейший и новейший
            $file_time = filemtime($file);
            if ($file_time !== false) {
                if ($file_time < $oldest_time) {
                    $oldest_time = $file_time;
                    $oldest_file = $basename;
                }
                if ($file_time > $newest_time) {
                    $newest_time = $file_time;
                    $newest_file = $basename;
                }
            }
        }
    }
    
    return [
        'total_files' => $total_files,
        'total_size' => $total_size,
        'total_size_mb' => round($total_size / 1024 / 1024, 2),
        'compressed_files' => $compressed_files,
        'uncompressed_files' => $uncompressed_files,
        'compression_ratio' => $total_files > 0 ? round(($compressed_files / $total_files) * 100, 1) : 0,
        'oldest_log' => $oldest_file,
        'oldest_date' => $oldest_time !== PHP_INT_MAX ? date('Y-m-d H:i:s', $oldest_time) : null,
        'newest_log' => $newest_file,
        'newest_date' => $newest_time > 0 ? date('Y-m-d H:i:s', $newest_time) : null,
        'types' => $types,
    ];
}

/**
 * Принудительная очистка ВСЕХ логов (для обслуживания)
 * 
 * v1.0.0: Базовая функция
 * v1.1.0: Также удаляет сжатые и архивные файлы
 * 
 * ВНИМАНИЕ: Удаляет ВСЕ файлы логов, независимо от возраста!
 * Используйте только для ручного обслуживания или в крайних случаях.
 * 
 * @return int Количество удалённых файлов
 */
function clearAllLogs(): int
{
    global $log_config;
    
    if (!isset($log_config)) {
        $cache_file = __DIR__ . '/cache.php';
        if (file_exists($cache_file)) {
            require_once $cache_file;
        }
        
        if (!isset($log_config)) {
            $log_config = ['log_dir' => __DIR__ . '/logs/'];
        }
    }
    
    $log_dir = $log_config['log_dir'] ?? __DIR__ . '/logs/';
    
    if (!is_dir($log_dir)) {
        return 0;
    }
    
    // v1.1.0: Удаляем все типы файлов логов
    $patterns = [
        $log_dir . '*.txt',
        $log_dir . '*.gz',
        $log_dir . '*.archive',
        $log_dir . '*.archive.gz',
    ];
    
    $deleted_count = 0;
    
    foreach ($patterns as $pattern) {
        $files = glob($pattern);
        
        if ($files === false) {
            continue;
        }
        
        foreach ($files as $file) {
            // Не удаляем папку debug/
            if (strpos($file, '/debug/') !== false) {
                continue;
            }
            
            if (is_file($file) && @unlink($file)) {
                $deleted_count++;
            }
        }
    }
    
    error_log("[logs.php] clearAllLogs() - INFO: Deleted {$deleted_count} log file(s)");
    
    return $deleted_count;
}

/**
 * ═══════════════════════════════════════════════════════════════════════════
 * END logs.php v1.1.0 ENHANCED ✅
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 1.1.0 ENHANCED
 * Дата: 2025-10-25
 * Статус: PRODUCTION READY - ENHANCED
 * 
 * ДОБАВЛЕНО v1.1.0 ENHANCED:
 *   ✨ NEW: Кэш проверки директории (микрооптимизация производительности)
 *   ✨ NEW: Валидация размера сообщения (защита от гигантских логов)
 *   ✨ NEW: Ротация по размеру файла (max_file_size)
 *   ✨ NEW: Автоматическое сжатие старых логов gzip (compress_old_logs)
 *   ✨ NEW: Функция rotateLogFile() - ротация с архивированием
 *   ✨ NEW: Функция compressOldLogs() - ручное сжатие старых логов
 *   ✨ NEW: Функция getLogStats() - детальная статистика логов
 *   ✨ NEW: Улучшена cleanOldLogs() - работает с архивами и сжатыми
 *   ✨ NEW: Улучшена getRecentLogs() - показывает сжатые файлы
 *   ✨ NEW: Улучшена clearAllLogs() - удаляет все типы файлов
 *   
 *   ✅ SAFE: 100% обратная совместимость с v1.0.0
 *   ✅ SAFE: Все новые функции опциональны
 *   ✅ SAFE: Старая конфигурация продолжит работать
 * 
 * ВОЗМОЖНОСТИ v1.0.0 (сохранены):
 *   ✅ Логи по датам (log_2025-10-25.txt)
 *   ✅ Автоматическая очистка старых логов (настраивается)
 *   ✅ Защита от race conditions (LOCK_EX)
 *   ✅ Оптимизация производительности (cleanOldLogs раз в час)
 *   ✅ Поддержка трёх типов: log, input, error
 *   ✅ Вспомогательные функции для анализа логов
 * 
 * ОСНОВНЫЕ ФУНКЦИИ:
 *   v1.0.0:
 *   - writeLog(string $message, string $type = 'log'): bool
 *     Записать сообщение в лог с автоматической ротацией
 *   
 *   - cleanOldLogs(string $dir, int $days = 7): int
 *     Удалить логи старше N дней
 *   
 *   - getRecentLogs(string $type = '*', int $days = 7): array
 *     Получить список логов за последние N дней
 *   
 *   - clearAllLogs(): int
 *     Принудительно удалить все логи (для обслуживания)
 *   
 *   v1.1.0 NEW:
 *   - rotateLogFile(string $filepath, array $config): bool
 *     Ротация файла по размеру с архивированием
 *   
 *   - compressOldLogs(int $age_days = 3): array
 *     Сжать старые логи в gzip
 *   
 *   - getLogStats(): array
 *     Получить детальную статистику логов
 * 
 * КОНФИГУРАЦИЯ:
 *   Настройки в cache.php v3.6.0 LOGS_ENHANCED → $log_config:
 *   
 *   БАЗОВЫЕ:
 *   - rotation_enabled: bool - включить ротацию
 *   - keep_days: int - количество дней хранения (7)
 *   - use_date_format: bool - формат log_2025-10-25.txt
 *   - log_dir: string - директория логов
 *   - cleanup_interval: int - интервал очистки (3600 сек)
 *   
 *   РАСШИРЕННЫЕ (v1.1.0):
 *   - max_message_size: int - макс размер сообщения (10 KB)
 *   - max_file_size: int - макс размер файла (10 MB)
 *   - compress_old_logs: bool - автосжатие (true)
 *   - compress_age_days: int - возраст для сжатия (3 дня)
 *   - archive_timestamp: bool - timestamp в архивах (true)
 *   - show_message_size: bool - показывать размер (false)
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Кэш проверки директории (v1.1.0)
 *   - Автоочистка выполняется только раз в cleanup_interval
 *   - LOCK_EX предотвращает коллизии при параллельной записи
 *   - Минимальное влияние на скорость работы скриптов
 *   - Сжатие логов экономит до 90% места на диске
 * 
 * СОВМЕСТИМОСТЬ:
 *   - Работает с cache.php v3.6.0 LOGS_ENHANCED
 *   - Обратная совместимость с v1.0.0
 *   - PHP 7.4+, PHP 8.0+, PHP 8.1+
 *   - Требует gzencode() для сжатия (опционально)
 * 
 * ПРАВА ДОСТУПА:
 *   Файл: 644 (rw-r--r--)
 *   Владелец: www-root
 *   Группа: www-data
 * 
 *   Папка /logs/: 775 (rwxrwxr-x)
 *   Владелец: www-root
 *   Группа: www-data
 * 
 * ИСПОЛЬЗОВАНИЕ:
 *   require_once 'logs.php';
 *   
 *   // Простая запись (v1.0.0)
 *   writeLog('Сообщение');
 *   writeLog('Входные данные', 'input');
 *   writeLog('Ошибка', 'error');
 *   
 *   // Получить список логов (v1.0.0)
 *   $logs = getRecentLogs('log', 7);
 *   
 *   // Получить статистику (v1.1.0 NEW)
 *   $stats = getLogStats();
 *   echo "Всего файлов: {$stats['total_files']}\n";
 *   echo "Общий размер: {$stats['total_size_mb']} MB\n";
 *   echo "Сжато: {$stats['compression_ratio']}%\n";
 *   
 *   // Сжать старые логи вручную (v1.1.0 NEW)
 *   $result = compressOldLogs(3);
 *   echo "Сжато файлов: {$result['compressed']}\n";
 *   echo "Сэкономлено: " . round($result['saved_bytes']/1024, 2) . " KB\n";
 *   
 *   // Принудительная очистка (v1.0.0)
 *   clearAllLogs();
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */