/**
 * DEEPMAX FEATURES v1.0 - Page-specific features (conditionally loaded)
 * deepmax.ru project
 * @version 1.0 | @date 2025-11-03 | @status PRODUCTION READY ✅
 */

(function() {
    'use strict';
    console.log('[DeepMax Features] v1.0 - Initializing...');
    

    // [SECTION: Stock Page Features - from stock-v3.1.3.js]
    // (IIFE wrapper removed for merging)
    
    if (window.StockV6 && window.StockV6.initialized) {
        console.warn('[StockV6] Already initialized');
        return;
    }
    
    var eventHandlers = [];
    
    // =======================================================================
    // POLYFILLS
    // =======================================================================
    
    if (window.NodeList && !NodeList.prototype.forEach) {
        NodeList.prototype.forEach = Array.prototype.forEach;
    }
    
    if (!Element.prototype.matches) {
        Element.prototype.matches = 
            Element.prototype.msMatchesSelector || 
            Element.prototype.webkitMatchesSelector;
    }
    
    if (!Element.prototype.closest) {
        Element.prototype.closest = function(selector) {
            var element = this;
            
            while (element && element.nodeType === 1) {
                if (element.matches(selector)) {
                    return element;
                }
                element = element.parentElement;
            }
            
            return null;
        };
    }
    
    if (!String.prototype.trim) {
        String.prototype.trim = function() {
            return this.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, '');
        };
    }
    
    // =======================================================================
    // UTILITIES
    // =======================================================================
    
    function addTrackedEventListener(element, event, handler) {
        element.addEventListener(event, handler);
        eventHandlers.push({
            element: element,
            event: event,
            handler: handler
        });
    }
    
    function checkCSSClass(className) {
        var testElement = document.createElement('div');
        testElement.className = className;
        testElement.style.position = 'absolute';
        testElement.style.visibility = 'hidden';
        document.body.appendChild(testElement);
        
        var computedStyle = window.getComputedStyle(testElement);
        var transitionValue = computedStyle.transition || computedStyle.webkitTransition || '';
        var hasTransition = transitionValue.indexOf('max-height') !== -1 || 
                           transitionValue.indexOf('all') !== -1;
        
        document.body.removeChild(testElement);
        
        return hasTransition;
    }
    
    // =======================================================================
    // COPY FUNCTIONS
    // =======================================================================
    
    function copyTextV6(elementId) {
        var element = document.getElementById(elementId);
        
        if (!element) {
            console.error('[StockV6] Element not found:', elementId);
            showToastV6('Error: element not found');
            return false;
        }
        
        var text = element.textContent || element.innerText;
        
        if (!text || text.trim() === '') {
            console.error('[StockV6] Text is empty');
            showToastV6('Error: no text');
            return false;
        }
        
        text = text.trim();
        
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text)
                .then(function() {
                    showToastV6('Copied: ' + text);
                })
                .catch(function(err) {
                    console.error('[StockV6] Clipboard API error:', err);
                    fallbackCopyV6(text);
                });
        } else {
            fallbackCopyV6(text);
        }
        
        return true;
    }
    
    function fallbackCopyV6(text) {
        var textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        textarea.setAttribute('readonly', '');
        textarea.setAttribute('aria-hidden', 'true');
        
        document.body.appendChild(textarea);
        textarea.focus();
        textarea.select();
        
        var successful = false;
        
        try {
            successful = document.execCommand('copy');
            if (successful) {
                showToastV6('Copied: ' + text);
            } else {
                showToastV6('Copy error');
            }
        } catch (err) {
            console.error('[StockV6] execCommand error:', err);
            showToastV6('Copy error');
        }
        
        document.body.removeChild(textarea);
        return successful;
    }
    
    function showToastV6(message) {
        var toast = document.getElementById('toast');
        var toastText = document.getElementById('toast-text');
        
        if (!toast || !toastText) {
            console.error('[StockV6] Toast elements not found');
            return;
        }
        
        toastText.textContent = message;
        toast.classList.add('show');
        
        setTimeout(function() {
            toast.classList.remove('show');
        }, 2000);
    }
    
    // =======================================================================
    // INITIALIZE COPY BUTTONS (FIXED)
    // =======================================================================
    
    function initCopyButtons() {
        var copyButtons = document.querySelectorAll('.copy-btn');
        var count = 0;
        var warnings = [];
        
        // Check if CSS for .copying exists
        var hasCopyingCSS = false;
        var testBtn = document.createElement('button');
        testBtn.className = 'copy-btn copying';
        testBtn.style.position = 'absolute';
        testBtn.style.visibility = 'hidden';
        document.body.appendChild(testBtn);
        
        var computedStyle = window.getComputedStyle(testBtn);
        hasCopyingCSS = computedStyle.transform && computedStyle.transform !== 'none';
        
        document.body.removeChild(testBtn);
        
        if (!hasCopyingCSS) {
            console.warn('[StockV6] CSS for .copy-btn.copying not found, using fallback');
        }
        
        for (var i = 0; i < copyButtons.length; i++) {
            (function(button) {
                var target = button.getAttribute('data-target');
                
                if (!target) {
                    console.warn('[StockV6] Button without data-target:', button);
                    return;
                }
                
                // FIX: Check that target element exists
                var targetElement = document.getElementById(target);
                if (!targetElement) {
                    warnings.push('Target element not found: ' + target);
                    console.warn('[StockV6] Target element not found:', target);
                    // Still add handler, copyTextV6 will show error on click
                }
                
                function handleClick(e) {
                    e.preventDefault();
                    
                    // Visual feedback
                    button.classList.add('copying');
                    
                    // FIX: Fallback styles if CSS not loaded
                    if (!hasCopyingCSS) {
                        button.style.transform = 'scale(0.95)';
                    }
                    
                    setTimeout(function() {
                        button.classList.remove('copying');
                        if (!hasCopyingCSS) {
                            button.style.transform = '';
                        }
                    }, 300);
                    
                    // Copy
                    copyTextV6(target);
                }
                
                addTrackedEventListener(button, 'click', handleClick);
                count++;
                
            })(copyButtons[i]);
        }
        
        if (count > 0) {
            console.log('[StockV6] Initialized ' + count + ' copy buttons');
        }
        
        if (warnings.length > 0) {
            console.warn('[StockV6] Found issues with ' + warnings.length + ' buttons');
            for (var j = 0; j < warnings.length; j++) {
                console.warn('  ⚠️  ' + warnings[j]);
            }
        }
    }
    
    // =======================================================================
    // DETAILS ANIMATION
    // =======================================================================
    
    function initDetailsAnimation() {
        var detailsElements = document.querySelectorAll('.details-section');
        
        if (detailsElements.length === 0) {
            return;
        }
        
        var hasCSS = checkCSSClass('details-animated');
        
        if (!hasCSS) {
            console.warn('[StockV6] CSS class .details-animated not found, using fallback');
        }
        
        for (var i = 0; i < detailsElements.length; i++) {
            (function(details) {
                var content = details.querySelector('.details-content');
                var summary = details.querySelector('.details-summary');
                var activeTimeout = null;
                
                if (!content) {
                    return;
                }
                
                content.classList.add('details-animated');
                
                if (!hasCSS) {
                    content.style.transition = 'max-height 0.3s ease, opacity 0.3s ease';
                    content.style.overflow = 'hidden';
                }
                
                function handleTransitionEnd(event) {
                    if (event.propertyName !== 'max-height') {
                        return;
                    }
                    
                    if (details.open) {
                        content.style.maxHeight = 'none';
                    }
                }
                
                addTrackedEventListener(content, 'transitionend', handleTransitionEnd);
                
                function handleToggle() {
                    var isOpen = details.open;
                    
                    if (activeTimeout) {
                        clearTimeout(activeTimeout);
                        activeTimeout = null;
                    }
                    
                    if (isOpen) {
                        content.style.maxHeight = 'none';
                        var height = content.scrollHeight;
                        
                        if (height <= 0) {
                            console.warn('[StockV6] scrollHeight = 0');
                            return;
                        }
                        
                        content.style.maxHeight = '0';
                        var forceReflow = content.offsetHeight;
                        void forceReflow;
                        
                        content.style.maxHeight = height + 'px';
                        
                    } else {
                        var currentHeight = content.scrollHeight;
                        
                        if (currentHeight > 0) {
                            content.style.maxHeight = currentHeight + 'px';
                            var forceReflow2 = content.offsetHeight;
                            void forceReflow2;
                            content.style.maxHeight = '0';
                        }
                    }
                    
                    if (summary) {
                        summary.setAttribute('aria-expanded', isOpen ? 'true' : 'false');
                    }
                }
                
                addTrackedEventListener(details, 'toggle', handleToggle);
                
                if (details.open) {
                    content.style.maxHeight = 'none';
                } else {
                    content.style.maxHeight = '0';
                }
                
                if (summary) {
                    if (!summary.hasAttribute('tabindex')) {
                        summary.setAttribute('tabindex', '0');
                    }
                    if (!summary.hasAttribute('role')) {
                        summary.setAttribute('role', 'button');
                    }
                    summary.setAttribute('aria-expanded', details.open ? 'true' : 'false');
                }
                
            })(detailsElements[i]);
        }
        
        console.log('[StockV6] Initialized ' + detailsElements.length + ' details elements');
    }
    
    // =======================================================================
    // ACCESSIBILITY
    // =======================================================================
    
    function initAccessibility() {
        var count = 0;
        
        var metricCards = document.querySelectorAll('.metric-card');
        for (var i = 0; i < metricCards.length; i++) {
            if (!metricCards[i].hasAttribute('role')) {
                metricCards[i].setAttribute('role', 'article');
                metricCards[i].setAttribute('aria-label', 'Metric');
                count++;
            }
        }
        
        var idCards = document.querySelectorAll('.id-card');
        for (var j = 0; j < idCards.length; j++) {
            if (!idCards[j].hasAttribute('role')) {
                idCards[j].setAttribute('role', 'group');
                idCards[j].setAttribute('aria-label', 'Identifier');
                count++;
            }
        }
        
        if (count > 0) {
            console.log('[StockV6] Added ARIA attributes for ' + count + ' elements');
        }
    }
    
    // =======================================================================
    // IMAGE FALLBACK
    // =======================================================================
    
    function initImageFallback() {
        var logoImg = document.querySelector('.emitent-logo img');
        
        if (logoImg && !logoImg.dataset.errorHandled) {
            logoImg.dataset.errorHandled = 'true';
            
            function handleImageError() {
                var parent = this.parentElement;
                
                if (parent && !parent.querySelector('.logo-placeholder')) {
                    this.style.display = 'none';
                    
                    var placeholder = document.createElement('div');
                    placeholder.className = 'logo-placeholder';
                    placeholder.style.cssText = 
                        'width:100%;height:100%;display:flex;align-items:center;' +
                        'justify-content:center;font-size:32px;color:#7a9292;' +
                        'background:#e0f2f2;border-radius:8px;';
                    placeholder.textContent = '🏢';
                    placeholder.setAttribute('aria-label', 'Emitent logo');
                    
                    parent.appendChild(placeholder);
                    
                    console.log('[StockV6] Logo fallback loaded');
                }
            }
            
            addTrackedEventListener(logoImg, 'error', handleImageError);
        }
    }
    
    // =======================================================================
    // CLEANUP
    // =======================================================================
    
    function cleanup() {
        console.log('[StockV6] Cleanup: removing ' + eventHandlers.length + ' handlers');
        
        for (var i = 0; i < eventHandlers.length; i++) {
            var handler = eventHandlers[i];
            try {
                handler.element.removeEventListener(handler.event, handler.handler);
            } catch (e) {
                // Ignore
            }
        }
        
        eventHandlers.length = 0;
    }
    
    // =======================================================================
    // MAIN INITIALIZATION
    // =======================================================================
    
    function init() {
        try {
            console.log('[StockV6] Initialization v6.0.4 CSP COMPATIBLE ENGLISH...');
            
            // Start modules
            initCopyButtons();
            initDetailsAnimation();
            initAccessibility();
            initImageFallback();
            
            // Register cleanup
            addTrackedEventListener(window, 'beforeunload', cleanup);
            
            // Export functions to window
            window.copyTextV6 = copyTextV6;
            window.showToastV6 = showToastV6;
            
            // Mark success
            window.StockV6.initialized = true;
            
            console.log('[StockV6] ✅ Initialization completed successfully');
            
        } catch (error) {
            console.error('[StockV6] ❌ Initialization error:', error);
            
            window.StockV6.initialized = true;
            window.StockV6.error = error;
        }
    }
    
    // =======================================================================
    // STARTUP - v7.2: Wait for CSS before initializing
    // =======================================================================
    
    function initStockWhenReady() {
        // Wait for CSS to load
        if (!document.body.classList.contains('css-ready')) {
            setTimeout(initStockWhenReady, 50);
            return;
        }
        init();
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initStockWhenReady);
    } else {
        initStockWhenReady();
    }
    
    // =======================================================================
    // PUBLIC API
    // =======================================================================
    
    window.StockV6 = window.StockV6 || {};
    window.StockV6.version = '6.0.4-csp-english';
    window.StockV6.initialized = false;
    window.StockV6.error = null;
    window.StockV6.cleanup = cleanup;
    // [END SECTION: Stock Page Features]



    // [SECTION: Bond Countdown + Navigation - from bond-countdown.js]
    // (IIFE wrapper removed for merging)

    // ═══════════════════════════════════════════════════════════════════════
    // CONFIGURATION
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Debug mode toggle
     * Set to true for development logging, false for production
     */
    var DEBUG = true;

    /**
     * Scroll animation duration in milliseconds
     * Default: 800ms for smooth, comfortable scroll
     */
    var SCROLL_DURATION = 800;

    /**
     * Scroll offset in pixels (for sticky headers/breadcrumbs)
     * Default: 160px (header + breadcrumbs + spacing)
     * v1.6.0: Increased from 150px to 160px for better clearance
     */
    var SCROLL_OFFSET = 160;

    /**
     * Console logging helper (respects DEBUG flag)
     */
    function log(message, data) {
        if (DEBUG && typeof console !== 'undefined' && console.log) {
            var prefix = '[bond-countdown.js]';
            if (data !== undefined) {
                console.log(prefix + ' ' + message, data);
            } else {
                console.log(prefix + ' ' + message);
            }
        }
    }

    // ═══════════════════════════════════════════════════════════════════════
    // POLYFILLS FOR IE11 COMPATIBILITY (RULE 17)
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * NodeList.forEach polyfill for IE11
     */
    if (window.NodeList && typeof NodeList.prototype.forEach !== 'function') {
        NodeList.prototype.forEach = function(callback, thisArg) {
            thisArg = thisArg || window;
            for (var i = 0; i < this.length; i++) {
                callback.call(thisArg, this[i], i, this);
            }
        };
        log('NodeList.forEach polyfill installed');
    }

    // ═══════════════════════════════════════════════════════════════════════
    // MEMORY MANAGEMENT (RULE 13)
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Array to track all active intervals
     * 
     * RULE 13: Memory leaks prevention
     */
    var intervals = [];

    /**
     * Array to track all event listeners
     * Structure: [{element: DOMElement, event: 'click', handler: function}]
     * 
     * RULE 13: Memory leaks prevention
     */
    var eventHandlers = [];

    /**
     * Flag to prevent duplicate beforeunload listeners
     */
    var cleanupRegistered = false;

    /**
     * Add interval to tracked list
     */
    function trackInterval(intervalId) {
        intervals.push(intervalId);
        log('Interval tracked. Total active:', intervals.length);
    }

    /**
     * Add event listener and track it for cleanup
     * 
     * RULE 13: Memory leaks prevention
     * RULE 14: Element checks before addEventListener
     */
    function addTrackedEventListener(element, event, handler) {
        if (!element) {
            log('WARNING: Attempted to add listener to null element');
            return;
        }
        
        element.addEventListener(event, handler);
        eventHandlers.push({
            element: element,
            event: event,
            handler: handler
        });
        log('Event listener tracked. Total active:', eventHandlers.length);
    }

    /**
     * Cleanup all intervals and event listeners on page unload
     */
    function cleanup() {
        log('Cleaning up intervals. Total:', intervals.length);
        for (var i = 0; i < intervals.length; i++) {
            clearInterval(intervals[i]);
        }
        intervals = [];
        
        log('Cleaning up event listeners. Total:', eventHandlers.length);
        for (var j = 0; j < eventHandlers.length; j++) {
            var handler = eventHandlers[j];
            handler.element.removeEventListener(handler.event, handler.handler);
        }
        eventHandlers = [];
        
        log('Cleanup complete');
    }

    /**
     * Register cleanup handler (once only)
     */
    function registerCleanup() {
        if (!cleanupRegistered) {
            window.addEventListener('beforeunload', cleanup);
            cleanupRegistered = true;
            log('Cleanup handler registered');
        }
    }

    // Register cleanup on script load
    registerCleanup();

    // ═══════════════════════════════════════════════════════════════════════
    // PROGRESS BARS INITIALIZATION v1.3.0 (CSP COMPATIBLE + TIMELINE SUPPORT)
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Initialize progress bars from data-progress attribute
     * 
     * v1.3.0 NEW: Special handling for .timeline-progress elements
     * Timeline progress bars need a small delay to trigger CSS transition.
     * 
     * RULE 11: CSP compatible - no inline styles in HTML
     * Width is set via JavaScript from data-progress attribute.
     * 
     * Types of progress bars:
     *   1. Regular (.progress-bar-fill, .coupon-progress-bar-fill, etc.)
     *      - Set instantly (no delay)
     *      - May have instant or fast transitions in CSS
     * 
     *   2. Timeline (.timeline-progress)
     *      - Set with 100ms delay to trigger CSS transition
     *      - CSS has transition: width 5s ease
     *      - Creates smooth animation effect
     * 
     * Why delay is needed:
     *   Browser optimizations can "collapse" rapid style changes.
     *   Setting width: 0 then width: 45% in same frame = no transition.
     *   Adding setTimeout ensures browser commits initial state first.
     */
    function initProgressBars() {
        var progressBars = document.querySelectorAll('[data-progress]');
        
        log('Initializing progress bars. Found:', progressBars.length);
        
        progressBars.forEach(function(bar) {
            var progress = bar.getAttribute('data-progress');
            
            // Validate progress value
            if (progress !== null && progress !== '') {
                var progressValue = parseFloat(progress);
                
                if (!isNaN(progressValue)) {
                    // Clamp to 0-100 range
                    progressValue = Math.max(0, Math.min(100, progressValue));
                    
                    // Check if this is a timeline progress bar
                    var isTimeline = bar.classList.contains('timeline-progress');
                    
                    if (isTimeline) {
                        // Timeline progress: delay to trigger transition
                        log('Timeline progress bar detected, using delay for animation');
                        
                        // Set initial width to 0 (should already be 0 from CSS)
                        bar.style.width = '0%';
                        
                        // After 100ms, set target width (triggers 5s transition)
                        setTimeout(function() {
                            bar.style.width = progressValue + '%';
                            log('Timeline progress animated to ' + progressValue + '% (5s transition)');
                        }, 100);
                    } else {
                        // Regular progress: set instantly
                        bar.style.width = progressValue + '%';
                        log('Regular progress bar set to ' + progressValue + '%');
                    }
                }
            }
        });
    }

    // ═══════════════════════════════════════════════════════════════════════
    // COUNTDOWN TIMER FUNCTIONS
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Pluralize Russian numerals correctly
     * 
     * Russian has 3 forms for different numbers:
     *   - "one" form: 1, 21, 31... (год, месяц, день)
     *   - "two" form: 2-4, 22-24... (года, месяца, дня)
     *   - "five" form: 5-20, 25-30... (лет, месяцев, дней)
     */
    function pluralize(number, one, two, five) {
        var n = Math.abs(number) % 100;
        var n1 = n % 10;
        
        if (n > 10 && n < 20) {
            return five;
        }
        if (n1 > 1 && n1 < 5) {
            return two;
        }
        if (n1 === 1) {
            return one;
        }
        return five;
    }

    /**
     * Update label display for time unit (show correct form, hide others)
     * 
     * v1.2.0 CSP-PERFECT: Uses classList.add/remove instead of style.display
     * This ensures full CSP compatibility (RULE 11).
     */
    function updateLabel(container, unit, value) {
        var labelContainer = container.querySelector('.timer-label');
        if (!labelContainer) return;

        var spans = labelContainer.querySelectorAll('span[data-label]');
        
        var forms = {
            'years': ['год', 'года', 'лет'],
            'months': ['месяц', 'месяца', 'месяцев'],
            'days': ['день', 'дня', 'дней'],
            'hours': ['час', 'часа', 'часов'],
            'minutes': ['минута', 'минуты', 'минут'],
            'seconds': ['секунда', 'секунды', 'секунд']
        };

        if (!forms[unit]) return;

        var correctForm = pluralize(value, forms[unit][0], forms[unit][1], forms[unit][2]);

        for (var i = 0; i < spans.length; i++) {
            var span = spans[i];
            var labelType = span.getAttribute('data-label');
            
            if ((labelType === 'one' && correctForm === forms[unit][0]) ||
                (labelType === 'two' && correctForm === forms[unit][1]) ||
                (labelType === 'five' && correctForm === forms[unit][2])) {
                span.classList.remove('hidden');
            } else {
                span.classList.add('hidden');
            }
        }
    }

    /**
     * Animate value change with scale effect
     */
    function animateValue(element, newValue) {
        var currentValue = element.textContent;
        
        if (String(currentValue) !== String(newValue)) {
            element.classList.add('animating');
            element.textContent = newValue;
            
            setTimeout(function() {
                element.classList.remove('animating');
            }, 150);
        }
    }

    /**
     * Update countdown timer display
     * 
     * Calculates time remaining and updates all 6 time blocks.
     * 
     * @returns {boolean} - false if time expired (stops interval)
     */
    function updateCountdown(container) {
        var maturity = parseInt(container.getAttribute('data-maturity'), 10);
        if (!maturity) return false;

        var now = Math.floor(Date.now() / 1000);
        var diff = maturity - now;

        // Time expired
        if (diff <= 0) {
            log('Bond matured, showing expired message');
            
            container.textContent = '';
            
            var expiredDiv = document.createElement('div');
            expiredDiv.className = 'countdown-expired';
            
            var icon = document.createElement('span');
            icon.className = 'countdown-expired-icon';
            icon.textContent = '💎';
            
            var text = document.createElement('strong');
            text.className = 'countdown-expired-text';
            text.textContent = 'Облигация погашена';
            
            expiredDiv.appendChild(icon);
            expiredDiv.appendChild(text);
            container.appendChild(expiredDiv);
            
            return false;
        }

        // Convert to time units (month = 30 days)
        var seconds = diff;
        var minutes = Math.floor(seconds / 60);
        var hours = Math.floor(minutes / 60);
        var days = Math.floor(hours / 24);
        var months = Math.floor(days / 30);
        var years = Math.floor(months / 12);

        seconds = seconds % 60;
        minutes = minutes % 60;
        hours = hours % 24;
        days = days % 30;
        months = months % 12;

        var units = {
            'years': years,
            'months': months,
            'days': days,
            'hours': hours,
            'minutes': minutes,
            'seconds': seconds
        };

        // Update each time block
        for (var unit in units) {
            if (units.hasOwnProperty(unit)) {
                var valueElement = container.querySelector('[data-unit="' + unit + '"]');
                if (valueElement) {
                    animateValue(valueElement, units[unit]);
                    
                    var block = valueElement.closest('.timer-block');
                    if (block) {
                        updateLabel(block, unit, units[unit]);
                    }
                }
            }
        }

        return true;
    }

    /**
     * Initialize countdown timer(s) on the page
     * 
     * Protection against re-initialization via data-initialized attribute.
     * RULE 13: Tracks all intervals for cleanup on unload.
     */
    function initCountdown() {
        var containers = document.querySelectorAll('.countdown-timer-card[data-maturity]');
        
        log('Initializing countdown timers. Found:', containers.length);
        
        for (var i = 0; i < containers.length; i++) {
            var container = containers[i];
            
            if (container.getAttribute('data-initialized') === 'true') {
                log('Timer already initialized, skipping');
                continue;
            }
            
            container.setAttribute('data-initialized', 'true');
            log('Timer marked as initialized');
            
            if (updateCountdown(container)) {
                (function(cont) {
                    var intervalId = setInterval(function() {
                        if (!updateCountdown(cont)) {
                            clearInterval(intervalId);
                            log('Timer expired, interval cleared');
                        }
                    }, 1000);
                    
                    trackInterval(intervalId);
                })(container);
            } else {
                log('Timer already expired on init');
            }
        }
        
        log('Countdown initialization complete');
    }

    // ═══════════════════════════════════════════════════════════════════════
    // SMOOTH SCROLL WITH OFFSET (v1.5.0)
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Custom smooth scroll to element with offset and configurable duration
     * 
     * Uses requestAnimationFrame for smooth 60fps animation with easeInOutQuad easing.
     * Supports offset for sticky headers/breadcrumbs.
     * 
     * @param {HTMLElement} targetElement - Element to scroll to
     * @param {number} duration - Animation duration in milliseconds (default: SCROLL_DURATION)
     * @param {number} offset - Offset in pixels from top (default: SCROLL_OFFSET)
     * 
     * RULE 17: IE11 compatible (var, function, no arrow functions)
     * 
     * Easing function: easeInOutQuad
     * - Progress < 0.5: accelerates (ease in)
     * - Progress >= 0.5: decelerates (ease out)
     * - Creates smooth, natural scrolling motion
     */
    function smoothScrollTo(targetElement, duration, offset) {
        if (!targetElement) {
            log('WARNING: smoothScrollTo called with null element');
            return;
        }
        
        // Use defaults if not provided
        duration = duration || SCROLL_DURATION;
        offset = offset || SCROLL_OFFSET;
        
        var startPosition = window.pageYOffset || document.documentElement.scrollTop;
        var targetPosition = targetElement.getBoundingClientRect().top + startPosition - offset;
        var distance = targetPosition - startPosition;
        var startTime = null;
        
        log('Smooth scroll initiated. Distance: ' + distance + 'px, Duration: ' + duration + 'ms, Offset: ' + offset + 'px');
        
        /**
         * Animation frame callback
         * Called on each frame (~60fps) until duration elapsed
         */
        function animation(currentTime) {
            if (startTime === null) {
                startTime = currentTime;
            }
            
            var timeElapsed = currentTime - startTime;
            var progress = Math.min(timeElapsed / duration, 1);
            
            // easeInOutQuad easing function
            var ease;
            if (progress < 0.5) {
                // Ease in: accelerate
                ease = 2 * progress * progress;
            } else {
                // Ease out: decelerate
                ease = 1 - Math.pow(-2 * progress + 2, 2) / 2;
            }
            
            var newPosition = startPosition + distance * ease;
            window.scrollTo(0, newPosition);
            
            if (timeElapsed < duration) {
                requestAnimationFrame(animation);
            } else {
                log('Smooth scroll complete. Final position: ' + newPosition);
            }
        }
        
        requestAnimationFrame(animation);
    }

    // ═══════════════════════════════════════════════════════════════════════
    // NAVIGATION FOR BONDS CATALOG (v1.4.0)
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Initialize smooth scroll navigation for bonds catalog page
     * 
     * Finds all anchor links in .bonds-type-navigation and adds smooth scroll
     * behavior when clicking on them. Also handles hash in URL on page load.
     * 
     * RULE 14: Element checks before addEventListener
     * RULE 13: All listeners tracked for cleanup
     * RULE 17: IE11 compatible (var, function, no arrow functions)
     * RULE 11: CSP compatible (addEventListener, no inline scripts)
     */
    function initNavigation() {
        var navLinks = document.querySelectorAll('.bonds-type-navigation a[href^="#"]');
        
        if (navLinks.length === 0) {
            log('Navigation not found on page, skipping');
            return;
        }
        
        log('Initializing navigation. Found links:', navLinks.length);
        
        // Add click handlers to all navigation links
        navLinks.forEach(function(link) {
            // RULE 14: Element check
            if (!link) return;
            
            var clickHandler = function(e) {
                e.preventDefault();
                
                // Get target element ID (remove # from href)
                var targetId = link.getAttribute('href').substring(1);
                var targetElement = document.getElementById(targetId);
                
                if (targetElement) {
                    // Custom smooth scroll with offset (v1.5.0)
                    smoothScrollTo(targetElement, SCROLL_DURATION, SCROLL_OFFSET);
                    
                    // Update URL without page reload
                    if (history.pushState) {
                        history.pushState(null, null, '#' + targetId);
                    }
                    
                    log('Scrolled to:', targetId);
                } else {
                    log('WARNING: Target element not found:', targetId);
                }
            };
            
            // RULE 13: Track event listener for cleanup
            addTrackedEventListener(link, 'click', clickHandler);
        });
        
        // Handle hash on page load
        if (window.location.hash) {
            var targetId = window.location.hash.substring(1);
            var targetElement = document.getElementById(targetId);
            
            if (targetElement) {
                log('Hash detected on load, scrolling to:', targetId);
                
                // Small delay for proper scroll after page load
                setTimeout(function() {
                    smoothScrollTo(targetElement, SCROLL_DURATION, SCROLL_OFFSET);
                }, 100);
            }
        }
        
        log('Navigation initialization complete');
    }

    // ═══════════════════════════════════════════════════════════════════════
    // INITIALIZATION
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Initialize all features when DOM is ready
     * 
     * v1.4.0: Added navigation initialization
     * v1.3.0: Enhanced progress bar initialization with timeline support
     */
    function init() {
        log('Initialization started (v1.4.0 CATALOG-NAV)');
        
        // Initialize progress bars (includes timeline animation)
        initProgressBars();
        
        // Initialize countdown timers
        initCountdown();
        
        // Initialize navigation (bonds catalog page)
        initNavigation();
        
        log('Initialization complete');
    }

    // Initialize when DOM is ready - v7.2: Wait for CSS
    function initBondWhenReady() {
        // Wait for CSS to load
        if (!document.body.classList.contains('css-ready')) {
            setTimeout(initBondWhenReady, 50);
            return;
        }
        init();
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initBondWhenReady);
    } else {
        initBondWhenReady();
    }
    // [END SECTION: Bond Countdown + Navigation]

// Check if we're on emitents catalog page (MUST have .emitents-catalog-page wrapper)
if (document.querySelector('.emitents-catalog-page .alphabet-nav')) {
    
    console.log('[AlphabetNav] Initializing alphabet navigation');
    
    // Get all letter sections and links (scoped to emitents catalog)
    var letterSections = document.querySelectorAll('.emitents-catalog-page [data-letter]');
    var letterLinks = document.querySelectorAll('.emitents-catalog-page .letter-link');
    
    console.log('[AlphabetNav] Found ' + letterSections.length + ' sections and ' + letterLinks.length + ' links');
    
    /**
     * Highlight active letter based on scroll position
     * Active = section in upper half of viewport
     */
    function highlightActiveLetter() {
        var currentLetter = '';
        var viewportMiddle = window.innerHeight / 2;
        
        // Find which section is currently in view
        letterSections.forEach(function(section) {
            var rect = section.getBoundingClientRect();
            
            // Section is in upper half of viewport
            if (rect.top >= 0 && rect.top <= viewportMiddle) {
                currentLetter = section.getAttribute('data-letter');
            }
        });
        
        // Remove active class from all links
        letterLinks.forEach(function(link) {
            link.classList.remove('active');
        });
        
        // Add active class to current letter (scoped)
        if (currentLetter) {
            var activeLink = document.querySelector('.emitents-catalog-page .letter-link[data-letter="' + currentLetter + '"]');
            if (activeLink) {
                activeLink.classList.add('active');
            }
        }
    }
    
    /**
     * Throttled scroll handler
     * Prevents excessive function calls on scroll
     */
    var scrollTimeout;
    function onScroll() {
        if (scrollTimeout) {
            clearTimeout(scrollTimeout);
        }
        
        scrollTimeout = setTimeout(function() {
            highlightActiveLetter();
        }, 100); // 100ms throttle
    }
    
    // Attach scroll event listener
    window.addEventListener('scroll', onScroll);
    
    /**
     * Smooth scroll to letter section on click
     * Uses custom smoothScrollTo() with offset (same as bonds catalog)
     */
    letterLinks.forEach(function(link) {
        link.addEventListener('click', function(e) {
            var href = this.getAttribute('href');
            
            // Only handle anchor links
            if (href && href.charAt(0) === '#') {
                e.preventDefault();
                
                var targetSection = document.querySelector(href);
                if (targetSection) {
                    // Use custom smooth scroll with offset (v1.5.0)
                    smoothScrollTo(targetSection, SCROLL_DURATION, SCROLL_OFFSET);
                    
                    console.log('[AlphabetNav] Scrolled to:', href);
                } else {
                    console.log('[AlphabetNav] WARNING: Target not found:', href);
                }
            }
        });
    });
    
    /**
     * Initial highlight on page load
     */
    highlightActiveLetter();
    
    console.log('[AlphabetNav] Initialization complete');
    
    /**
     * Cleanup on page unload (memory leak prevention)
     */
    window.addEventListener('beforeunload', function() {
        window.removeEventListener('scroll', onScroll);
        console.log('[AlphabetNav] Cleanup complete');
    });
}

    // ═══════════════════════════════════════════════════════════════════════
    // INITIALIZE COPY BUTTONS
    // ═══════════════════════════════════════════════════════════════════════
    initCopyButtons();
    
    console.log('[DeepMax Features] ✅ Initialized');
})();