/**
 * DeepMax Debug Panel - Event Handlers
 * Version: 1.1.0 IE11 COMPATIBLE
 * Date: 2025-10-18
 * 
 * External JavaScript file for Debug Panel v4.4.1
 * CSP Compatible - no inline scripts
 * IE11 Compatible - var, function, polyfills
 * Memory Safe - cleanup on beforeunload
 * 
 * ПРАВИЛО 17: IE11 совместимость (var, function, for вместо const/let/arrow)
 * ПРАВИЛО 13: Memory leaks prevention (cleanup handlers)
 * ПРАВИЛО 11: CSP совместимость (external JS + addEventListener)
 * ПРАВИЛО 14: Проверка элементов перед addEventListener
 */

(function() {
    // Event handlers array for cleanup (ПРАВИЛО 13)
    var eventHandlers = [];
    
    /**
     * Add tracked event listener for cleanup
     * @param {Element} element - DOM element
     * @param {string} event - Event name
     * @param {Function} handler - Event handler
     */
    function addTrackedEventListener(element, event, handler) {
        if (element && typeof element.addEventListener === 'function') {
            element.addEventListener(event, handler);
            eventHandlers.push({
                element: element,
                event: event,
                handler: handler
            });
        }
    }
    
    /**
     * Cleanup all event listeners (ПРАВИЛО 13: Memory leaks prevention)
     */
    function cleanup() {
        for (var i = 0; i < eventHandlers.length; i++) {
            var item = eventHandlers[i];
            if (item.element && typeof item.element.removeEventListener === 'function') {
                item.element.removeEventListener(item.event, item.handler);
            }
        }
        eventHandlers = [];
        
        if (window.console && window.console.log) {
            window.console.log('[DeepMaxDebug] Cleanup completed');
        }
    }
    
    /**
     * Initialize Debug Panel
     */
    function initDebugPanel() {
        // ПРАВИЛО 14: Проверка элементов перед использованием
        var panel = document.getElementById('deepmax-debug-panel');
        var content = document.getElementById('debug-content');
        var toggleBtn = document.getElementById('debug-toggle');
        var closeBtn = document.getElementById('debug-close');
        var header = document.getElementById('debug-header');
        
        // Check if panel exists
        if (!panel) {
            if (window.console && window.console.warn) {
                window.console.warn('[DeepMaxDebug] Panel element not found');
            }
            return;
        }
        
        /**
         * Toggle content visibility
         */
        function toggleContent() {
            if (content && toggleBtn) {
                if (content.style.display === 'none' || content.style.display === '') {
                    content.style.display = 'block';
                    toggleBtn.innerHTML = '&#9650;'; // Up arrow
                } else {
                    content.style.display = 'none';
                    toggleBtn.innerHTML = '&#9660;'; // Down arrow
                }
            }
        }
        
        /**
         * Toggle button click handler
         */
        function onToggleClick(e) {
            if (e && typeof e.stopPropagation === 'function') {
                e.stopPropagation();
            }
            toggleContent();
        }
        
        /**
         * Header click handler
         */
        function onHeaderClick() {
            toggleContent();
        }
        
        /**
         * Close button click handler
         */
        function onCloseClick(e) {
            if (e && typeof e.stopPropagation === 'function') {
                e.stopPropagation();
            }
            if (panel) {
                panel.style.display = 'none';
            }
        }
        
        /**
         * Keyboard shortcut handler: Ctrl+Shift+D
         */
        function onKeyDown(e) {
            // Check for Ctrl+Shift+D (IE11 compatible)
            var isCtrlShiftD = e.ctrlKey && e.shiftKey && (e.key === 'D' || e.keyCode === 68);
            
            if (isCtrlShiftD) {
                if (e.preventDefault) {
                    e.preventDefault();
                }
                if (panel) {
                    panel.style.display = (panel.style.display === 'none') ? 'block' : 'none';
                }
            }
        }
        
        // ПРАВИЛО 14: Проверка существования элементов перед addEventListener
        if (toggleBtn) {
            addTrackedEventListener(toggleBtn, 'click', onToggleClick);
        } else {
            if (window.console && window.console.warn) {
                window.console.warn('[DeepMaxDebug] Toggle button not found');
            }
        }
        
        if (header) {
            addTrackedEventListener(header, 'click', onHeaderClick);
        }
        
        if (closeBtn) {
            addTrackedEventListener(closeBtn, 'click', onCloseClick);
        } else {
            if (window.console && window.console.warn) {
                window.console.warn('[DeepMaxDebug] Close button not found');
            }
        }
        
        // Keyboard shortcut
        addTrackedEventListener(document, 'keydown', onKeyDown);
        
        // ПРАВИЛО 13: Cleanup on beforeunload
        addTrackedEventListener(window, 'beforeunload', cleanup);
        
        if (window.console && window.console.log) {
            window.console.log('[DeepMaxDebug] Panel initialized successfully (v1.1.0 IE11 compatible)');
        }
    }
    
    // Wait for DOM to be fully loaded (IE11 compatible)
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initDebugPanel);
    } else {
        // DOM already loaded
        initDebugPanel();
    }
})();