/**
 * CSS Activator v1.0 - CSP-safe async CSS loading
 * DeepMax.ru
 * Date: 2025-11-10
 * Status: PRODUCTION READY
 * 
 * Активирует preload CSS без inline обработчиков (CSP-совместимо)
 * Добавляет класс .css-ready к body после загрузки всех CSS
 * 
 * ПРАВИЛО 17: IE11 совместимость (var, function, polyfills)
 * ПРАВИЛО 11: CSP совместимость (external JS + addEventListener)
 */

(function() {
    'use strict';
    
    /**
     * Активирует preload CSS ссылки
     */
    function activateCSS() {
        // Находим все preload CSS
        var preloadLinks = document.querySelectorAll('link[rel="preload"][as="style"]');
        var loadedCount = 0;
        var totalLinks = preloadLinks.length;
        
        if (totalLinks === 0) {
            // Нет preload CSS, сразу помечаем как готово
            document.body.classList.add('css-ready');
            if (window.console && window.console.log) {
                window.console.log('[CSS Activator] No preload CSS found, marking as ready');
            }
            return;
        }
        
        if (window.console && window.console.log) {
            window.console.log('[CSS Activator] Found ' + totalLinks + ' preload CSS links');
        }
        
        /**
         * Проверяет завершение загрузки всех CSS
         */
        function checkAllLoaded() {
            loadedCount++;
            
            if (window.console && window.console.log) {
                window.console.log('[CSS Activator] CSS loaded: ' + loadedCount + '/' + totalLinks);
            }
            
            if (loadedCount >= totalLinks) {
                // Все CSS загружены
                document.body.classList.add('css-ready');
                
                if (window.console && window.console.log) {
                    window.console.log('[CSS Activator] ✅ All CSS loaded, page ready!');
                }
            }
        }
        
        // Активируем каждую ссылку
        for (var i = 0; i < preloadLinks.length; i++) {
            var link = preloadLinks[i];
            
            // Меняем rel на stylesheet
            link.rel = 'stylesheet';
            
            // Добавляем событие загрузки
            if (link.sheet) {
                // Уже загружен
                checkAllLoaded();
            } else {
                // Ждём загрузки
                link.addEventListener('load', checkAllLoaded);
                link.addEventListener('error', function(e) {
                    if (window.console && window.console.error) {
                        window.console.error('[CSS Activator] Error loading CSS:', e.target.href);
                    }
                    checkAllLoaded(); // Всё равно считаем как "загруженный"
                });
            }
        }
    }
    
    /**
     * Инициализация
     */
    function init() {
        // Проверяем что body существует
        if (!document.body) {
            if (window.console && window.console.warn) {
                window.console.warn('[CSS Activator] Body not found, waiting...');
            }
            setTimeout(init, 10);
            return;
        }
        
        activateCSS();
    }
    
    // Запускаем как можно раньше
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        // DOM уже загружен
        init();
    }
})();
