<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ КАТАЛОГА АКЦИЙ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.1 FIX - ЗАГРУЗКА ДАННЫХ ИЗ БД
 * 
 * CHANGELOG v1.1 FIX (2025-10-22):
 *   🔧 FIX: Добавлен метод loadCatalogDataFromDB() для загрузки данных из БД
 *   🔧 FIX: buildEnhancedJsonLd() теперь загружает данные сам, не требует $page_data[1]
 *   ✅ Работает с $page_data = ['CATALOG_STOCKS', null]
 *   ✅ SQL запрос с prepared statement для безопасности
 *   ✅ Эффективная группировка: listing → sector → emitent → stocks
 *   ✅ Все 466 строк оригинального кода сохранены
 * 
 * НАЗНАЧЕНИЕ:
 *   - Генерация Schema.org разметки для страницы каталога акций
 *   - Полная иерархия: Листинг → Сектор → Эмитент → Акция
 *   - Все акции без лимитов (100+)
 *   - Оптимизированная генерация для PAGE cache
 * 
 * СТРУКТУРА JSON-LD:
 *   @graph [
 *     BreadcrumbList - навигационные крошки
 *     CollectionPage - страница каталога
 *       mainEntity: ItemList - уровни листинга
 *         ItemList - сектора экономики
 *           ItemList - эмитенты
 *             FinancialProduct - акции
 *   ]
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Размер JSON: ~50-70 KB
 *   - С gzip: ~10-15 KB
 *   - Генерация: 1 раз/месяц (PAGE cache)
 *   - Из кэша: 0.001 сек
 * 
 * ОСНОВАНО НА:
 *   - SchemaOrgBase v1.1
 *   - SchemaOrgEmitent v1.1 (структура)
 *   - SchemaOrgSector v1.2 (минимализм)
 * 
 * @version 1.1 FIX
 * @date 2025-10-22
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgStocksCatalog extends SchemaOrgBase {
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение JSON-LD разметки для каталога акций
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * v1.1 FIX: Теперь загружает данные из БД сам через loadCatalogDataFromDB()
     * 
     * @param array $page_data Данные страницы [0 => 'CATALOG_STOCKS', 1 => null]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка при ошибке
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (empty($page_data) || !is_array($page_data)) {
            error_log('SchemaOrgStocksCatalog: Invalid page_data format');
            return '';
        }
        
        // v1.1 FIX: Загружаем данные каталога из БД
        // Раньше: извлекали готовый объект из $page_data[1]
        // Теперь: загружаем сами через $this->db
        $catalog = $this->loadCatalogDataFromDB();
        
        // Проверка что данные загружены
        if (empty($catalog) || !is_object($catalog)) {
            error_log('SchemaOrgStocksCatalog: Failed to load catalog data from DB');
            return '';
        }
        
        // Валидация критичных полей
        if (!property_exists($catalog, 'data') || !is_array($catalog->data)) {
            error_log('SchemaOrgStocksCatalog: Missing or invalid catalog data after DB load');
            return '';
        }
        
        // Построение @graph
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        if (!empty($breadcrumbs) && is_array($breadcrumbs)) {
            $breadcrumbList = $this->buildBreadcrumbList($breadcrumbs);
            if (!empty($breadcrumbList)) {
                $graph[] = $breadcrumbList;
            }
        }
        
        // 2. CollectionPage с полной иерархией
        $collectionPage = $this->buildCatalogCollectionPage($catalog);
        
        if (!empty($collectionPage)) {
            $graph[] = $collectionPage;
        }
        
        // Если нет данных для вывода - возвращаем пустую строку
        if (empty($graph)) {
            error_log('SchemaOrgStocksCatalog: Empty graph, nothing to output');
            return '';
        }
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * НОВЫЙ МЕТОД v1.1 FIX - Загрузка данных каталога из БД
     * 
     * Загружает все акции (type 1) с группировкой по:
     * - Уровням листинга (LIST_SECTION_ID)
     * - Секторам экономики (ID_SECTOR_ECONOMIKI)
     * - Эмитентам (Emitent_Id)
     * 
     * @return object|null Объект каталога или null при ошибке
     */
    private function loadCatalogDataFromDB() {
        // Проверка подключения к БД
        if (!$this->db instanceof mysqli) {
            error_log('SchemaOrgStocksCatalog: Invalid database connection');
            return null;
        }
        
        // SQL запрос: получаем все акции обыкновенные (type 1) с JOIN эмитентов
        $sql = "SELECT 
                    i.Instrument_LIST_SECTION_ID,
                    i.Instrument_LIST_SECTION,
                    i.Instrument_TRADE_CODE,
                    i.Instrument_txt_short,
                    i.Instrument_TYPE,
                    e.Id as emitent_id,
                    e.EMITENT_SHORT_NAME,
                    e.EMITENT_URL,
                    e.INN,
                    e.ID_SECTOR_ECONOMIKI,
                    e.SECTOR_ECONOMIKI
                FROM Instrument i
                LEFT JOIN Emitent e ON i.Emitent_Id = e.Id
                WHERE i.Instrument_SUPERTYPE_ID = ?
                ORDER BY i.Instrument_LIST_SECTION_ID, e.ID_SECTOR_ECONOMIKI, e.EMITENT_SHORT_NAME";
        
        // Подготовка prepared statement
        $stmt = $this->db->prepare($sql);
        
        if (!$stmt) {
            error_log('SchemaOrgStocksCatalog: SQL prepare failed - ' . $this->db->error);
            return null;
        }
        
        // Привязка параметров: type 1 = Акции обыкновенные
        $supertype_id = 1;
        $stmt->bind_param('i', $supertype_id);
        
        // Выполнение запроса
        if (!$stmt->execute()) {
            error_log('SchemaOrgStocksCatalog: SQL execute failed - ' . $stmt->error);
            $stmt->close();
            return null;
        }
        
        // Получение результата
        $result = $stmt->get_result();
        
        if (!$result) {
            error_log('SchemaOrgStocksCatalog: Failed to get result');
            $stmt->close();
            return null;
        }
        
        // Инициализация структуры данных
        $data = [];
        $listing_levels = [];
        $total_stocks = 0;
        
        // Обработка результатов - группировка по 4 уровням
        while ($row = $result->fetch_assoc()) {
            // Валидация обязательных полей
            if (empty($row['Instrument_TRADE_CODE']) || empty($row['emitent_id'])) {
                continue;
            }
            
            $listing_id = (int)$row['Instrument_LIST_SECTION_ID'];
            $sector_id = (int)$row['ID_SECTOR_ECONOMIKI'];
            $emitent_id = (int)$row['emitent_id'];
            
            // Инициализация уровня листинга
            if (!isset($data[$listing_id])) {
                $data[$listing_id] = [
                    'sectors' => []
                ];
                
                // Заполнение справочника уровней листинга
                if (!isset($listing_levels[$listing_id])) {
                    $listing_levels[$listing_id] = [
                        'name' => !empty($row['Instrument_LIST_SECTION']) 
                            ? $row['Instrument_LIST_SECTION'] 
                            : 'Уровень ' . $listing_id
                    ];
                }
            }
            
            // Инициализация сектора
            if (!isset($data[$listing_id]['sectors'][$sector_id])) {
                $data[$listing_id]['sectors'][$sector_id] = [
                    'name' => !empty($row['SECTOR_ECONOMIKI']) 
                        ? $row['SECTOR_ECONOMIKI'] 
                        : 'Сектор ' . $sector_id,
                    'emitents' => []
                ];
            }
            
            // Инициализация эмитента
            if (!isset($data[$listing_id]['sectors'][$sector_id]['emitents'][$emitent_id])) {
                $data[$listing_id]['sectors'][$sector_id]['emitents'][$emitent_id] = [
                    'name' => !empty($row['EMITENT_SHORT_NAME']) 
                        ? $row['EMITENT_SHORT_NAME'] 
                        : 'Эмитент ' . $emitent_id,
                    'url' => !empty($row['EMITENT_URL']) ? $row['EMITENT_URL'] : '',
                    'inn' => !empty($row['INN']) ? $row['INN'] : '',
                    'stocks' => []
                ];
            }
            
            // Добавление акции
            $data[$listing_id]['sectors'][$sector_id]['emitents'][$emitent_id]['stocks'][] = [
                'code' => $row['Instrument_TRADE_CODE'],
                'name' => !empty($row['Instrument_txt_short']) ? $row['Instrument_txt_short'] : '',
                'type' => !empty($row['Instrument_TYPE']) ? $row['Instrument_TYPE'] : ''
            ];
            
            $total_stocks++;
        }
        
        // Закрытие statement
        $stmt->close();
        
        // Проверка что получены данные
        if (empty($data)) {
            error_log('SchemaOrgStocksCatalog: No stocks found in DB');
            return null;
        }
        
        // Логирование успешной загрузки
        error_log(sprintf(
            'SchemaOrgStocksCatalog: Loaded %d stocks from DB (listings: %d, sectors: %d)',
            $total_stocks,
            count($data),
            array_sum(array_map(function($listing) {
                return count($listing['sectors']);
            }, $data))
        ));
        
        // Формирование объекта каталога с той же структурой что ожидается
        return (object)[
            'data' => $data,
            'listing_levels' => $listing_levels,
            'total_stocks' => $total_stocks
        ];
    }
    
    /**
     * Построение CollectionPage с полной 5-уровневой иерархией
     * 
     * @param object $catalog Объект каталога с данными
     * @return array Schema.org CollectionPage объект
     */
    private function buildCatalogCollectionPage($catalog) {
        // Проверка объекта
        if (!is_object($catalog)) {
            return [];
        }
        
        $page = [
            '@type' => 'CollectionPage',
            '@id' => $this->baseUrl . '/stocks/#catalog',
            'name' => 'Каталог акций Московской биржи',
            'description' => 'Полный каталог акций российских эмитентов, структурированный по уровням листинга, секторам экономики и компаниям',
            'url' => $this->baseUrl . '/stocks/'
        ];
        
        // Добавление метрик каталога
        if (property_exists($catalog, 'total_stocks') && $this->isValidNumber($catalog->total_stocks)) {
            $page['numberOfItems'] = $this->safeInteger($catalog->total_stocks);
        }
        
        // mainEntity - ItemList уровней листинга (УРОВЕНЬ 1)
        $listingsItemList = $this->buildListingsItemList($catalog);
        
        if (!empty($listingsItemList)) {
            $page['mainEntity'] = $listingsItemList;
        }
        
        return $page;
    }
    
    /**
     * УРОВЕНЬ 1: Построение ItemList уровней листинга
     * 
     * @param object $catalog Объект каталога с данными
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildListingsItemList($catalog) {
        // Проверка наличия данных
        if (!property_exists($catalog, 'data') || !is_array($catalog->data) || empty($catalog->data)) {
            return [];
        }
        
        // Проверка справочника уровней листинга
        $listingLevels = property_exists($catalog, 'listing_levels') ? $catalog->listing_levels : [];
        
        if (!is_array($listingLevels) || empty($listingLevels)) {
            error_log('SchemaOrgStocksCatalog: Missing listing_levels in catalog');
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Уровни листинга',
            'description' => 'Акции структурированы по уровням листинга Московской биржи',
            'numberOfItems' => count($catalog->data),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор уровней листинга
        foreach ($catalog->data as $listingId => $listingData) {
            // Валидация структуры уровня
            if (!is_array($listingData) || !isset($listingData['sectors']) || !is_array($listingData['sectors'])) {
                continue;
            }
            
            // Получение информации об уровне листинга
            $listingInfo = isset($listingLevels[$listingId]) ? $listingLevels[$listingId] : null;
            
            if (!is_array($listingInfo) || !isset($listingInfo['name'])) {
                continue;
            }
            
            // УРОВЕНЬ 2: ItemList секторов для этого уровня листинга
            $sectorsItemList = $this->buildSectorsItemList($listingData['sectors'], $catalog);
            
            if (empty($sectorsItemList)) {
                continue;
            }
            
            // Элемент списка уровня листинга
            $listItem = [
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $this->safeString($listingInfo['name']),
                'item' => $sectorsItemList
            ];
            
            $itemList['itemListElement'][] = $listItem;
            $position++;
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * УРОВЕНЬ 2: Построение ItemList секторов экономики
     * 
     * @param array $sectors Массив секторов
     * @param object $catalog Объект каталога (для sector_icons)
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildSectorsItemList($sectors, $catalog) {
        // Валидация входных данных
        if (!is_array($sectors) || empty($sectors)) {
            return [];
        }
        
        // Получение иконок секторов (опционально)
        $sectorIcons = property_exists($catalog, 'sector_icons') ? $catalog->sector_icons : [];
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Секторы экономики',
            'numberOfItems' => count($sectors),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор секторов
        foreach ($sectors as $sectorId => $sectorData) {
            // Валидация структуры сектора
            if (!is_array($sectorData) || !isset($sectorData['name']) || !isset($sectorData['emitents'])) {
                continue;
            }
            
            if (!is_array($sectorData['emitents']) || empty($sectorData['emitents'])) {
                continue;
            }
            
            // УРОВЕНЬ 3: ItemList эмитентов для этого сектора
            $emitentsItemList = $this->buildEmitentsItemList($sectorData['emitents'], $sectorId);
            
            if (empty($emitentsItemList)) {
                continue;
            }
            
            // Элемент списка сектора
            $listItem = [
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $this->safeString($sectorData['name']),
                'item' => $emitentsItemList
            ];
            
            $itemList['itemListElement'][] = $listItem;
            $position++;
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * УРОВЕНЬ 3: Построение ItemList эмитентов
     * 
     * @param array $emitents Массив эмитентов
     * @param int $sectorId ID сектора (для URL)
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildEmitentsItemList($emitents, $sectorId) {
        // Валидация входных данных
        if (!is_array($emitents) || empty($emitents)) {
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Эмитенты',
            'numberOfItems' => count($emitents),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор эмитентов
        foreach ($emitents as $emitentId => $emitentData) {
            // Валидация структуры эмитента
            if (!is_array($emitentData) || !isset($emitentData['name']) || !isset($emitentData['stocks'])) {
                continue;
            }
            
            if (!is_array($emitentData['stocks']) || empty($emitentData['stocks'])) {
                continue;
            }
            
            // УРОВЕНЬ 4: ItemList акций для этого эмитента
            $stocksItemList = $this->buildStocksItemList($emitentData, $emitentId);
            
            if (empty($stocksItemList)) {
                continue;
            }
            
            // Элемент списка эмитента
            $listItem = [
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $this->safeString($emitentData['name']),
                'item' => $stocksItemList
            ];
            
            // Добавление URL эмитента (если есть)
            if (isset($emitentData['url']) && $this->isValidString($emitentData['url'])) {
                $listItem['url'] = $this->baseUrl . '/' . $this->safeString($emitentData['url']) . '/';
            }
            
            $itemList['itemListElement'][] = $listItem;
            $position++;
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * УРОВЕНЬ 4: Построение ItemList акций (с FinancialProduct)
     * 
     * @param array $emitentData Данные эмитента со списком акций
     * @param int $emitentId ID эмитента
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildStocksItemList($emitentData, $emitentId) {
        // Валидация входных данных
        if (!is_array($emitentData) || !isset($emitentData['stocks']) || !is_array($emitentData['stocks'])) {
            return [];
        }
        
        $stocks = $emitentData['stocks'];
        
        if (empty($stocks)) {
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Акции',
            'numberOfItems' => count($stocks),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор ВСЕХ акций (без лимитов!)
        foreach ($stocks as $stockData) {
            // Валидация структуры акции
            if (!is_array($stockData) || !isset($stockData['code'])) {
                continue;
            }
            
            // УРОВЕНЬ 5: FinancialProduct для каждой акции
            $financialProduct = $this->buildStockFinancialProduct($stockData, $emitentData);
            
            if (empty($financialProduct)) {
                continue;
            }
            
            // Элемент списка акции
            $listItem = [
                '@type' => 'ListItem',
                'position' => $position,
                'item' => $financialProduct
            ];
            
            $itemList['itemListElement'][] = $listItem;
            $position++;
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * УРОВЕНЬ 5: Построение FinancialProduct для акции
     * 
     * @param array $stockData Данные акции
     * @param array $emitentData Данные эмитента
     * @return array Schema.org FinancialProduct объект или пустой массив
     */
    private function buildStockFinancialProduct($stockData, $emitentData) {
        // Валидация входных данных
        if (!is_array($stockData) || !isset($stockData['code']) || !$this->isValidString($stockData['code'])) {
            return [];
        }
        
        $code = $this->safeString($stockData['code']);
        
        $product = [
            '@type' => 'FinancialProduct',
            'name' => $code
        ];
        
        // Добавление полного названия акции (если есть)
        if (isset($stockData['name']) && $this->isValidString($stockData['name'])) {
            $product['description'] = $this->safeString($stockData['name']);
        }
        
        // Добавление типа инструмента (если есть)
        if (isset($stockData['type']) && $this->isValidString($stockData['type'])) {
            $product['category'] = $this->safeString($stockData['type']);
        }
        
        // URL акции
        $product['url'] = $this->baseUrl . '/stocks/' . $code . '/';
        
        // Identifier - торговый код
        $product['identifier'] = $code;
        
        // Информация об эмитенте (issuer)
        $issuer = [
            '@type' => 'Organization'
        ];
        
        if (isset($emitentData['name']) && $this->isValidString($emitentData['name'])) {
            $issuer['name'] = $this->safeString($emitentData['name']);
        }
        
        if (isset($emitentData['inn']) && $this->isValidString($emitentData['inn'])) {
            $issuer['taxID'] = $this->safeString($emitentData['inn']);
        }
        
        if (isset($emitentData['url']) && $this->isValidString($emitentData['url'])) {
            $issuer['url'] = $this->baseUrl . '/' . $this->safeString($emitentData['url']) . '/';
        }
        
        // Добавляем issuer только если есть хотя бы имя
        if (isset($issuer['name'])) {
            $product['issuer'] = $issuer;
        }
        
        return $product;
    }
}