<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ КАТАЛОГА СЕКТОРОВ ЭКОНОМИКИ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.0 - INITIAL RELEASE
 * 
 * CHANGELOG v1.0 (2025-11-02):
 *   ✨ NEW: Класс для генерации Schema.org разметки каталога секторов
 *   ✨ NEW: Метод loadCatalogDataFromDB() для загрузки данных из БД
 *   ✨ NEW: Простая структура CollectionPage с ItemList секторов
 *   ✅ Подсчёт количества эмитентов в каждом секторе
 *   ✅ SQL запрос с GROUP BY и LEFT JOIN
 *   ✅ Prepared statement для безопасности
 *   ✅ Полная валидация данных
 * 
 * НАЗНАЧЕНИЕ:
 *   - Генерация Schema.org разметки для страницы /sectors/
 *   - CollectionPage с полным списком секторов экономики
 *   - Каждый сектор: название, URL, количество эмитентов
 *   - Оптимизировано для PAGE cache (генерация 1 раз/месяц)
 * 
 * СТРУКТУРА JSON-LD:
 *   @graph [
 *     BreadcrumbList - навигационные крошки
 *     CollectionPage - страница каталога
 *       name: "Каталог секторов экономики"
 *       mainEntity: ItemList - список секторов
 *         ListItem - каждый сектор с названием, URL, описанием
 *   ]
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Размер JSON: ~5-10 KB (15 секторов)
 *   - С gzip: ~2-3 KB
 *   - Генерация: 1 раз/месяц (PAGE cache)
 *   - Из кэша: 0.001 сек
 * 
 * SQL ЗАПРОС:
 *   SELECT s.*, COUNT(e.Id) as emitent_count
 *   FROM SECTOR_ECONOMIKI s
 *   LEFT JOIN Emitent e ON s.Id = e.ID_SECTOR_ECONOMIKI
 *   GROUP BY s.Id
 *   ORDER BY s.SECTOR_NAME
 * 
 * ОСНОВАНО НА:
 *   - SchemaOrgBase v1.1
 *   - SchemaOrgStocksCatalog v1.1 FIX (структура loadCatalogDataFromDB)
 *   - SchemaOrgSector v1.2 (минимализм CollectionPage)
 * 
 * @version 1.0
 * @date 2025-11-02
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgSectorsCatalog extends SchemaOrgBase {
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение JSON-LD разметки для каталога секторов
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'MENU', 1 => объект MENU]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка при ошибке
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (empty($page_data) || !is_array($page_data)) {
            error_log('SchemaOrgSectorsCatalog: Invalid page_data format');
            return '';
        }
        
        // Загружаем данные каталога из БД
        $catalog = $this->loadCatalogDataFromDB();
        
        // Проверка что данные загружены
        if (empty($catalog) || !is_object($catalog)) {
            error_log('SchemaOrgSectorsCatalog: Failed to load catalog data from DB');
            return '';
        }
        
        // Валидация критичных полей
        if (!property_exists($catalog, 'sectors') || !is_array($catalog->sectors)) {
            error_log('SchemaOrgSectorsCatalog: Missing or invalid sectors data after DB load');
            return '';
        }
        
        // Построение @graph
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        if (!empty($breadcrumbs) && is_array($breadcrumbs)) {
            $breadcrumbList = $this->buildBreadcrumbList($breadcrumbs);
            if (!empty($breadcrumbList)) {
                $graph[] = $breadcrumbList;
            }
        }
        
        // 2. CollectionPage со списком секторов
        $collectionPage = $this->buildCatalogCollectionPage($catalog);
        
        if (!empty($collectionPage)) {
            $graph[] = $collectionPage;
        }
        
        // Если нет данных для вывода - возвращаем пустую строку
        if (empty($graph)) {
            error_log('SchemaOrgSectorsCatalog: Empty graph, nothing to output');
            return '';
        }
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * МЕТОД ЗАГРУЗКИ ДАННЫХ - Загрузка каталога секторов из БД
     * 
     * Загружает все сектора экономики с подсчётом количества эмитентов
     * в каждом секторе через LEFT JOIN и GROUP BY
     * 
     * @return object|null Объект каталога или null при ошибке
     */
    private function loadCatalogDataFromDB() {
        // Проверка подключения к БД
        if (!$this->db instanceof mysqli) {
            error_log('SchemaOrgSectorsCatalog: Invalid database connection');
            return null;
        }
        
        // SQL запрос: получаем все сектора с подсчётом эмитентов
        $sql = "SELECT 
                    s.Id,
                    s.SECTOR_NAME,
                    s.SECTOR_ECONOMIKI_URL,
                    s.SECTOR_ECONOMIKI_TEXT,
                    COUNT(e.Id) as emitent_count
                FROM SECTOR_ECONOMIKI s
                LEFT JOIN Emitent e ON s.Id = e.ID_SECTOR_ECONOMIKI
                GROUP BY s.Id, s.SECTOR_NAME, s.SECTOR_ECONOMIKI_URL, s.SECTOR_ECONOMIKI_TEXT
                ORDER BY s.SECTOR_NAME";
        
        // Выполнение запроса (без prepared statement т.к. нет параметров)
        $result = $this->db->query($sql);
        
        if (!$result) {
            error_log('SchemaOrgSectorsCatalog: SQL query failed - ' . $this->db->error);
            return null;
        }
        
        // Инициализация структуры данных
        $sectors = [];
        $total_sectors = 0;
        $total_emitents = 0;
        
        // Обработка результатов
        while ($row = $result->fetch_assoc()) {
            // Валидация обязательных полей
            if (empty($row['Id']) || empty($row['SECTOR_NAME'])) {
                continue;
            }
            
            $sector_id = (int)$row['Id'];
            $emitent_count = isset($row['emitent_count']) ? (int)$row['emitent_count'] : 0;
            
            // Формирование данных сектора
            $sectors[] = [
                'id' => $sector_id,
                'name' => trim($row['SECTOR_NAME']),
                'url' => !empty($row['SECTOR_ECONOMIKI_URL']) ? trim($row['SECTOR_ECONOMIKI_URL']) : null,
                'description' => !empty($row['SECTOR_ECONOMIKI_TEXT']) ? trim($row['SECTOR_ECONOMIKI_TEXT']) : null,
                'emitent_count' => $emitent_count
            ];
            
            $total_sectors++;
            $total_emitents += $emitent_count;
        }
        
        // Освобождение результата
        $result->free();
        
        // Проверка что есть данные
        if (empty($sectors)) {
            error_log('SchemaOrgSectorsCatalog: No sectors found in database');
            return null;
        }
        
        // Формирование объекта каталога
        $catalog = new stdClass();
        $catalog->sectors = $sectors;
        $catalog->total_sectors = $total_sectors;
        $catalog->total_emitents = $total_emitents;
        
        return $catalog;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ COLLECTIONPAGE - Построение Schema.org CollectionPage
     * 
     * Создаёт CollectionPage с mainEntity содержащим ItemList всех секторов
     * 
     * @param object $catalog Объект каталога с данными секторов
     * @return array Schema.org CollectionPage объект или пустой массив
     */
    private function buildCatalogCollectionPage($catalog) {
        // Валидация входных данных
        if (empty($catalog) || !is_object($catalog)) {
            return [];
        }
        
        if (!property_exists($catalog, 'sectors') || !is_array($catalog->sectors)) {
            return [];
        }
        
        if (empty($catalog->sectors)) {
            return [];
        }
        
        // Базовая структура CollectionPage
        $page = [
            '@type' => 'CollectionPage',
            'name' => 'Каталог секторов экономики',
            'description' => 'Полный список секторов экономики российских эмитентов',
            'url' => $this->baseUrl . '/sectors/'
        ];
        
        // Добавление статистики (если доступна)
        if (property_exists($catalog, 'total_sectors') && $catalog->total_sectors > 0) {
            $page['description'] .= ' (' . $catalog->total_sectors . ' ' . $this->pluralizeSectors($catalog->total_sectors) . ')';
        }
        
        // Построение mainEntity: ItemList секторов
        $itemList = $this->buildSectorsItemList($catalog->sectors);
        
        if (!empty($itemList)) {
            $page['mainEntity'] = $itemList;
        }
        
        return $page;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ ITEMLIST - Построение ItemList секторов
     * 
     * @param array $sectors Массив секторов из БД
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildSectorsItemList($sectors) {
        // Валидация входных данных
        if (!is_array($sectors) || empty($sectors)) {
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Сектора экономики',
            'numberOfItems' => count($sectors),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор всех секторов
        foreach ($sectors as $sectorData) {
            // Валидация структуры сектора
            if (!is_array($sectorData) || !isset($sectorData['name'])) {
                continue;
            }
            
            // Построение ListItem для сектора
            $listItem = $this->buildSectorListItem($sectorData, $position);
            
            if (!empty($listItem)) {
                $itemList['itemListElement'][] = $listItem;
                $position++;
            }
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ LISTITEM - Построение ListItem для одного сектора
     * 
     * @param array $sectorData Данные сектора
     * @param int $position Позиция в списке
     * @return array Schema.org ListItem объект или пустой массив
     */
    private function buildSectorListItem($sectorData, $position) {
        // Валидация входных данных
        if (!is_array($sectorData) || !isset($sectorData['name'])) {
            return [];
        }
        
        $listItem = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => $this->safeString($sectorData['name'])
        ];
        
        // Добавление URL сектора (если есть)
        if (isset($sectorData['url']) && !empty($sectorData['url'])) {
            $listItem['url'] = $this->baseUrl . '/' . $this->safeString($sectorData['url']) . '/';
        }
        
        // Построение описания с количеством эмитентов
        $description = '';
        
        if (isset($sectorData['emitent_count']) && $sectorData['emitent_count'] > 0) {
            $count = (int)$sectorData['emitent_count'];
            $description = $count . ' ' . $this->pluralizeEmitents($count) . ' в секторе';
        } else {
            $description = 'Сектор экономики';
        }
        
        // Добавление текстового описания (если есть и не слишком длинное)
        if (isset($sectorData['description']) && !empty($sectorData['description'])) {
            $desc = trim($sectorData['description']);
            if (mb_strlen($desc, 'UTF-8') <= 200) {
                $description = $this->safeString($desc);
            }
        }
        
        $listItem['description'] = $description;
        
        return $listItem;
    }
    
    /**
     * ВСПОМОГАТЕЛЬНЫЙ МЕТОД - Плюрализация слова "сектор"
     * 
     * @param int $count Количество секторов
     * @return string Правильная форма слова
     */
    private function pluralizeSectors($count) {
        $count = abs($count) % 100;
        $leastSignificant = $count % 10;
        
        if ($count >= 11 && $count <= 19) {
            return 'секторов';
        }
        
        if ($leastSignificant == 1) {
            return 'сектор';
        }
        
        if ($leastSignificant >= 2 && $leastSignificant <= 4) {
            return 'сектора';
        }
        
        return 'секторов';
    }
    
    /**
     * ВСПОМОГАТЕЛЬНЫЙ МЕТОД - Плюрализация слова "эмитент"
     * 
     * @param int $count Количество эмитентов
     * @return string Правильная форма слова
     */
    private function pluralizeEmitents($count) {
        $count = abs($count) % 100;
        $leastSignificant = $count % 10;
        
        if ($count >= 11 && $count <= 19) {
            return 'эмитентов';
        }
        
        if ($leastSignificant == 1) {
            return 'эмитент';
        }
        
        if ($leastSignificant >= 2 && $leastSignificant <= 4) {
            return 'эмитента';
        }
        
        return 'эмитентов';
    }
}