<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ ДОЛЕВЫХ ИНСТРУМЕНТОВ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.3 - IMPROVEMENTS: Улучшения логики полей
 * 
 * CHANGELOG v1.3 (2025-10-21):
 *   🔧 FIX: circulatingVolume добавляется только если ≠ issueVolume
 *   🔧 FIX: alternateName требует минимум 2 разных имени
 *   ✨ NEW: identifier как массив PropertyValue для международных ID
 *   📝 DOCS: Обновлены комментарии с версией v1.3
 * 
 * CHANGELOG v1.2 (2025-10-14):
 *   🔧 FIX: Исправлено имя поля DISCLOSURE_PART_smart-lab (было smartlab)
 *   ✨ NEW: Добавлена проверка типа инструмента в buildEnhancedJsonLd()
 *   📝 DOCS: Обновлены комментарии о поддерживаемых типах (1, 2, 4)
 * 
 * CHANGELOG v1.1 (2025-10-12):
 *   - ИЗМЕНЕНО: "class" → "tradingClass" (соответствие Schema.org)
 *   - ИЗМЕНЕНО: "торгуется" → "Active" (международная совместимость)
 *   - ПРОВЕРЕНО: ucfirst() работает корректно в строке ~471
 * 
 * НАЗНАЧЕНИЕ:
 *   - Schema.org разметка для страниц акций (type 1)
 *   - Schema.org разметка для депозитарных расписок (type 2)
 *   - Schema.org разметка для инвестиционных паёв (type 4)
 *   - Построение FinancialProduct объекта с полными данными
 *   - Интеграция с BreadcrumbList
 * 
 * ВАЖНО: Этот класс работает ТОЛЬКО с equity инструментами (types 1, 2, 4)
 *        Для debt инструментов (types 3, 5, 6) используется SchemaOrgDebt
 * 
 * ОСНОВАНО НА:
 *   BreadcrumbBuilder v4.0 методы:
 *   - getStockFinancialProductData() → collectEquityData()
 *   - buildFinancialProductObject() → buildFinancialProduct()
 * 
 * @version 1.3
 * @date 2025-10-21
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgEquity extends SchemaOrgBase {
    
    // Instrument type constants
    const INSTRUMENT_TYPE_STOCKS = 1;
    const INSTRUMENT_TYPE_RECEIPTS = 2;
    const INSTRUMENT_TYPE_FUNDS = 4;
    
    // Cache constants
    const CACHE_NOT_FOUND = '__NOT_FOUND__';
    const CACHE_TTL = 3600;
    
    // Static cache
    private static $sectorCache = [];
    
    // Instrument types mapping (соответствует BreadcrumbBuilder v5.0)
    private static $instrumentTypes = [
        self::INSTRUMENT_TYPE_STOCKS => [
            'name' => 'Акции',
            'slug' => 'stocks'
        ],
        self::INSTRUMENT_TYPE_RECEIPTS => [
            'name' => 'Депозитарные расписки',
            'slug' => 'receipts'
        ],
        self::INSTRUMENT_TYPE_FUNDS => [
            'name' => 'Инвестиционные паи',
            'slug' => 'funds'
        ]
    ];
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение расширенной JSON-LD разметки для долевых инструментов
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'STOCK', 1 => instrument, 2 => security, 3 => emitent]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (!is_array($page_data) || count($page_data) < 4) {
            return '';
        }
        
        // Проверка типа страницы
        if (!isset($page_data[0]) || $page_data[0] !== 'STOCK') {
            return '';
        }
        
        // v1.2: НОВОЕ - Проверка что это equity инструмент (types 1, 2, 4)
        if (isset($page_data[1]) && is_object($page_data[1])) {
            $supertypeId = (int)$this->getObjectProperty($page_data[1], 'Instrument_SUPERTYPE_ID', 0);
            $validTypes = [
                self::INSTRUMENT_TYPE_STOCKS,    // 1 - Акции
                self::INSTRUMENT_TYPE_RECEIPTS,  // 2 - Расписки
                self::INSTRUMENT_TYPE_FUNDS      // 4 - Паи
            ];
            
            if (!in_array($supertypeId, $validTypes, true)) {
                error_log("SchemaOrgEquity: Invalid instrument type {$supertypeId} for equity class. Expected 1, 2, or 4.");
                return '';
            }
        }
        
        // Сбор данных для FinancialProduct
        $data = $this->collectEquityData($page_data);
        
        if ($data === null) {
            return '';
        }
        
        // Построение @graph с BreadcrumbList и FinancialProduct
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        $graph[] = $this->buildBreadcrumbList($breadcrumbs);
        
        // 2. FinancialProduct с полными данными
        $graph[] = $this->buildFinancialProduct($data);
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * Сбор и подготовка всех данных для FinancialProduct
     * Основано на BreadcrumbBuilder v4.0::getStockFinancialProductData()
     * 
     * @param array $page_data Данные страницы
     * @return array|null Массив данных или null при ошибке
     */
    private function collectEquityData($page_data) {
        try {
            // Проверка наличия всех необходимых элементов в page_data
            if (!isset($page_data[1]) || !isset($page_data[2]) || !isset($page_data[3])) {
                error_log('SchemaOrgEquity: Missing required page_data elements (need indexes 1, 2, 3)');
                return null;
            }
            
            $instrument = $page_data[1];
            $security = $page_data[2];
            $emitent = $page_data[3];
            
            // Проверка что все объекты валидны
            if (!is_object($instrument) || !is_object($security) || !is_object($emitent)) {
                error_log('SchemaOrgEquity: Invalid objects in page_data');
                return null;
            }
            
            // Получение данных сектора если есть
            $sector = null;
            $sectorId = $this->getObjectProperty($emitent, 'ID_SECTOR_ECONOMIKI', 0);
            if ($sectorId > 0) {
                $sector = $this->getSectorData($sectorId);
            }
            
            // Формирование массива данных с безопасным чтением свойств
            $data = [
                'instrument' => [
                    'trade_code' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_TRADE_CODE', '')),
                    'isin' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_ISIN', '')),
                    'registry_number' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_REGISTRY_NUMBER', '')),
                    'type' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_TYPE', '')),
                    'supertype' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_SUPERTYPE', '')),
                    'supertype_id' => (int)$this->getObjectProperty($instrument, 'Instrument_SUPERTYPE_ID', 0),
                    'list_section' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_LIST_SECTION', '')),
                    'free_float' => $this->safeDecimal($this->getObjectProperty($instrument, 'Instrument_MoexFreeFloat', null)),
                    'restricted_weight' => $this->safeDecimal($this->getObjectProperty($instrument, 'Instrument_MoexOgranichVes', null))
                ],
                'security' => [
                    'sec_name' => $this->safeString($this->getObjectProperty($security, 'SecName', '')),
                    'short_name' => $this->safeString($this->getObjectProperty($security, 'ShortName', '')),
                    'sec_code' => $this->safeString($this->getObjectProperty($security, 'SecCode', '')),
                    'isin' => $this->safeString($this->getObjectProperty($security, 'ISIN', '')),
                    'class_name' => $this->safeString($this->getObjectProperty($security, 'ClassName', '')),
                    'class_code' => $this->safeString($this->getObjectProperty($security, 'ClassCode', '')),
                    'face_value' => $this->safeDecimal($this->getObjectProperty($security, 'FaceValue', null)),
                    'face_unit' => $this->safeString($this->getObjectProperty($security, 'FaceUnit', '')),
                    'currency_id' => $this->safeString($this->getObjectProperty($security, 'CurrencyId', '')),
                    'lot_size' => $this->safeInteger($this->getObjectProperty($security, 'LotSize', null)),
                    'issue_size' => $this->safeInteger($this->getObjectProperty($security, 'IssueSize', null)),
                    'issue_size_placed' => $this->safeInteger($this->getObjectProperty($security, 'IssueSizePlaced', null)),
                    'status' => $this->safeString($this->getObjectProperty($security, 'SecStatus', '')),
                    'min_price_step' => $this->safeDecimal($this->getObjectProperty($security, 'MinPriceStep', null)),
                    'scale' => $this->safeInteger($this->getObjectProperty($security, 'Scale', null)),
                    'quote_basis' => $this->safeString($this->getObjectProperty($security, 'QuoteBasis', '')),
                    'settle_code' => $this->safeString($this->getObjectProperty($security, 'SettleCode', '')),
                    'settle_date' => $this->safeString($this->getObjectProperty($security, 'SettleDate', '')),
                    'timestamp' => $this->safeString($this->getObjectProperty($security, 'Timestamp', '')),
                    'cfi' => $this->safeString($this->getObjectProperty($security, 'CFI', '')),
                    'bloomberg_id' => $this->safeString($this->getObjectProperty($security, 'BloombergID', '')),
                    'sedol' => $this->safeString($this->getObjectProperty($security, 'SEDOL', '')),
                    'ric' => $this->safeString($this->getObjectProperty($security, 'RIC', '')),
                    'cusip' => $this->safeString($this->getObjectProperty($security, 'CUSIP', ''))
                ],
                'emitent' => [
                    'id' => (int)$this->getObjectProperty($emitent, 'Id', 0),
                    'short_name' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_SHORT_NAME', '')),
                    'full_name' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_FULL_NAME', '')),
                    'inn' => $this->safeString($this->getObjectProperty($emitent, 'INN', '')),
                    'url_slug' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_URL', '')),
                    'og_image' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_og_image', '')),
                    'txt_short' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_TXT_SHORT', '')),
                    'disclosure_page' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_PAGE', '')),
                    'disclosure_rf' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_RF_INFO_PAGE', '')),
                    'wiki' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_wiki', '')),
                    'smart_lab' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_smart-lab', '')),  // v1.2: ИСПРАВЛЕНО
                    'dohod' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_dohod', '')),
                    'moex' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_moex', '')),
                    'founding_date' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_FOUNDING_DATE', '')),
                    'lei' => $this->safeString($this->getObjectProperty($emitent, 'LEI', ''))
                ],
                'sector' => $sector,
                'urls' => [
                    'instrument' => $this->buildInstrumentUrl($instrument),
                    'emitent' => $this->buildEmitentUrl($emitent),
                    'sector' => $sector ? $this->baseUrl . '/' . $this->safeString(isset($sector['url']) ? $sector['url'] : '') . '/' : null
                ]
            ];
            
            return $data;
            
        } catch (Exception $e) {
            error_log('SchemaOrgEquity::collectEquityData error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Построение URL инструмента с проверкой на пустоту
     * 
     * @param object $instrument Объект инструмента
     * @return string URL инструмента
     */
    private function buildInstrumentUrl($instrument) {
        $supertypeId = (int)$this->getObjectProperty($instrument, 'Instrument_SUPERTYPE_ID', 0);
        $tradeCode = $this->safeString($this->getObjectProperty($instrument, 'Instrument_TRADE_CODE', ''));
        
        // Если trade_code пустой - возвращаем базовый URL без двойного слеша
        if (empty($tradeCode)) {
            $slug = $this->getInstrumentSlug($supertypeId);
            return $this->baseUrl . '/' . $slug . '/';
        }
        
        $slug = $this->getInstrumentSlug($supertypeId);
        return $this->baseUrl . '/' . $slug . '/' . $tradeCode . '/';
    }
    
    /**
     * Построение URL эмитента с проверкой на пустоту
     * 
     * @param object $emitent Объект эмитента
     * @return string URL эмитента
     */
    private function buildEmitentUrl($emitent) {
        $emitentUrl = $this->safeString($this->getObjectProperty($emitent, 'EMITENT_URL', ''));
        
        // Если emitent_url пустой - возвращаем только baseUrl
        if (empty($emitentUrl)) {
            return $this->baseUrl . '/';
        }
        
        return $this->baseUrl . '/' . $emitentUrl . '/';
    }
    
    /**
     * Получение slug инструмента по его типу
     * ВАЖНО: Работает только для equity типов (1, 2, 4)
     * Для debt типов (3, 5, 6) вернёт fallback со warning
     * 
     * @param int $supertypeId ID типа инструмента
     * @return string Slug для URL (stocks/receipts/funds)
     */
    private function getInstrumentSlug($supertypeId) {
        if (!isset(self::$instrumentTypes[$supertypeId])) {
            // ВНИМАНИЕ: Если попал сюда debt тип (3,5,6) или неизвестный тип
            // это значит SchemaOrgEquity вызван для неправильного типа инструмента
            error_log("SchemaOrgEquity: Invalid supertypeId {$supertypeId} for equity instruments. Expected 1, 2, or 4. Using 'stocks' fallback.");
            return 'stocks';
        }
        
        return self::$instrumentTypes[$supertypeId]['slug'];
    }
    
    /**
     * Безопасное получение свойства объекта
     * Возвращает значение свойства или default если свойство не существует
     * 
     * @param object $object Объект
     * @param string $property Имя свойства
     * @param mixed $default Значение по умолчанию
     * @return mixed Значение свойства или default
     */
    private function getObjectProperty($object, $property, $default = null) {
        if (!is_object($object)) {
            return $default;
        }
        
        if (!property_exists($object, $property)) {
            return $default;
        }
        
        return $object->$property ?? $default;
    }
    
    /**
     * Получение данных сектора из БД с кэшированием
     * 
     * @param int $sector_id ID сектора
     * @return array|null Данные сектора или null
     */
    private function getSectorData($sector_id) {
        $cacheKey = "sector_{$sector_id}";
        
        // Проверка кэша
        if (isset(self::$sectorCache[$cacheKey])) {
            $cached = self::$sectorCache[$cacheKey];
            
            if ($cached === self::CACHE_NOT_FOUND) {
                return null;
            }
            
            // Проверка что кэш валиден и не истёк
            if (isset($cached['expires']) && isset($cached['data']) && $cached['expires'] > time()) {
                return $cached['data'];
            }
        }
        
        // Запрос к БД
        try {
            $stmt = $this->db->prepare('SELECT SECTOR_NAME, SECTOR_ECONOMIKI_URL FROM SECTOR_ECONOMIKI WHERE Id = ? LIMIT 1');
            
            if (!$stmt) {
                return null;
            }
            
            $stmt->bind_param('i', $sector_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            
            if ($row) {
                $sectorData = [
                    'name' => $this->safeString($row['SECTOR_NAME']),
                    'url' => $this->safeString($row['SECTOR_ECONOMIKI_URL'])
                ];
                
                // Сохранение в кэш
                self::$sectorCache[$cacheKey] = [
                    'data' => $sectorData,
                    'expires' => time() + self::CACHE_TTL
                ];
                
                return $sectorData;
            } else {
                // Сохранение NOT_FOUND в кэш
                self::$sectorCache[$cacheKey] = self::CACHE_NOT_FOUND;
                return null;
            }
            
        } catch (Exception $e) {
            error_log('SchemaOrgEquity::getSectorData error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Построение FinancialProduct объекта для Schema.org
     * Основано на BreadcrumbBuilder v4.0::buildFinancialProductObject()
     * 
     * @param array $data Данные собранные в collectEquityData()
     * @return array Schema.org FinancialProduct объект
     */
    private function buildFinancialProduct($data) {
        $product = [];
        
        // Проверка обязательного URL инструмента
        if (!isset($data['urls']) || !is_array($data['urls']) || 
            !isset($data['urls']['instrument']) || !$this->isValidString($data['urls']['instrument'])) {
            error_log('SchemaOrgEquity: Missing or invalid instrument URL, cannot build FinancialProduct');
            return $product;
        }
        
        // Базовые поля
        $product['@type'] = 'FinancialProduct';
        $product['@id'] = $data['urls']['instrument'] . '#product';
        $product['additionalType'] = 'https://schema.org/Security';
        
        // Название (TYPE + CODE + SecName)
        $nameParts = [];
        if (isset($data['instrument']['type']) && $this->isValidString($data['instrument']['type'])) {
            $nameParts[] = $data['instrument']['type'];
        }
        if (isset($data['instrument']['trade_code']) && $this->isValidString($data['instrument']['trade_code'])) {
            $nameParts[] = $data['instrument']['trade_code'];
        }
        if (isset($data['security']['sec_name']) && $this->isValidString($data['security']['sec_name'])) {
            $nameParts[] = $data['security']['sec_name'];
        }
        
        if (count($nameParts) > 0) {
            $product['name'] = implode(' ', $nameParts);
        }
        
        // ISIN код
        if (isset($data['instrument']['isin']) && $this->isValidString($data['instrument']['isin'])) {
            $product['isinCode'] = $data['instrument']['isin'];
        }
        
        // CFI код (международный стандарт классификации финансовых инструментов)
        if (isset($data['security']['cfi']) && $this->isValidString($data['security']['cfi'])) {
            $product['cfiCode'] = $data['security']['cfi'];
        }
        
        // Регистрационный номер
        if (isset($data['instrument']['registry_number']) && $this->isValidString($data['instrument']['registry_number'])) {
            $product['registryNumber'] = $data['instrument']['registry_number'];
        }
        
        // Уровень листинга
        if (isset($data['instrument']['list_section']) && $this->isValidString($data['instrument']['list_section'])) {
            $product['listingLevel'] = $data['instrument']['list_section'];
        }
        
        // Категория (всегда Security для финансовых инструментов)
        $product['category'] = 'Security';
        
        if (isset($data['instrument']['type']) && $this->isValidString($data['instrument']['type'])) {
            $product['subCategory'] = $data['instrument']['type'];
        }
        
        // Описание
        $descParts = ['Акция'];
        if (isset($data['instrument']['trade_code']) && $this->isValidString($data['instrument']['trade_code'])) {
            $descParts[] = $data['instrument']['trade_code'];
        }
        if (isset($data['emitent']['short_name']) && $this->isValidString($data['emitent']['short_name'])) {
            $descParts[] = 'компании ' . $data['emitent']['short_name'];
        }
        if (isset($data['security']['class_name']) && $this->isValidString($data['security']['class_name'])) {
            $descParts[] = 'торгуется на Московской бирже в режиме ' . $data['security']['class_name'];
        }
        
        if (count($descParts) > 1) {
            $product['description'] = implode(' ', $descParts);
        }
        
        // Альтернативные названия
        $alternateNames = [];
        if (isset($data['security']['sec_name']) && isset($data['security']['short_name']) && 
            $this->isValidString($data['security']['sec_name']) && 
            $this->isValidString($data['security']['short_name']) && 
            $data['security']['sec_name'] !== $data['security']['short_name']) {
            $alternateNames[] = $data['security']['sec_name'];
        }
        if (isset($data['security']['short_name']) && $this->isValidString($data['security']['short_name'])) {
            $alternateNames[] = $data['security']['short_name'];
        }
        // v1.3: Добавляем только если есть минимум 2 разных имени
        $uniqueNames = array_values(array_unique($alternateNames));
        if (count($uniqueNames) > 1) {
            $product['alternateName'] = $uniqueNames;
        }
        
        // URL страницы (обязательное поле, уже проверено в начале метода)
        $product['url'] = $data['urls']['instrument'];
        
        // Изображение (логотип эмитента)
        if (isset($data['emitent']['og_image']) && $this->isValidString($data['emitent']['og_image'])) {
            $product['image'] = [
                '@type' => 'ImageObject',
                'url' => $this->baseUrl . '/images/' . $data['emitent']['og_image'],
                'caption' => 'Логотип ' . (isset($data['emitent']['short_name']) ? $data['emitent']['short_name'] : 'эмитента')
            ];
        }
        
        // Бренд
        if (isset($data['emitent']['short_name']) && isset($data['urls']['emitent']) && 
            $this->isValidString($data['emitent']['short_name']) && 
            $this->isValidString($data['urls']['emitent'])) {
            $product['brand'] = [
                '@type' => 'Brand',
                'name' => $data['emitent']['short_name'],
                'url' => $data['urls']['emitent']
            ];
        }
        
        // Класс торговли (ИЗМЕНЕНО v1.1: class → tradingClass)
        if (isset($data['security']['class_name']) && $this->isValidString($data['security']['class_name'])) {
            $product['tradingClass'] = $data['security']['class_name'];
        }
        
        if (isset($data['security']['class_code']) && $this->isValidString($data['security']['class_code'])) {
            $product['classCode'] = $data['security']['class_code'];
        }
        
        // Номинальная стоимость
        if (isset($data['security']['face_value']) && $this->isValidNumber($data['security']['face_value'])) {
            $product['nominal'] = $data['security']['face_value'];
        }
        
        // Валюта (конвертируем SUR → RUB для Schema.org)
        if (isset($data['security']['currency_id']) && $this->isValidString($data['security']['currency_id'])) {
            $currency = $data['security']['currency_id'];
            // Конвертируем биржевой код SUR в стандартный RUB
            $product['currency'] = ($currency === 'SUR') ? 'RUB' : $currency;
        }
        
        if (isset($data['security']['face_unit']) && $this->isValidString($data['security']['face_unit'])) {
            $faceUnit = $data['security']['face_unit'];
            // Конвертируем биржевой код SUR в стандартный RUB
            $product['currencyUnit'] = ($faceUnit === 'SUR') ? 'RUB' : $faceUnit;
        }
        
        // Размер лота
        if (isset($data['security']['lot_size']) && $this->isValidNumber($data['security']['lot_size'])) {
            $product['lotSize'] = $data['security']['lot_size'];
        }
        
        // Объём выпуска
        if (isset($data['security']['issue_size']) && $this->isValidNumber($data['security']['issue_size'])) {
            $product['issueVolume'] = $data['security']['issue_size'];
        }
        
        // Объём в обращении (v1.3: добавляем только если отличается от выпуска)
        if (isset($data['security']['issue_size_placed']) && 
            $this->isValidNumber($data['security']['issue_size_placed'])) {
            
            $placed = (int)$data['security']['issue_size_placed'];
            $issued = isset($data['security']['issue_size']) ? (int)$data['security']['issue_size'] : null;
            
            // Добавляем только если размещено ≠ выпущено
            if ($issued === null || $placed !== $issued) {
                $product['circulatingVolume'] = $placed;
            }
        }
        
        // Статус торгов (ИЗМЕНЕНО v1.1: конвертация в международные стандарты)
        if (isset($data['security']['status']) && $this->isValidString($data['security']['status'])) {
            $status = $data['security']['status'];
            // Конвертация биржевых статусов в международные стандарты
            if ($status === 'торгуется') {
                $product['status'] = 'Active';
            } elseif ($status === 'приостановлена') {
                $product['status'] = 'Suspended';
            } elseif ($status === 'не торгуется') {
                $product['status'] = 'Inactive';
            } else {
                // Для неизвестных статусов оставляем как есть
                $product['status'] = $status;
            }
        }
        
        // Шаг цены
        if (isset($data['security']['min_price_step']) && $this->isValidNumber($data['security']['min_price_step'])) {
            $product['priceStep'] = $data['security']['min_price_step'];
        }
        
        // Точность цены
        if (isset($data['security']['scale']) && $this->isValidNumber($data['security']['scale'])) {
            $product['pricePrecision'] = $data['security']['scale'];
        }
        
        // Free Float
        if (isset($data['instrument']['free_float']) && $this->isValidNumber($data['instrument']['free_float'])) {
            $product['freeFloat'] = $data['instrument']['free_float'];
        }
        
        // Restricted Weight
        if (isset($data['instrument']['restricted_weight']) && $this->isValidNumber($data['instrument']['restricted_weight'])) {
            $product['restrictedWeight'] = $data['instrument']['restricted_weight'];
        }
        
        // Базис котировок
        if (isset($data['security']['quote_basis']) && $this->isValidString($data['security']['quote_basis'])) {
            $product['quoteBasis'] = $data['security']['quote_basis'];
        }
        
        // Код расчётов
        if (isset($data['security']['settle_code']) && $this->isValidString($data['security']['settle_code'])) {
            $product['settlementCode'] = $data['security']['settle_code'];
        }
        
        // Дата расчётов
        if (isset($data['security']['settle_date']) && $this->isValidString($data['security']['settle_date'])) {
            $product['settlementDate'] = $data['security']['settle_date'];
        }
        
        // Дата последнего обновления
        if (isset($data['security']['timestamp']) && $this->isValidString($data['security']['timestamp'])) {
            $product['dateModified'] = $data['security']['timestamp'];
        }
        
        // Международные идентификаторы (v1.3: через PropertyValue)
        $identifiers = [
            $data['instrument']['trade_code']  // основной тикер
        ];
        
        // Добавляем дополнительные ID через PropertyValue (только если заполнены)
        if (isset($data['security']['bloomberg_id']) && 
            $this->isValidString($data['security']['bloomberg_id'])) {
            $identifiers[] = [
                '@type' => 'PropertyValue',
                'propertyID' => 'BloombergID',
                'value' => $data['security']['bloomberg_id']
            ];
        }
        
        if (isset($data['security']['ric']) && 
            $this->isValidString($data['security']['ric'])) {
            $identifiers[] = [
                '@type' => 'PropertyValue',
                'propertyID' => 'RIC',
                'value' => $data['security']['ric']
            ];
        }
        
        if (isset($data['security']['sedol']) && 
            $this->isValidString($data['security']['sedol'])) {
            $identifiers[] = [
                '@type' => 'PropertyValue',
                'propertyID' => 'SEDOL',
                'value' => $data['security']['sedol']
            ];
        }
        
        if (isset($data['security']['cusip']) && 
            $this->isValidString($data['security']['cusip'])) {
            $identifiers[] = [
                '@type' => 'PropertyValue',
                'propertyID' => 'CUSIP',
                'value' => $data['security']['cusip']
            ];
        }
        
        // Если есть дополнительные ID - делаем массив, иначе простая строка
        $product['identifier'] = count($identifiers) > 1 ? $identifiers : $identifiers[0];
        
        // Место торгов (только для инструментов торгующихся на MOEX)
        // Проверяем по ClassCode начинающемуся с 'TQ' или статусу торгов
        if (isset($data['security']['class_code']) && $this->isValidString($data['security']['class_code'])) {
            $classCode = $data['security']['class_code'];
            $status = isset($data['security']['status']) ? $data['security']['status'] : '';
            
            // Проверка начинается ли ClassCode с 'TQ' или статус = 'торгуется' (до конвертации)
            if (substr($classCode, 0, 2) === 'TQ' || $status === 'торгуется') {
                $product['availableAtOrFrom'] = [
                    '@type' => 'Place',
                    'name' => 'Московская Биржа',
                    'url' => 'https://www.moex.com',
                    'identifier' => 'MOEX'
                ];
                
                // Область обслуживания (только для MOEX инструментов)
                $product['areaServed'] = [
                    '@type' => 'Country',
                    'name' => 'Россия',
                    'identifier' => 'RU'
                ];
            }
        }
        
        // Эмитент (issuer)
        // Проверка обязательного URL эмитента
        if (!isset($data['urls']['emitent']) || !$this->isValidString($data['urls']['emitent'])) {
            error_log('SchemaOrgEquity: Missing emitent URL for issuer');
            // Не добавляем issuer если нет URL
        } else {
            $issuer = [
                '@type' => 'Organization',
                '@id' => $data['urls']['emitent'] . '#organization'
            ];
            
            if (isset($data['emitent']['short_name']) && $this->isValidString($data['emitent']['short_name'])) {
                $issuer['name'] = $data['emitent']['short_name'];
            }
            
            if (isset($data['emitent']['full_name']) && $this->isValidString($data['emitent']['full_name'])) {
                $issuer['legalName'] = $data['emitent']['full_name'];
            }
            
            if (isset($data['emitent']['inn']) && $this->isValidString($data['emitent']['inn'])) {
                $issuer['taxID'] = $data['emitent']['inn'];
            }
            
            if (isset($data['emitent']['lei']) && $this->isValidString($data['emitent']['lei'])) {
                $issuer['leiCode'] = $data['emitent']['lei'];
            }
            
            if (isset($data['emitent']['founding_date']) && $this->isValidString($data['emitent']['founding_date'])) {
                $issuer['foundingDate'] = $data['emitent']['founding_date'];
            }
            
            if (isset($data['emitent']['id']) && $data['emitent']['id'] > 0) {
                $issuer['identifier'] = (string)$data['emitent']['id'];
            }
            
            $issuer['url'] = $data['urls']['emitent'];
            
            if (isset($data['emitent']['og_image']) && $this->isValidString($data['emitent']['og_image'])) {
                $logoUrl = $this->baseUrl . '/images/' . $data['emitent']['og_image'];
                $issuer['logo'] = $logoUrl;
                $issuer['image'] = $logoUrl;
            }
            
            // ПРОВЕРЕНО v1.1: ucfirst() работает корректно
            // txt_short уже обработан через safeString() в collectEquityData()
            // ucfirst() применяется к результату safeString() - это правильно!
            if (isset($data['emitent']['txt_short']) && $this->isValidString($data['emitent']['txt_short'])) {
                // Capitalize первую букву description для правильного оформления
                $issuer['description'] = ucfirst($data['emitent']['txt_short']);
            }
            
            // Ссылки sameAs для эмитента
            $sameAs = [];
            if (isset($data['emitent']['disclosure_page']) && $this->isValidString($data['emitent']['disclosure_page'])) {
                $sameAs[] = $data['emitent']['disclosure_page'];
            }
            if (isset($data['emitent']['disclosure_rf']) && $this->isValidString($data['emitent']['disclosure_rf'])) {
                $sameAs[] = $data['emitent']['disclosure_rf'];
            }
            if (isset($data['emitent']['wiki']) && $this->isValidString($data['emitent']['wiki'])) {
                $sameAs[] = $data['emitent']['wiki'];
            }
            if (isset($data['emitent']['smart_lab']) && $this->isValidString($data['emitent']['smart_lab'])) {
                $sameAs[] = $data['emitent']['smart_lab'];
            }
            if (isset($data['emitent']['dohod']) && $this->isValidString($data['emitent']['dohod'])) {
                $sameAs[] = $data['emitent']['dohod'];
            }
            if (isset($data['emitent']['moex']) && $this->isValidString($data['emitent']['moex'])) {
                $sameAs[] = $data['emitent']['moex'];
            }
            
            if (count($sameAs) > 0) {
                $issuer['sameAs'] = $sameAs;
            }
            
            $product['issuer'] = $issuer;
        }
        
        // Действие (potentialAction)
        if (isset($data['instrument']['trade_code']) && isset($data['emitent']['short_name']) && 
            $this->isValidString($data['instrument']['trade_code']) && 
            $this->isValidString($data['emitent']['short_name'])) {
            $instrumentType = isset($data['instrument']['type']) ? $data['instrument']['type'] : 'акции';
            $product['potentialAction'] = [
                '@type' => 'ViewAction',
                'target' => $data['urls']['instrument'],
                'name' => 'Анализ ' . $data['instrument']['trade_code'],
                'description' => 'Полная аналитика по ' . 
                    $instrumentType . ' ' . 
                    $data['instrument']['trade_code'] . 
                    ' от компании ' . $data['emitent']['short_name']
            ];
        }
        
        return $product;
    }
}