<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ КАТАЛОГА ЭМИТЕНТОВ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.0 - INITIAL RELEASE
 * 
 * CHANGELOG v1.0 (2025-11-03):
 *   ✨ NEW: Класс для генерации Schema.org разметки каталога эмитентов
 *   ✨ NEW: Метод loadCatalogDataFromDB() для загрузки данных из БД
 *   ✨ NEW: Простая структура CollectionPage с ItemList эмитентов
 *   ✅ Подсчёт количества инструментов у каждого эмитента
 *   ✅ SQL запрос с GROUP BY и LEFT JOIN
 *   ✅ Prepared statement для безопасности
 *   ✅ Полная валидация данных
 * 
 * НАЗНАЧЕНИЕ:
 *   - Генерация Schema.org разметки для страницы /emitents/
 *   - CollectionPage с полным списком эмитентов Московской биржи
 *   - Каждый эмитент: название, URL, сектор, количество инструментов
 *   - Оптимизировано для PAGE cache (генерация 1 раз/месяц)
 * 
 * СТРУКТУРА JSON-LD:
 *   @graph [
 *     BreadcrumbList - навигационные крошки
 *     CollectionPage - страница каталога
 *       name: "Каталог эмитентов Московской биржи"
 *       mainEntity: ItemList - список эмитентов
 *         ListItem - каждый эмитент с названием, URL, сектором, кол-вом инструментов
 *   ]
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Размер JSON: ~40-60 KB (682 эмитента)
 *   - С gzip: ~8-12 KB
 *   - Генерация: 1 раз/месяц (PAGE cache)
 *   - Из кэша: 0.001 сек
 * 
 * SQL ЗАПРОС:
 *   SELECT e.*, COUNT(i.Id) as instrument_count
 *   FROM Emitent e
 *   LEFT JOIN Instrument i ON e.Id = i.Emitent_Id
 *   GROUP BY e.Id
 *   ORDER BY e.EMITENT_SHORT_NAME
 * 
 * ОСНОВАНО НА:
 *   - SchemaOrgBase v1.1
 *   - SchemaOrgSectorsCatalog v1.0 (структура и подход)
 * 
 * @version 1.0
 * @date 2025-11-03
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgEmitentsCatalog extends SchemaOrgBase {
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение JSON-LD разметки для каталога эмитентов
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'MENU', 1 => объект MENU]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка при ошибке
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (empty($page_data) || !is_array($page_data)) {
            error_log('SchemaOrgEmitentsCatalog: Invalid page_data format');
            return '';
        }
        
        // Загружаем данные каталога из БД
        $catalog = $this->loadCatalogDataFromDB();
        
        // Проверка что данные загружены
        if (empty($catalog) || !is_object($catalog)) {
            error_log('SchemaOrgEmitentsCatalog: Failed to load catalog data from DB');
            return '';
        }
        
        // Валидация критичных полей
        if (!property_exists($catalog, 'emitents') || !is_array($catalog->emitents)) {
            error_log('SchemaOrgEmitentsCatalog: Missing or invalid emitents data after DB load');
            return '';
        }
        
        // Построение @graph
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        if (!empty($breadcrumbs) && is_array($breadcrumbs)) {
            $breadcrumbList = $this->buildBreadcrumbList($breadcrumbs);
            if (!empty($breadcrumbList)) {
                $graph[] = $breadcrumbList;
            }
        }
        
        // 2. CollectionPage со списком эмитентов
        $collectionPage = $this->buildCatalogCollectionPage($catalog);
        
        if (!empty($collectionPage)) {
            $graph[] = $collectionPage;
        }
        
        // Если нет данных для вывода - возвращаем пустую строку
        if (empty($graph)) {
            error_log('SchemaOrgEmitentsCatalog: Empty graph, nothing to output');
            return '';
        }
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * МЕТОД ЗАГРУЗКИ ДАННЫХ - Загрузка каталога эмитентов из БД
     * 
     * Загружает все эмитенты с подсчётом количества инструментов
     * у каждого эмитента через LEFT JOIN и GROUP BY
     * 
     * @return object|null Объект каталога или null при ошибке
     */
    private function loadCatalogDataFromDB() {
        // Проверка подключения к БД
        if (!$this->db instanceof mysqli) {
            error_log('SchemaOrgEmitentsCatalog: Invalid database connection');
            return null;
        }
        
        // SQL запрос: получаем все эмитенты с подсчётом инструментов
        $sql = "SELECT 
                    e.Id,
                    e.EMITENT_SHORT_NAME,
                    e.EMITENT_URL,
                    e.SECTOR_ECONOMIKI,
                    e.INN,
                    COUNT(i.Id) as instrument_count
                FROM Emitent e
                LEFT JOIN Instrument i ON e.Id = i.Emitent_Id
                GROUP BY e.Id, e.EMITENT_SHORT_NAME, e.EMITENT_URL, e.SECTOR_ECONOMIKI, e.INN
                ORDER BY e.EMITENT_SHORT_NAME";
        
        // Выполнение запроса (без prepared statement т.к. нет параметров)
        $result = $this->db->query($sql);
        
        if (!$result) {
            error_log('SchemaOrgEmitentsCatalog: SQL query failed - ' . $this->db->error);
            return null;
        }
        
        // Инициализация структуры данных
        $emitents = [];
        $total_emitents = 0;
        $total_instruments = 0;
        
        // Обработка результатов
        while ($row = $result->fetch_assoc()) {
            // Валидация обязательных полей
            if (empty($row['Id']) || empty($row['EMITENT_SHORT_NAME'])) {
                continue;
            }
            
            $emitent_id = (int)$row['Id'];
            $instrument_count = isset($row['instrument_count']) ? (int)$row['instrument_count'] : 0;
            
            // Формирование данных эмитента
            $emitents[] = [
                'id' => $emitent_id,
                'name' => trim($row['EMITENT_SHORT_NAME']),
                'url' => !empty($row['EMITENT_URL']) ? trim($row['EMITENT_URL']) : null,
                'sector' => !empty($row['SECTOR_ECONOMIKI']) ? trim($row['SECTOR_ECONOMIKI']) : 'Не указан',
                'inn' => !empty($row['INN']) ? trim($row['INN']) : null,
                'instrument_count' => $instrument_count
            ];
            
            $total_emitents++;
            $total_instruments += $instrument_count;
        }
        
        // Освобождение результата
        $result->free();
        
        // Проверка что есть данные
        if (empty($emitents)) {
            error_log('SchemaOrgEmitentsCatalog: No emitents found in database');
            return null;
        }
        
        // Формирование объекта каталога
        $catalog = new stdClass();
        $catalog->emitents = $emitents;
        $catalog->total_emitents = $total_emitents;
        $catalog->total_instruments = $total_instruments;
        
        return $catalog;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ COLLECTIONPAGE - Построение Schema.org CollectionPage
     * 
     * Создаёт CollectionPage с mainEntity содержащим ItemList всех эмитентов
     * 
     * @param object $catalog Объект каталога с данными эмитентов
     * @return array Schema.org CollectionPage объект или пустой массив
     */
    private function buildCatalogCollectionPage($catalog) {
        // Валидация входных данных
        if (empty($catalog) || !is_object($catalog)) {
            return [];
        }
        
        if (!property_exists($catalog, 'emitents') || !is_array($catalog->emitents)) {
            return [];
        }
        
        if (empty($catalog->emitents)) {
            return [];
        }
        
        // Базовая структура CollectionPage
        $page = [
            '@type' => 'CollectionPage',
            'name' => 'Каталог эмитентов Московской биржи',
            'description' => 'Полный список эмитентов ценных бумаг, торгующихся на Московской бирже',
            'url' => $this->baseUrl . '/emitents/'
        ];
        
        // Добавление статистики (если доступна)
        if (property_exists($catalog, 'total_emitents') && $catalog->total_emitents > 0) {
            $emitent_word = $this->pluralizeEmitents($catalog->total_emitents);
            $page['description'] .= ' (' . $catalog->total_emitents . ' ' . $emitent_word;
            
            if (property_exists($catalog, 'total_instruments') && $catalog->total_instruments > 0) {
                $instrument_word = $this->pluralizeInstruments($catalog->total_instruments);
                $page['description'] .= ', ' . $catalog->total_instruments . ' ' . $instrument_word;
            }
            
            $page['description'] .= ')';
        }
        
        // Построение mainEntity: ItemList эмитентов
        $itemList = $this->buildEmitentsItemList($catalog->emitents);
        
        if (!empty($itemList)) {
            $page['mainEntity'] = $itemList;
        }
        
        return $page;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ ITEMLIST - Построение ItemList эмитентов
     * 
     * @param array $emitents Массив эмитентов из БД
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildEmitentsItemList($emitents) {
        // Валидация входных данных
        if (!is_array($emitents) || empty($emitents)) {
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Эмитенты',
            'numberOfItems' => count($emitents),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор всех эмитентов
        foreach ($emitents as $emitentData) {
            // Валидация структуры эмитента
            if (!is_array($emitentData) || !isset($emitentData['name'])) {
                continue;
            }
            
            // Построение ListItem для эмитента
            $listItem = $this->buildEmitentListItem($emitentData, $position);
            
            if (!empty($listItem)) {
                $itemList['itemListElement'][] = $listItem;
                $position++;
            }
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ LISTITEM - Построение ListItem для одного эмитента
     * 
     * @param array $emitentData Данные эмитента
     * @param int $position Позиция в списке
     * @return array Schema.org ListItem объект или пустой массив
     */
    private function buildEmitentListItem($emitentData, $position) {
        // Валидация входных данных
        if (!is_array($emitentData) || !isset($emitentData['name'])) {
            return [];
        }
        
        $listItem = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => $this->safeString($emitentData['name'])
        ];
        
        // Добавление URL эмитента (если есть)
        if (isset($emitentData['url']) && !empty($emitentData['url'])) {
            $listItem['url'] = $this->baseUrl . '/' . $this->safeString($emitentData['url']) . '/';
        }
        
        // Построение описания с сектором и количеством инструментов
        $description_parts = [];
        
        // Добавление сектора
        if (isset($emitentData['sector']) && !empty($emitentData['sector'])) {
            $description_parts[] = $this->safeString($emitentData['sector']);
        }
        
        // Добавление количества инструментов
        if (isset($emitentData['instrument_count']) && $emitentData['instrument_count'] > 0) {
            $count = (int)$emitentData['instrument_count'];
            $instrument_word = $this->pluralizeInstruments($count);
            $description_parts[] = $count . ' ' . $instrument_word;
        }
        
        // Формирование финального описания
        if (!empty($description_parts)) {
            $listItem['description'] = implode(', ', $description_parts);
        } else {
            $listItem['description'] = 'Эмитент ценных бумаг';
        }
        
        return $listItem;
    }
    
    /**
     * ВСПОМОГАТЕЛЬНЫЙ МЕТОД - Плюрализация слова "эмитент"
     * 
     * @param int $count Количество эмитентов
     * @return string Правильная форма слова
     */
    private function pluralizeEmitents($count) {
        $count = abs($count) % 100;
        $leastSignificant = $count % 10;
        
        if ($count >= 11 && $count <= 19) {
            return 'эмитентов';
        }
        
        if ($leastSignificant == 1) {
            return 'эмитент';
        }
        
        if ($leastSignificant >= 2 && $leastSignificant <= 4) {
            return 'эмитента';
        }
        
        return 'эмитентов';
    }
    
    /**
     * ВСПОМОГАТЕЛЬНЫЙ МЕТОД - Плюрализация слова "инструмент"
     * 
     * @param int $count Количество инструментов
     * @return string Правильная форма слова
     */
    private function pluralizeInstruments($count) {
        $count = abs($count) % 100;
        $leastSignificant = $count % 10;
        
        if ($count >= 11 && $count <= 19) {
            return 'инструментов';
        }
        
        if ($leastSignificant == 1) {
            return 'инструмент';
        }
        
        if ($leastSignificant >= 2 && $leastSignificant <= 4) {
            return 'инструмента';
        }
        
        return 'инструментов';
    }
}