<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ СТРАНИЦ ЭМИТЕНТОВ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.1 - ИСПРАВЛЕНИЕ: использование правильного индекса $page_data[1]
 * 
 * CHANGELOG v1.1 (2025-10-13):
 *   🔧 FIX: Исправлена проверка индекса эмитента
 *   🔧 FIX: $page_data[0] → $page_data[1] (объект эмитента)
 *   ✅ Добавлен fallback для разных форматов данных
 *   ✅ Соответствует формату данных из DB/page.php
 * 
 * НАЗНАЧЕНИЕ:
 *   - Schema.org разметка для страниц эмитентов (компаний)
 *   - Построение Organization объекта с полными данными
 *   - Интеграция с BreadcrumbList
 * 
 * ОСНОВАНО НА:
 *   SchemaOrgBase v1.1 - базовый класс с helper методами
 *   SchemaOrgEquity v1.1 - структура и graceful error handling
 * 
 * @version 1.1
 * @date 2025-10-13
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgEmitent extends SchemaOrgBase {
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение JSON-LD разметки для страницы эмитента
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'EMITENT', 1 => Emitent object]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (empty($page_data)) {
            return '';
        }
        
        // Определение объекта эмитента (ИСПРАВЛЕНО v1.1)
        // Формат от DB/page.php: ['EMITENT', Emitent object]
        // [0] = 'EMITENT' (строка типа)
        // [1] = Emitent object (нужный нам объект)
        $emitent = null;
        
        if (is_array($page_data)) {
            // Стандартный формат от DB/page.php - используем индекс [1]
            if (isset($page_data[1]) && is_object($page_data[1])) {
                $emitent = $page_data[1];
            }
            // Альтернативный формат с ключом 'emitent' (на всякий случай)
            elseif (isset($page_data['emitent']) && is_object($page_data['emitent'])) {
                $emitent = $page_data['emitent'];
            }
        } elseif (is_object($page_data)) {
            // Если передан напрямую объект
            $emitent = $page_data;
        }
        
        // Проверка что эмитент найден и это объект
        if (!is_object($emitent)) {
            error_log('SchemaOrgEmitent: Invalid emitent data provided');
            return '';
        }
        
        // Построение @graph с BreadcrumbList и Organization
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        if (!empty($breadcrumbs)) {
            $graph[] = $this->buildBreadcrumbList($breadcrumbs);
        }
        
        // 2. Organization с полными данными эмитента
        $organization = $this->buildFullOrganization($emitent);
        
        if (!empty($organization)) {
            $graph[] = $organization;
        }
        
        // Если нет данных для вывода - возвращаем пустую строку
        if (empty($graph)) {
            return '';
        }
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * Построение полного Organization объекта для страницы эмитента
     * 
     * @param object $emitent Объект эмитента из БД
     * @return array Schema.org Organization объект
     */
    private function buildFullOrganization($emitent) {
        // Базовая проверка объекта
        if (!is_object($emitent)) {
            error_log('SchemaOrgEmitent: buildFullOrganization received non-object');
            return [];
        }
        
        $org = [
            '@type' => 'Organization'
        ];
        
        // @id - уникальный идентификатор организации
        if (property_exists($emitent, 'EMITENT_URL') && $this->isValidString($emitent->EMITENT_URL)) {
            $org['@id'] = $this->baseUrl . '/' . $this->safeString($emitent->EMITENT_URL) . '/#organization';
        }
        
        // Короткое название (основное имя)
        if (property_exists($emitent, 'EMITENT_SHORT_NAME') && $this->isValidString($emitent->EMITENT_SHORT_NAME)) {
            $org['name'] = $this->safeString($emitent->EMITENT_SHORT_NAME);
        }
        
        // Полное юридическое название
        if (property_exists($emitent, 'EMITENT_FULL_NAME') && $this->isValidString($emitent->EMITENT_FULL_NAME)) {
            $org['legalName'] = $this->safeString($emitent->EMITENT_FULL_NAME);
        }
        
        // ИНН (tax ID)
        if (property_exists($emitent, 'INN') && $this->isValidString($emitent->INN)) {
            $org['taxID'] = $this->safeString($emitent->INN);
        }
        
        // LEI код (Legal Entity Identifier) - может быть NULL
        if (property_exists($emitent, 'LEI') && $this->isValidString($emitent->LEI)) {
            $org['leiCode'] = $this->safeString($emitent->LEI);
        }
        
        // Дата основания - может быть NULL, формат YYYY-MM-DD
        if (property_exists($emitent, 'EMITENT_FOUNDING_DATE') && $this->isValidString($emitent->EMITENT_FOUNDING_DATE)) {
            // Проверка что дата не 0000-00-00
            $foundingDate = $this->safeString($emitent->EMITENT_FOUNDING_DATE);
            if ($foundingDate !== '0000-00-00' && $foundingDate !== '') {
                $org['foundingDate'] = $foundingDate;
            }
        }
        
        // ID эмитента в системе
        if (property_exists($emitent, 'Id') && !empty($emitent->Id)) {
            $org['identifier'] = (string)$emitent->Id;
        }
        
        // URL страницы эмитента
        if (property_exists($emitent, 'EMITENT_URL') && $this->isValidString($emitent->EMITENT_URL)) {
            $org['url'] = $this->baseUrl . '/' . $this->safeString($emitent->EMITENT_URL) . '/';
        }
        
        // Описание (SHORT текст с заглавной буквы)
        if (property_exists($emitent, 'EMITENT_TXT_SHORT') && $this->isValidString($emitent->EMITENT_TXT_SHORT)) {
            // Capitalize первую букву для правильного оформления
            $org['description'] = ucfirst($this->safeString($emitent->EMITENT_TXT_SHORT));
        }
        
        // Логотип и изображение
        if (property_exists($emitent, 'EMITENT_og_image') && $this->isValidString($emitent->EMITENT_og_image)) {
            $logoUrl = $this->baseUrl . '/images/' . $this->safeString($emitent->EMITENT_og_image);
            $org['logo'] = $logoUrl;
            $org['image'] = $logoUrl;
        }
        
        // Сектор экономики (industry)
        if (property_exists($emitent, 'SECTOR_ECONOMIKI') && $this->isValidString($emitent->SECTOR_ECONOMIKI)) {
            $org['industry'] = $this->safeString($emitent->SECTOR_ECONOMIKI);
        }
        
        // Ссылки sameAs (внешние ресурсы о компании)
        $sameAs = [];
        
        // Официальный сайт компании
        if (property_exists($emitent, 'DISCLOSURE_PART_PAGE') && $this->isValidString($emitent->DISCLOSURE_PART_PAGE)) {
            $sameAs[] = $this->safeString($emitent->DISCLOSURE_PART_PAGE);
        }
        
        // Раскрытие информации (e-disclosure)
        if (property_exists($emitent, 'DISCLOSURE_RF_INFO_PAGE') && $this->isValidString($emitent->DISCLOSURE_RF_INFO_PAGE)) {
            $sameAs[] = $this->safeString($emitent->DISCLOSURE_RF_INFO_PAGE);
        }
        
        // Википедия
        if (property_exists($emitent, 'DISCLOSURE_PART_wiki') && $this->isValidString($emitent->DISCLOSURE_PART_wiki)) {
            $sameAs[] = $this->safeString($emitent->DISCLOSURE_PART_wiki);
        }
        
        // Смартлаб (проверяем оба варианта названия поля)
        $smartLabValue = null;
        if (property_exists($emitent, 'DISCLOSURE_PART_smartlab')) {
            $smartLabValue = $emitent->DISCLOSURE_PART_smartlab;
        } elseif (property_exists($emitent, 'DISCLOSURE_PART_smart-lab')) {
            $smartLabValue = $emitent->{'DISCLOSURE_PART_smart-lab'};
        }
        if ($this->isValidString($smartLabValue)) {
            $sameAs[] = $this->safeString($smartLabValue);
        }
        
        // Доходъ
        if (property_exists($emitent, 'DISCLOSURE_PART_dohod') && $this->isValidString($emitent->DISCLOSURE_PART_dohod)) {
            $sameAs[] = $this->safeString($emitent->DISCLOSURE_PART_dohod);
        }
        
        // Московская биржа
        if (property_exists($emitent, 'DISCLOSURE_PART_moex') && $this->isValidString($emitent->DISCLOSURE_PART_moex)) {
            $sameAs[] = $this->safeString($emitent->DISCLOSURE_PART_moex);
        }
        
        // Добавляем sameAs только если есть хотя бы одна ссылка
        if (!empty($sameAs)) {
            $org['sameAs'] = $sameAs;
        }
        
        return $org;
    }
    
    /**
     * Безопасное получение свойства объекта
     * Возвращает значение свойства или default если свойство не существует
     * 
     * @param object $object Объект
     * @param string $property Имя свойства
     * @param mixed $default Значение по умолчанию
     * @return mixed Значение свойства или default
     */
    private function getObjectProperty($object, $property, $default = null) {
        if (!is_object($object)) {
            return $default;
        }
        
        if (!property_exists($object, $property)) {
            return $default;
        }
        
        return $object->$property ?? $default;
    }
}