<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ ДОЛГОВЫХ ИНСТРУМЕНТОВ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.2.1 FIX - ИСПРАВЛЕНИЕ: settlementDate в ISO 8601 формате
 * 
 * CHANGELOG v1.2.1 (2025-10-21):
 *   🐛 FIX: settlementDate теперь в ISO 8601 формате (YYYY-MM-DD вместо дд.мм.гггг)
 *   ✅ Полное соответствие Schema.org для всех полей типа Date
 * 
 * CHANGELOG v1.2 (2025-10-21):
 *   ✅ NEW: Метод convertTimestampToISO() для ISO 8601 формата с timezone
 *   ✅ FIX: dateModified теперь в формате ISO 8601 (YYYY-MM-DDTHH:MM:SS±HH:MM)
 *   ✅ NEW: Динамические @id фрагменты (#bond/#eurobond/#mortgage-note) вместо общего #product
 *   ✅ SEO: Улучшена совместимость с Google Rich Results и Yandex валидаторами
 * 
 * CHANGELOG v1.1 (2025-10-13):
 *   🔧 FIX: Изменён формат доступа к $page_data с ключей на индексы
 *   🔧 FIX: $page_data['instrument'] → $page_data[1]
 *   🔧 FIX: $page_data['security'] → $page_data[2]
 *   🔧 FIX: $page_data['emitent'] → $page_data[3]
 *   ✅ Соответствует формату данных из DB/page.php
 * 
 * НАЗНАЧЕНИЕ:
 *   - Schema.org разметка для страниц облигаций (type 6)
 *   - Schema.org разметка для еврооблигаций (type 3)
 *   - Schema.org разметка для ипотечных сертификатов (type 5)
 *   - Построение FinancialProduct объекта с облигационными полями
 *   - Интеграция с BreadcrumbList
 * 
 * ОСНОВАНО НА:
 *   SchemaOrgEquity v1.1 - структура и логика
 *   Адаптация под долговые инструменты
 * 
 * КЛЮЧЕВЫЕ ОТЛИЧИЯ ОТ EQUITY:
 *   - Используется ShortName вместо SecName для имени
 *   - Добавлены облигационные поля (maturityDate, couponRate, etc.)
 *   - Конвертация дат из дд.мм.гггг в ISO 8601
 *   - Конвертация timestamp в ISO 8601 с timezone
 *   - Расчёт количества купонных выплат в год
 *   - Категория "Bond" вместо "Security"
 *   - Специфичные @id фрагменты для каждого типа долгового инструмента
 * 
 * @version 1.2.1
 * @date 2025-10-21
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgDebt extends SchemaOrgBase {
    
    // Instrument type constants
    const INSTRUMENT_TYPE_EUROBONDS = 3;
    const INSTRUMENT_TYPE_MORTGAGE_NOTES = 5;
    const INSTRUMENT_TYPE_BONDS = 6;
    
    // Cache constants
    const CACHE_NOT_FOUND = '__NOT_FOUND__';
    const CACHE_TTL = 3600;
    
    // Static cache
    private static $sectorCache = [];
    
    // Instrument types mapping (соответствует BreadcrumbBuilder v5.0)
    private static $instrumentTypes = [
        self::INSTRUMENT_TYPE_EUROBONDS => [
            'name' => 'Еврооблигации',
            'slug' => 'eurobonds'
        ],
        self::INSTRUMENT_TYPE_MORTGAGE_NOTES => [
            'name' => 'Ипотечные сертификаты',
            'slug' => 'mortgage-notes'
        ],
        self::INSTRUMENT_TYPE_BONDS => [
            'name' => 'Облигации',
            'slug' => 'bonds'
        ]
    ];
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение расширенной JSON-LD разметки для долговых инструментов
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'BOND', 1 => instrument, 2 => security, 3 => emitent]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (!is_array($page_data) || count($page_data) < 4) {
            return '';
        }
        
        // Проверка наличия необходимых индексов (ИСПРАВЛЕНО v1.1)
        if (!isset($page_data[1]) || !isset($page_data[2]) || !isset($page_data[3])) {
            error_log('SchemaOrgDebt: Missing required indexes in page_data (need [1], [2], [3])');
            return '';
        }
        
        // Сбор данных для FinancialProduct
        $data = $this->collectDebtData($page_data);
        
        if ($data === null) {
            return '';
        }
        
        // Построение @graph с BreadcrumbList и FinancialProduct
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        $graph[] = $this->buildBreadcrumbList($breadcrumbs);
        
        // 2. FinancialProduct с полными данными
        $graph[] = $this->buildFinancialProduct($data);
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * Сбор и подготовка всех данных для FinancialProduct
     * Адаптировано из SchemaOrgEquity::collectEquityData()
     * 
     * @param array $page_data Данные страницы
     * @return array|null Массив данных или null при ошибке
     */
    private function collectDebtData($page_data) {
        try {
            // ИСПРАВЛЕНО v1.1: Используем числовые индексы вместо ключей
            $instrument = $page_data[1];
            $security = $page_data[2];
            $emitent = $page_data[3];
            
            // Проверка что все объекты валидны
            if (!is_object($instrument) || !is_object($security) || !is_object($emitent)) {
                error_log('SchemaOrgDebt: Invalid objects in page_data');
                return null;
            }
            
            // Получение данных сектора если есть
            $sector = null;
            $sectorId = $this->getObjectProperty($emitent, 'ID_SECTOR_ECONOMIKI', 0);
            if ($sectorId > 0) {
                $sector = $this->getSectorData($sectorId);
            }
            
            // Формирование массива данных с безопасным чтением свойств
            $data = [
                'instrument' => [
                    'trade_code' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_TRADE_CODE', '')),
                    'isin' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_ISIN', '')),
                    'registry_number' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_REGISTRY_NUMBER', '')),
                    'type' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_TYPE', '')),
                    'supertype' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_SUPERTYPE', '')),
                    'supertype_id' => (int)$this->getObjectProperty($instrument, 'Instrument_SUPERTYPE_ID', 0),
                    'list_section' => $this->safeString($this->getObjectProperty($instrument, 'Instrument_LIST_SECTION', '')),
                    'free_float' => $this->safeDecimal($this->getObjectProperty($instrument, 'Instrument_MoexFreeFloat', null)),
                    'restricted_weight' => $this->safeDecimal($this->getObjectProperty($instrument, 'Instrument_MoexOgranichVes', null))
                ],
                'security' => [
                    'sec_name' => $this->safeString($this->getObjectProperty($security, 'SecName', '')),
                    'short_name' => $this->safeString($this->getObjectProperty($security, 'ShortName', '')),
                    'sec_code' => $this->safeString($this->getObjectProperty($security, 'SecCode', '')),
                    'isin' => $this->safeString($this->getObjectProperty($security, 'ISIN', '')),
                    'class_name' => $this->safeString($this->getObjectProperty($security, 'ClassName', '')),
                    'class_code' => $this->safeString($this->getObjectProperty($security, 'ClassCode', '')),
                    'face_value' => $this->safeDecimal($this->getObjectProperty($security, 'FaceValue', null)),
                    'face_unit' => $this->safeString($this->getObjectProperty($security, 'FaceUnit', '')),
                    'currency_id' => $this->safeString($this->getObjectProperty($security, 'CurrencyId', '')),
                    'lot_size' => $this->safeInteger($this->getObjectProperty($security, 'LotSize', null)),
                    'issue_size' => $this->safeInteger($this->getObjectProperty($security, 'IssueSize', null)),
                    'issue_size_placed' => $this->safeInteger($this->getObjectProperty($security, 'IssueSizePlaced', null)),
                    'status' => $this->safeString($this->getObjectProperty($security, 'SecStatus', '')),
                    'min_price_step' => $this->safeDecimal($this->getObjectProperty($security, 'MinPriceStep', null)),
                    'scale' => $this->safeInteger($this->getObjectProperty($security, 'Scale', null)),
                    'quote_basis' => $this->safeString($this->getObjectProperty($security, 'QuoteBasis', '')),
                    'settle_code' => $this->safeString($this->getObjectProperty($security, 'SettleCode', '')),
                    'settle_date' => $this->safeString($this->getObjectProperty($security, 'SettleDate', '')),
                    'timestamp' => $this->safeString($this->getObjectProperty($security, 'Timestamp', '')),
                    'cfi' => $this->safeString($this->getObjectProperty($security, 'CFI', '')),
                    // ОБЛИГАЦИОННЫЕ ПОЛЯ:
                    'mat_date' => $this->safeString($this->getObjectProperty($security, 'MatDate', '')),
                    'coupon_value' => $this->safeDecimal($this->getObjectProperty($security, 'CouponValue', null)),
                    'coupon_period' => $this->safeInteger($this->getObjectProperty($security, 'CouponPeriod', null)),
                    'next_coupon' => $this->safeString($this->getObjectProperty($security, 'NextCoupon', '')),
                    'accrued_int' => $this->safeDecimal($this->getObjectProperty($security, 'AccruedInt', null)),
                    'buyback_date' => $this->safeString($this->getObjectProperty($security, 'BuyBackDate', '')),
                    'buyback_price' => $this->safeDecimal($this->getObjectProperty($security, 'BuyBackPrice', null)),
                    'bloomberg_id' => $this->safeString($this->getObjectProperty($security, 'BloombergID', '')),
                    'sedol' => $this->safeString($this->getObjectProperty($security, 'SEDOL', '')),
                    'ric' => $this->safeString($this->getObjectProperty($security, 'RIC', '')),
                    'cusip' => $this->safeString($this->getObjectProperty($security, 'CUSIP', ''))
                ],
                'emitent' => [
                    'id' => (int)$this->getObjectProperty($emitent, 'Id', 0),
                    'short_name' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_SHORT_NAME', '')),
                    'full_name' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_FULL_NAME', '')),
                    'inn' => $this->safeString($this->getObjectProperty($emitent, 'INN', '')),
                    'url_slug' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_URL', '')),
                    'og_image' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_og_image', '')),
                    'txt_short' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_TXT_SHORT', '')),
                    'disclosure_page' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_PAGE', '')),
                    'disclosure_rf' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_RF_INFO_PAGE', '')),
                    'wiki' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_wiki', '')),
                    'smart_lab' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_smartlab', '') ?: $this->getObjectProperty($emitent, 'DISCLOSURE_PART_smart-lab', '')),
                    'dohod' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_dohod', '')),
                    'moex' => $this->safeString($this->getObjectProperty($emitent, 'DISCLOSURE_PART_moex', '')),
                    'founding_date' => $this->safeString($this->getObjectProperty($emitent, 'EMITENT_FOUNDING_DATE', '')),
                    'lei' => $this->safeString($this->getObjectProperty($emitent, 'LEI', ''))
                ],
                'sector' => $sector,
                'urls' => [
                    'instrument' => $this->buildInstrumentUrl($instrument),
                    'emitent' => $this->buildEmitentUrl($emitent),
                    'sector' => $sector ? $this->baseUrl . '/' . $this->safeString(isset($sector['url']) ? $sector['url'] : '') . '/' : null
                ]
            ];
            
            return $data;
            
        } catch (Exception $e) {
            error_log('SchemaOrgDebt::collectDebtData error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Построение URL инструмента с проверкой на пустоту
     * 
     * @param object $instrument Объект инструмента
     * @return string URL инструмента
     */
    private function buildInstrumentUrl($instrument) {
        $supertypeId = (int)$this->getObjectProperty($instrument, 'Instrument_SUPERTYPE_ID', 0);
        $tradeCode = $this->safeString($this->getObjectProperty($instrument, 'Instrument_TRADE_CODE', ''));
        
        // Если trade_code пустой - возвращаем базовый URL без двойного слеша
        if (empty($tradeCode)) {
            $slug = $this->getInstrumentSlug($supertypeId);
            return $this->baseUrl . '/' . $slug . '/';
        }
        
        $slug = $this->getInstrumentSlug($supertypeId);
        return $this->baseUrl . '/' . $slug . '/' . $tradeCode . '/';
    }
    
    /**
     * Построение URL эмитента с проверкой на пустоту
     * 
     * @param object $emitent Объект эмитента
     * @return string URL эмитента
     */
    private function buildEmitentUrl($emitent) {
        $emitentUrl = $this->safeString($this->getObjectProperty($emitent, 'EMITENT_URL', ''));
        
        // Если emitent_url пустой - возвращаем только baseUrl
        if (empty($emitentUrl)) {
            return $this->baseUrl . '/';
        }
        
        return $this->baseUrl . '/' . $emitentUrl . '/';
    }
    
    /**
     * Получение slug инструмента по его типу
     * ВАЖНО: Работает только для debt типов (3, 5, 6)
     * Для equity типов (1, 2, 4) вернёт fallback со warning
     * 
     * @param int $supertypeId ID типа инструмента
     * @return string Slug для URL (bonds/eurobonds/mortgage-notes)
     */
    private function getInstrumentSlug($supertypeId) {
        if (!isset(self::$instrumentTypes[$supertypeId])) {
            // ВНИМАНИЕ: Если попал сюда equity тип (1,2,4) или неизвестный тип
            // это значит SchemaOrgDebt вызван для неправильного типа инструмента
            error_log("SchemaOrgDebt: Invalid supertypeId {$supertypeId} for debt instruments. Expected 3, 5, or 6. Using 'bonds' fallback.");
            return 'bonds';
        }
        
        return self::$instrumentTypes[$supertypeId]['slug'];
    }
    
    /**
     * Безопасное получение свойства объекта
     * Возвращает значение свойства или default если свойство не существует
     * 
     * @param object $object Объект
     * @param string $property Имя свойства
     * @param mixed $default Значение по умолчанию
     * @return mixed Значение свойства или default
     */
    private function getObjectProperty($object, $property, $default = null) {
        if (!is_object($object)) {
            return $default;
        }
        
        if (!property_exists($object, $property)) {
            return $default;
        }
        
        return $object->$property ?? $default;
    }
    
    /**
     * Получение данных сектора из БД с кэшированием
     * 
     * @param int $sector_id ID сектора
     * @return array|null Данные сектора или null
     */
    private function getSectorData($sector_id) {
        $cacheKey = "sector_{$sector_id}";
        
        // Проверка кэша
        if (isset(self::$sectorCache[$cacheKey])) {
            $cached = self::$sectorCache[$cacheKey];
            
            if ($cached === self::CACHE_NOT_FOUND) {
                return null;
            }
            
            // Проверка что кэш валиден и не истёк
            if (isset($cached['expires']) && isset($cached['data']) && $cached['expires'] > time()) {
                return $cached['data'];
            }
        }
        
        // Запрос к БД
        try {
            $stmt = $this->db->prepare('SELECT SECTOR_NAME, SECTOR_ECONOMIKI_URL FROM SECTOR_ECONOMIKI WHERE Id = ? LIMIT 1');
            
            if (!$stmt) {
                return null;
            }
            
            $stmt->bind_param('i', $sector_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            
            if ($row) {
                $sectorData = [
                    'name' => $this->safeString($row['SECTOR_NAME']),
                    'url' => $this->safeString($row['SECTOR_ECONOMIKI_URL'])
                ];
                
                // Сохранение в кэш
                self::$sectorCache[$cacheKey] = [
                    'data' => $sectorData,
                    'expires' => time() + self::CACHE_TTL
                ];
                
                return $sectorData;
            } else {
                // Сохранение NOT_FOUND в кэш
                self::$sectorCache[$cacheKey] = self::CACHE_NOT_FOUND;
                return null;
            }
            
        } catch (Exception $e) {
            error_log('SchemaOrgDebt::getSectorData error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Конвертация даты из формата дд.мм.гггг в ISO 8601 (гггг-мм-дд)
     * 
     * @param string $date Дата в формате дд.мм.гггг
     * @return string|null Дата в ISO формате или null
     */
    private function convertDateToISO($date) {
        if (!$this->isValidString($date)) {
            return null;
        }
        
        // Проверка формата дд.мм.гггг
        if (!preg_match('/^(\d{2})\.(\d{2})\.(\d{4})$/', $date, $matches)) {
            return null;
        }
        
        $day = $matches[1];
        $month = $matches[2];
        $year = $matches[3];
        
        // Валидация даты
        if (!checkdate((int)$month, (int)$day, (int)$year)) {
            return null;
        }
        
        // Возврат в ISO формате
        return $year . '-' . $month . '-' . $day;
    }
    
    /**
     * Конвертация MySQL timestamp в ISO 8601 формат с timezone
     * v1.2: Добавлен метод convertTimestampToISO() для ISO 8601 формата с timezone
     * 
     * @param string $timestamp Timestamp из БД (YYYY-MM-DD HH:MM:SS)
     * @return string|null ISO 8601 формат с timezone или null
     */
    private function convertTimestampToISO($timestamp) {
        if (!$this->isValidString($timestamp)) {
            return null;
        }
        
        try {
            // Создаём DateTime объект с московским timezone
            $dt = new DateTime($timestamp, new DateTimeZone('Europe/Moscow'));
            
            // Форматируем в ISO 8601 с timezone
            // Формат 'c': 2025-09-23T14:22:25+03:00
            return $dt->format('c');
            
        } catch (Exception $e) {
            error_log('SchemaOrgDebt: Failed to convert timestamp to ISO 8601: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Расчёт количества дней до указанной даты
     * 
     * @param string $date Дата в формате дд.мм.гггг
     * @return int|null Количество дней или null
     */
    private function calculateDaysToDate($date) {
        if (!$this->isValidString($date)) {
            return null;
        }
        
        // Конвертация в ISO для работы с DateTime
        $isoDate = $this->convertDateToISO($date);
        if ($isoDate === null) {
            return null;
        }
        
        try {
            $targetDate = new DateTime($isoDate);
            $currentDate = new DateTime();
            $interval = $currentDate->diff($targetDate);
            
            // Возвращаем положительное число для будущих дат, отрицательное для прошлых
            return $interval->invert ? -$interval->days : $interval->days;
        } catch (Exception $e) {
            return null;
        }
    }
    
    /**
     * Расчёт количества купонных выплат в год
     * 
     * @param int $couponPeriod Период купона в днях
     * @return int|null Количество выплат в год или null
     */
    private function calculateAnnualPayments($couponPeriod) {
        if (!$this->isValidNumber($couponPeriod) || $couponPeriod <= 0) {
            return null;
        }
        
        // 365 дней / период = количество выплат
        return (int)round(365 / $couponPeriod);
    }
    
    /**
     * Построение FinancialProduct объекта для Schema.org
     * Адаптировано из SchemaOrgEquity::buildFinancialProduct()
     * 
     * @param array $data Данные собранные в collectDebtData()
     * @return array Schema.org FinancialProduct объект
     */
    private function buildFinancialProduct($data) {
        $product = [];
        
        // Проверка обязательного URL инструмента
        if (!isset($data['urls']) || !is_array($data['urls']) || 
            !isset($data['urls']['instrument']) || !$this->isValidString($data['urls']['instrument'])) {
            error_log('SchemaOrgDebt: Missing or invalid instrument URL, cannot build FinancialProduct');
            return $product;
        }
        
        // Базовые поля
        $product['@type'] = 'FinancialProduct';
        
        // v1.2: Динамический выбор фрагмента @id в зависимости от типа инструмента
        $supertypeId = $data['instrument']['supertype_id'];
        $idFragment = '#product'; // Fallback на случай неизвестного типа
        
        if ($supertypeId === self::INSTRUMENT_TYPE_BONDS) {
            $idFragment = '#bond';
        } elseif ($supertypeId === self::INSTRUMENT_TYPE_EUROBONDS) {
            $idFragment = '#eurobond';
        } elseif ($supertypeId === self::INSTRUMENT_TYPE_MORTGAGE_NOTES) {
            $idFragment = '#mortgage-note';
        }
        
        $product['@id'] = $data['urls']['instrument'] . $idFragment;
        $product['additionalType'] = 'https://schema.org/Security';
        
        // Название (TYPE + CODE для облигаций используем ShortName + CODE)
        $nameParts = [];
        if (isset($data['instrument']['type']) && $this->isValidString($data['instrument']['type'])) {
            $nameParts[] = $data['instrument']['type'];
        }
        if (isset($data['instrument']['trade_code']) && $this->isValidString($data['instrument']['trade_code'])) {
            $nameParts[] = $data['instrument']['trade_code'];
        }
        if (isset($data['security']['short_name']) && $this->isValidString($data['security']['short_name'])) {
            $nameParts[] = $data['security']['short_name'];
        }
        
        if (count($nameParts) > 0) {
            $product['name'] = implode(' ', $nameParts);
        }
        
        // Идентификатор (торговый код)
        if (isset($data['instrument']['trade_code']) && $this->isValidString($data['instrument']['trade_code'])) {
            $product['identifier'] = $data['instrument']['trade_code'];
        }
        
        // ISIN код
        if (isset($data['instrument']['isin']) && $this->isValidString($data['instrument']['isin'])) {
            $product['isinCode'] = $data['instrument']['isin'];
        }
        
        // CFI код (международный стандарт классификации финансовых инструментов)
        if (isset($data['security']['cfi']) && $this->isValidString($data['security']['cfi'])) {
            $product['cfiCode'] = $data['security']['cfi'];
        }
        
        // Регистрационный номер
        if (isset($data['instrument']['registry_number']) && $this->isValidString($data['instrument']['registry_number'])) {
            $product['registryNumber'] = $data['instrument']['registry_number'];
        }
        
        // Уровень листинга
        if (isset($data['instrument']['list_section']) && $this->isValidString($data['instrument']['list_section'])) {
            $product['listingLevel'] = $data['instrument']['list_section'];
        }
        
        // Категория (Bond для облигаций)
        $product['category'] = 'Bond';
        
        if (isset($data['instrument']['type']) && $this->isValidString($data['instrument']['type'])) {
            $product['subCategory'] = $data['instrument']['type'];
        }
        
        // Описание
        $descParts = [];
        if (isset($data['instrument']['type']) && $this->isValidString($data['instrument']['type'])) {
            $descParts[] = $data['instrument']['type'];
        }
        if (isset($data['instrument']['trade_code']) && $this->isValidString($data['instrument']['trade_code'])) {
            $descParts[] = $data['instrument']['trade_code'];
        }
        if (isset($data['emitent']['short_name']) && $this->isValidString($data['emitent']['short_name'])) {
            $descParts[] = 'компании ' . $data['emitent']['short_name'];
        }
        if (isset($data['security']['class_name']) && $this->isValidString($data['security']['class_name'])) {
            $descParts[] = 'торгуется на Московской бирже в режиме ' . $data['security']['class_name'];
        }
        
        // Добавляем информацию о дате погашения в описание
        if (isset($data['security']['mat_date']) && $this->isValidString($data['security']['mat_date'])) {
            $descParts[] = 'с погашением ' . $data['security']['mat_date'];
        }
        
        if (count($descParts) > 1) {
            $product['description'] = implode(' ', $descParts);
        }
        
        // Альтернативные названия
        $alternateNames = [];
        if (isset($data['security']['sec_name']) && isset($data['security']['short_name']) && 
            $this->isValidString($data['security']['sec_name']) && 
            $this->isValidString($data['security']['short_name']) && 
            $data['security']['sec_name'] !== $data['security']['short_name']) {
            $alternateNames[] = $data['security']['sec_name'];
        }
        if (isset($data['security']['short_name']) && $this->isValidString($data['security']['short_name'])) {
            $alternateNames[] = $data['security']['short_name'];
        }
        if (count($alternateNames) > 0) {
            // array_values для сброса индексов после array_unique
            $product['alternateName'] = array_values(array_unique($alternateNames));
        }
        
        // URL страницы (обязательное поле, уже проверено в начале метода)
        $product['url'] = $data['urls']['instrument'];
        
        // Изображение (логотип эмитента)
        if (isset($data['emitent']['og_image']) && $this->isValidString($data['emitent']['og_image'])) {
            $product['image'] = [
                '@type' => 'ImageObject',
                'url' => $this->baseUrl . '/images/' . $data['emitent']['og_image'],
                'caption' => 'Логотип ' . (isset($data['emitent']['short_name']) ? $data['emitent']['short_name'] : 'эмитента')
            ];
        }
        
        // Бренд
        if (isset($data['emitent']['short_name']) && isset($data['urls']['emitent']) && 
            $this->isValidString($data['emitent']['short_name']) && 
            $this->isValidString($data['urls']['emitent'])) {
            $product['brand'] = [
                '@type' => 'Brand',
                'name' => $data['emitent']['short_name'],
                'url' => $data['urls']['emitent']
            ];
        }
        
        // Класс торговли (tradingClass для международной совместимости v1.1)
        if (isset($data['security']['class_name']) && $this->isValidString($data['security']['class_name'])) {
            $product['tradingClass'] = $data['security']['class_name'];
        }
        
        if (isset($data['security']['class_code']) && $this->isValidString($data['security']['class_code'])) {
            $product['classCode'] = $data['security']['class_code'];
        }
        
        // Номинальная стоимость
        if (isset($data['security']['face_value']) && $this->isValidNumber($data['security']['face_value'])) {
            $product['nominal'] = $data['security']['face_value'];
        }
        
        // Валюта (конвертируем SUR → RUB для Schema.org)
        if (isset($data['security']['currency_id']) && $this->isValidString($data['security']['currency_id'])) {
            $currency = $data['security']['currency_id'];
            // Конвертируем биржевой код SUR в стандартный RUB
            $product['priceCurrency'] = ($currency === 'SUR') ? 'RUB' : $currency;
        }
        
        // Размер лота
        if (isset($data['security']['lot_size']) && $this->isValidNumber($data['security']['lot_size'])) {
            $product['lotSize'] = $data['security']['lot_size'];
        }
        
        // Объём выпуска
        if (isset($data['security']['issue_size']) && $this->isValidNumber($data['security']['issue_size'])) {
            $product['issueVolume'] = $data['security']['issue_size'];
        }
        
        // Объём в обращении
        if (isset($data['security']['issue_size_placed']) && $this->isValidNumber($data['security']['issue_size_placed'])) {
            $product['circulatingVolume'] = $data['security']['issue_size_placed'];
        }
        
        // Статус торгов (конвертация в международные стандарты v1.1)
        if (isset($data['security']['status']) && $this->isValidString($data['security']['status'])) {
            $status = $data['security']['status'];
            // Конвертация биржевых статусов в международные стандарты
            if ($status === 'торгуется') {
                $product['status'] = 'Active';
            } elseif ($status === 'приостановлена') {
                $product['status'] = 'Suspended';
            } elseif ($status === 'не торгуется') {
                $product['status'] = 'Inactive';
            } else {
                // Для неизвестных статусов оставляем как есть
                $product['status'] = $status;
            }
        }
        
        // Шаг цены
        if (isset($data['security']['min_price_step']) && $this->isValidNumber($data['security']['min_price_step'])) {
            $product['priceStep'] = $data['security']['min_price_step'];
        }
        
        // Точность цены
        if (isset($data['security']['scale']) && $this->isValidNumber($data['security']['scale'])) {
            $product['pricePrecision'] = $data['security']['scale'];
        }
        
        // Базис котировок
        if (isset($data['security']['quote_basis']) && $this->isValidString($data['security']['quote_basis'])) {
            $product['quoteBasis'] = $data['security']['quote_basis'];
        }
        
        // Код расчётов
        if (isset($data['security']['settle_code']) && $this->isValidString($data['security']['settle_code'])) {
            $product['settlementCode'] = $data['security']['settle_code'];
        }
        
        // v1.2.1: Дата расчётов (ISO 8601 формат)
        if (isset($data['security']['settle_date']) && $this->isValidString($data['security']['settle_date'])) {
            $settleDateISO = $this->convertDateToISO($data['security']['settle_date']);
            if ($settleDateISO !== null) {
                $product['settlementDate'] = $settleDateISO;
            }
        }
        
        // v1.2: Дата последнего обновления (ISO 8601 формат с timezone)
        if (isset($data['security']['timestamp']) && $this->isValidString($data['security']['timestamp'])) {
            $isoTimestamp = $this->convertTimestampToISO($data['security']['timestamp']);
            if ($isoTimestamp !== null) {
                $product['dateModified'] = $isoTimestamp;
            }
        }
        
        // ОБЛИГАЦИОННЫЕ ПОЛЯ:
        
        // Дата погашения (maturityDate)
        if (isset($data['security']['mat_date']) && $this->isValidString($data['security']['mat_date'])) {
            $matDateISO = $this->convertDateToISO($data['security']['mat_date']);
            if ($matDateISO !== null) {
                $product['maturityDate'] = $matDateISO;
            }
        }
        
        // Размер купона (couponRate)
        if (isset($data['security']['coupon_value']) && $this->isValidNumber($data['security']['coupon_value'])) {
            $product['couponRate'] = $data['security']['coupon_value'];
        }
        
        // Период купона в днях (couponPeriodDays)
        if (isset($data['security']['coupon_period']) && $this->isValidNumber($data['security']['coupon_period'])) {
            $product['couponPeriodDays'] = $data['security']['coupon_period'];
            
            // Расчёт количества выплат в год (annualCouponPayments)
            $annualPayments = $this->calculateAnnualPayments($data['security']['coupon_period']);
            if ($annualPayments !== null) {
                $product['annualCouponPayments'] = $annualPayments;
            }
        }
        
        // Дата следующего купона (nextCouponDate)
        if (isset($data['security']['next_coupon']) && $this->isValidString($data['security']['next_coupon'])) {
            $nextCouponISO = $this->convertDateToISO($data['security']['next_coupon']);
            if ($nextCouponISO !== null) {
                $product['nextCouponDate'] = $nextCouponISO;
            }
        }
        
        // НКД - накопленный купонный доход (accruedInterest)
        if (isset($data['security']['accrued_int']) && $this->isValidNumber($data['security']['accrued_int'])) {
            $product['accruedInterest'] = $data['security']['accrued_int'];
        }
        
        // Дата оферты (buyBackDate)
        if (isset($data['security']['buyback_date']) && $this->isValidString($data['security']['buyback_date'])) {
            $buybackDateISO = $this->convertDateToISO($data['security']['buyback_date']);
            if ($buybackDateISO !== null) {
                $product['buyBackDate'] = $buybackDateISO;
            }
        }
        
        // Цена оферты (buyBackPrice)
        if (isset($data['security']['buyback_price']) && $this->isValidNumber($data['security']['buyback_price'])) {
            $product['buyBackPrice'] = $data['security']['buyback_price'];
        }
        
        // Международные идентификаторы
        if (isset($data['security']['bloomberg_id']) && $this->isValidString($data['security']['bloomberg_id'])) {
            $product['bloombergIdentifier'] = $data['security']['bloomberg_id'];
        }
        
        if (isset($data['security']['sedol']) && $this->isValidString($data['security']['sedol'])) {
            $product['sedol'] = $data['security']['sedol'];
        }
        
        if (isset($data['security']['ric']) && $this->isValidString($data['security']['ric'])) {
            $product['ric'] = $data['security']['ric'];
        }
        
        if (isset($data['security']['cusip']) && $this->isValidString($data['security']['cusip'])) {
            $product['cusip'] = $data['security']['cusip'];
        }
        
        // Место торгов (только для инструментов торгующихся на MOEX)
        // Проверяем по ClassCode начинающемуся с 'TQ' или статусу торгов
        if (isset($data['security']['class_code']) && $this->isValidString($data['security']['class_code'])) {
            $classCode = $data['security']['class_code'];
            $status = isset($data['security']['status']) ? $data['security']['status'] : '';
            
            // Проверка начинается ли ClassCode с 'TQ' или статус = 'торгуется' (до конвертации)
            if (substr($classCode, 0, 2) === 'TQ' || $status === 'торгуется') {
                $product['availableAtOrFrom'] = [
                    '@type' => 'Place',
                    'name' => 'Московская Биржа',
                    'url' => 'https://www.moex.com',
                    'identifier' => 'MOEX'
                ];
                
                // Область обслуживания (только для MOEX инструментов)
                $product['areaServed'] = [
                    '@type' => 'Country',
                    'name' => 'Россия',
                    'identifier' => 'RU'
                ];
            }
        }
        
        // Эмитент (issuer)
        // Проверка обязательного URL эмитента
        if (!isset($data['urls']['emitent']) || !$this->isValidString($data['urls']['emitent'])) {
            error_log('SchemaOrgDebt: Missing emitent URL for issuer');
            // Не добавляем issuer если нет URL
        } else {
            $issuer = [
                '@type' => 'Organization',
                '@id' => $data['urls']['emitent'] . '#organization'
            ];
            
            if (isset($data['emitent']['short_name']) && $this->isValidString($data['emitent']['short_name'])) {
                $issuer['name'] = $data['emitent']['short_name'];
            }
            
            if (isset($data['emitent']['full_name']) && $this->isValidString($data['emitent']['full_name'])) {
                $issuer['legalName'] = $data['emitent']['full_name'];
            }
            
            if (isset($data['emitent']['inn']) && $this->isValidString($data['emitent']['inn'])) {
                $issuer['taxID'] = $data['emitent']['inn'];
            }
            
            if (isset($data['emitent']['lei']) && $this->isValidString($data['emitent']['lei'])) {
                $issuer['leiCode'] = $data['emitent']['lei'];
            }
            
            if (isset($data['emitent']['founding_date']) && $this->isValidString($data['emitent']['founding_date'])) {
                $issuer['foundingDate'] = $data['emitent']['founding_date'];
            }
            
            if (isset($data['emitent']['id']) && $data['emitent']['id'] > 0) {
                $issuer['identifier'] = (string)$data['emitent']['id'];
            }
            
            $issuer['url'] = $data['urls']['emitent'];
            
            if (isset($data['emitent']['og_image']) && $this->isValidString($data['emitent']['og_image'])) {
                $logoUrl = $this->baseUrl . '/images/' . $data['emitent']['og_image'];
                $issuer['logo'] = $logoUrl;
                $issuer['image'] = $logoUrl;
            }
            
            // Описание эмитента с ucfirst
            if (isset($data['emitent']['txt_short']) && $this->isValidString($data['emitent']['txt_short'])) {
                // Capitalize первую букву description для правильного оформления
                $issuer['description'] = ucfirst($data['emitent']['txt_short']);
            }
            
            // Ссылки sameAs для эмитента
            $sameAs = [];
            if (isset($data['emitent']['disclosure_page']) && $this->isValidString($data['emitent']['disclosure_page'])) {
                $sameAs[] = $data['emitent']['disclosure_page'];
            }
            if (isset($data['emitent']['disclosure_rf']) && $this->isValidString($data['emitent']['disclosure_rf'])) {
                $sameAs[] = $data['emitent']['disclosure_rf'];
            }
            if (isset($data['emitent']['wiki']) && $this->isValidString($data['emitent']['wiki'])) {
                $sameAs[] = $data['emitent']['wiki'];
            }
            if (isset($data['emitent']['smart_lab']) && $this->isValidString($data['emitent']['smart_lab'])) {
                $sameAs[] = $data['emitent']['smart_lab'];
            }
            if (isset($data['emitent']['dohod']) && $this->isValidString($data['emitent']['dohod'])) {
                $sameAs[] = $data['emitent']['dohod'];
            }
            if (isset($data['emitent']['moex']) && $this->isValidString($data['emitent']['moex'])) {
                $sameAs[] = $data['emitent']['moex'];
            }
            
            if (count($sameAs) > 0) {
                $issuer['sameAs'] = $sameAs;
            }
            
            $product['issuer'] = $issuer;
        }
        
        // Действие (potentialAction)
        if (isset($data['instrument']['trade_code']) && isset($data['emitent']['short_name']) && 
            $this->isValidString($data['instrument']['trade_code']) && 
            $this->isValidString($data['emitent']['short_name'])) {
            $instrumentType = isset($data['instrument']['type']) ? $data['instrument']['type'] : 'облигации';
            $product['potentialAction'] = [
                '@type' => 'ViewAction',
                'target' => $data['urls']['instrument'],
                'name' => 'Анализ ' . $data['instrument']['trade_code'],
                'description' => 'Полная аналитика по ' . 
                    $instrumentType . ' ' . 
                    $data['instrument']['trade_code'] . 
                    ' от компании ' . $data['emitent']['short_name']
            ];
        }
        
        return $product;
    }
}