<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС SCHEMA.ORG РАЗМЕТКИ ДЛЯ КАТАЛОГА ОБЛИГАЦИЙ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.0 - INITIAL RELEASE (SIMPLE VERSION)
 * 
 * CHANGELOG v1.0 (2025-11-02):
 *   ✨ NEW: Простой класс для генерации Schema.org разметки каталога облигаций
 *   ✨ NEW: Метод loadCatalogDataFromDB() для загрузки данных из БД
 *   ✨ NEW: Группировка по уровням листинга
 *   ✅ Подсчёт количества облигаций в каждом уровне листинга
 *   ✅ SQL запрос с GROUP BY и COUNT
 *   ✅ Полная валидация данных
 * 
 * НАЗНАЧЕНИЕ:
 *   - Генерация Schema.org разметки для страницы /bonds/
 *   - CollectionPage с группировкой облигаций по уровням листинга
 *   - Простая структура (как SectorsCatalog, НЕ как StocksCatalog)
 *   - Оптимизировано для PAGE cache (генерация 1 раз/месяц)
 * 
 * СТРУКТУРА JSON-LD:
 *   @graph [
 *     BreadcrumbList - навигационные крошки
 *     CollectionPage - страница каталога
 *       name: "Каталог облигаций"
 *       mainEntity: ItemList - список уровней листинга
 *         ListItem - каждый уровень с количеством облигаций
 *   ]
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Размер JSON: ~3-5 KB (3 уровня листинга)
 *   - С gzip: ~1-2 KB
 *   - Генерация: 1 раз/месяц (PAGE cache)
 *   - Из кэша: 0.001 сек
 * 
 * SQL ЗАПРОС:
 *   SELECT 
 *     Instrument_LIST_SECTION_ID,
 *     Instrument_LIST_SECTION,
 *     COUNT(*) as bond_count
 *   FROM Instrument
 *   WHERE Instrument_SUPERTYPE_ID = 6
 *   GROUP BY Instrument_LIST_SECTION_ID, Instrument_LIST_SECTION
 *   ORDER BY Instrument_LIST_SECTION_ID
 * 
 * ОСНОВАНО НА:
 *   - SchemaOrgBase v1.1
 *   - SchemaOrgSectorsCatalog v1.0 (простая структура)
 * 
 * @version 1.0
 * @date 2025-11-02
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgBondsCatalog extends SchemaOrgBase {
    
    /**
     * ГЛАВНЫЙ МЕТОД - Построение JSON-LD разметки для каталога облигаций
     * Вызывается из BreadcrumbBuilder::getJsonLdEnhanced()
     * 
     * @param array $page_data Данные страницы [0 => 'MENU', 1 => объект MENU]
     * @param array $breadcrumbs Навигационные крошки из BreadcrumbBuilder
     * @return string JSON-LD разметка или пустая строка при ошибке
     */
    public function buildEnhancedJsonLd($page_data, $breadcrumbs) {
        // Проверка формата данных
        if (empty($page_data) || !is_array($page_data)) {
            error_log('SchemaOrgBondsCatalog: Invalid page_data format');
            return '';
        }
        
        // Загружаем данные каталога из БД
        $catalog = $this->loadCatalogDataFromDB();
        
        // Проверка что данные загружены
        if (empty($catalog) || !is_object($catalog)) {
            error_log('SchemaOrgBondsCatalog: Failed to load catalog data from DB');
            return '';
        }
        
        // Валидация критичных полей
        if (!property_exists($catalog, 'listing_levels') || !is_array($catalog->listing_levels)) {
            error_log('SchemaOrgBondsCatalog: Missing or invalid listing_levels data after DB load');
            return '';
        }
        
        // Построение @graph
        $graph = [];
        
        // 1. BreadcrumbList из навигации
        if (!empty($breadcrumbs) && is_array($breadcrumbs)) {
            $breadcrumbList = $this->buildBreadcrumbList($breadcrumbs);
            if (!empty($breadcrumbList)) {
                $graph[] = $breadcrumbList;
            }
        }
        
        // 2. CollectionPage со списком уровней листинга
        $collectionPage = $this->buildCatalogCollectionPage($catalog);
        
        if (!empty($collectionPage)) {
            $graph[] = $collectionPage;
        }
        
        // Если нет данных для вывода - возвращаем пустую строку
        if (empty($graph)) {
            error_log('SchemaOrgBondsCatalog: Empty graph, nothing to output');
            return '';
        }
        
        // Финальная обёртка JSON-LD
        return $this->buildJsonLd($graph);
    }
    
    /**
     * МЕТОД ЗАГРУЗКИ ДАННЫХ - Загрузка каталога облигаций из БД
     * 
     * Загружает все облигации (type 6) с группировкой по уровням листинга
     * 
     * @return object|null Объект каталога или null при ошибке
     */
    private function loadCatalogDataFromDB() {
        // Проверка подключения к БД
        if (!$this->db instanceof mysqli) {
            error_log('SchemaOrgBondsCatalog: Invalid database connection');
            return null;
        }
        
        // SQL запрос: получаем облигации с группировкой по уровням листинга
        $sql = "SELECT 
                    i.Instrument_LIST_SECTION_ID,
                    i.Instrument_LIST_SECTION,
                    COUNT(*) as bond_count
                FROM Instrument i
                WHERE i.Instrument_SUPERTYPE_ID = 6
                GROUP BY i.Instrument_LIST_SECTION_ID, i.Instrument_LIST_SECTION
                ORDER BY i.Instrument_LIST_SECTION_ID";
        
        // Выполнение запроса
        $result = $this->db->query($sql);
        
        if (!$result) {
            error_log('SchemaOrgBondsCatalog: SQL query failed - ' . $this->db->error);
            return null;
        }
        
        // Инициализация структуры данных
        $listing_levels = [];
        $total_bonds = 0;
        
        // Обработка результатов
        while ($row = $result->fetch_assoc()) {
            // Валидация обязательных полей
            if (empty($row['Instrument_LIST_SECTION_ID'])) {
                continue;
            }
            
            $listing_id = (int)$row['Instrument_LIST_SECTION_ID'];
            $bond_count = isset($row['bond_count']) ? (int)$row['bond_count'] : 0;
            
            // Формирование данных уровня листинга
            $listing_levels[] = [
                'id' => $listing_id,
                'name' => !empty($row['Instrument_LIST_SECTION']) 
                    ? trim($row['Instrument_LIST_SECTION']) 
                    : 'Уровень листинга ' . $listing_id,
                'bond_count' => $bond_count
            ];
            
            $total_bonds += $bond_count;
        }
        
        // Освобождение результата
        $result->free();
        
        // Проверка что есть данные
        if (empty($listing_levels)) {
            error_log('SchemaOrgBondsCatalog: No bonds found in database');
            return null;
        }
        
        // Формирование объекта каталога
        $catalog = new stdClass();
        $catalog->listing_levels = $listing_levels;
        $catalog->total_listing_levels = count($listing_levels);
        $catalog->total_bonds = $total_bonds;
        
        return $catalog;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ COLLECTIONPAGE - Построение Schema.org CollectionPage
     * 
     * Создаёт CollectionPage с mainEntity содержащим ItemList уровней листинга
     * 
     * @param object $catalog Объект каталога с данными уровней листинга
     * @return array Schema.org CollectionPage объект или пустой массив
     */
    private function buildCatalogCollectionPage($catalog) {
        // Валидация входных данных
        if (empty($catalog) || !is_object($catalog)) {
            return [];
        }
        
        if (!property_exists($catalog, 'listing_levels') || !is_array($catalog->listing_levels)) {
            return [];
        }
        
        if (empty($catalog->listing_levels)) {
            return [];
        }
        
        // Базовая структура CollectionPage
        $page = [
            '@type' => 'CollectionPage',
            'name' => 'Каталог облигаций Московской биржи',
            'description' => 'Полный каталог облигаций российских эмитентов, структурированный по уровням листинга',
            'url' => $this->baseUrl . '/bonds/'
        ];
        
        // Добавление общего количества облигаций
        if (property_exists($catalog, 'total_bonds') && $catalog->total_bonds > 0) {
            $page['numberOfItems'] = $catalog->total_bonds;
        }
        
        // Построение mainEntity: ItemList уровней листинга
        $itemList = $this->buildListingLevelsItemList($catalog->listing_levels);
        
        if (!empty($itemList)) {
            $page['mainEntity'] = $itemList;
        }
        
        return $page;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ ITEMLIST - Построение ItemList уровней листинга
     * 
     * @param array $listing_levels Массив уровней листинга из БД
     * @return array Schema.org ItemList объект или пустой массив
     */
    private function buildListingLevelsItemList($listing_levels) {
        // Валидация входных данных
        if (!is_array($listing_levels) || empty($listing_levels)) {
            return [];
        }
        
        $itemList = [
            '@type' => 'ItemList',
            'name' => 'Уровни листинга',
            'description' => 'Облигации структурированы по уровням листинга Московской биржи',
            'numberOfItems' => count($listing_levels),
            'itemListElement' => []
        ];
        
        $position = 1;
        
        // Перебор всех уровней листинга
        foreach ($listing_levels as $listingData) {
            // Валидация структуры уровня листинга
            if (!is_array($listingData) || !isset($listingData['name'])) {
                continue;
            }
            
            // Построение ListItem для уровня листинга
            $listItem = $this->buildListingLevelListItem($listingData, $position);
            
            if (!empty($listItem)) {
                $itemList['itemListElement'][] = $listItem;
                $position++;
            }
        }
        
        // Если нет элементов - возвращаем пустой массив
        if (empty($itemList['itemListElement'])) {
            return [];
        }
        
        return $itemList;
    }
    
    /**
     * МЕТОД ПОСТРОЕНИЯ LISTITEM - Построение ListItem для одного уровня листинга
     * 
     * @param array $listingData Данные уровня листинга
     * @param int $position Позиция в списке
     * @return array Schema.org ListItem объект или пустой массив
     */
    private function buildListingLevelListItem($listingData, $position) {
        // Валидация входных данных
        if (!is_array($listingData) || !isset($listingData['name'])) {
            return [];
        }
        
        $listItem = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => $this->safeString($listingData['name'])
        ];
        
        // Построение описания с количеством облигаций
        $description = '';
        
        if (isset($listingData['bond_count']) && $listingData['bond_count'] > 0) {
            $count = (int)$listingData['bond_count'];
            $description = $count . ' ' . $this->pluralizeBonds($count);
        } else {
            $description = 'Уровень листинга облигаций';
        }
        
        $listItem['description'] = $description;
        
        return $listItem;
    }
    
    /**
     * ВСПОМОГАТЕЛЬНЫЙ МЕТОД - Плюрализация слова "облигация"
     * 
     * @param int $count Количество облигаций
     * @return string Правильная форма слова
     */
    private function pluralizeBonds($count) {
        $count = abs($count) % 100;
        $leastSignificant = $count % 10;
        
        if ($count >= 11 && $count <= 19) {
            return 'облигаций';
        }
        
        if ($leastSignificant == 1) {
            return 'облигация';
        }
        
        if ($leastSignificant >= 2 && $leastSignificant <= 4) {
            return 'облигации';
        }
        
        return 'облигаций';
    }
}