<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * БАЗОВЫЙ КЛАСС ДЛЯ ВСЕХ SCHEMA.ORG КЛАССОВ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.1 - ОБНОВЛЕНИЕ: safeInteger/safeDecimal возвращают null
 * 
 * CHANGELOG v1.1 (2025-10-12):
 *   - ИЗМЕНЕНО: safeInteger() возвращает null вместо 0
 *   - ИЗМЕНЕНО: safeDecimal() возвращает null вместо 0.0
 *   - ПРИЧИНА: Более корректная семантика "нет значения" = null
 * 
 * НАЗНАЧЕНИЕ:
 *   - Общие helper методы для безопасной работы с данными
 *   - Построение базовых Schema.org объектов (BreadcrumbList, Organization)
 *   - Финальная обёртка JSON-LD
 *   - Работа с кэшем (опционально)
 * 
 * НАСЛЕДНИКИ:
 *   - SchemaOrgEquity - для долевых инструментов (акции, расписки, паи)
 *   - SchemaOrgDebt - для долговых инструментов (облигации)
 *   - SchemaOrgEmitent - для страниц эмитентов
 *   - SchemaOrgSector - для страниц секторов
 * 
 * @version 1.1
 * @date 2025-10-12
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * ═══════════════════════════════════════════════════════════════════════════
 */

class SchemaOrgBase {
    
    protected $db;
    protected $baseUrl;
    
    /**
     * Конструктор базового класса
     * 
     * @param mysqli $db Подключение к базе данных
     * @throws InvalidArgumentException Если передан невалидный тип БД
     */
    public function __construct($db) {
        // Проверка типа подключения к БД
        if (!($db instanceof mysqli)) {
            throw new InvalidArgumentException('SchemaOrgBase: Database connection must be instance of mysqli');
        }
        
        $this->db = $db;
        
        if (!defined('SITE_URL') || empty(SITE_URL)) {
            $this->baseUrl = 'https://deepmax.ru';
        } else {
            $this->baseUrl = rtrim(SITE_URL, '/');
        }
    }
    
    /**
     * Безопасное преобразование строки для Schema.org
     * Удаляет HTML теги, обрезает пробелы
     * 
     * @param mixed $value Входное значение
     * @return string Безопасная строка или пустая строка
     */
    protected function safeString($value) {
        if ($value === null || $value === '') {
            return '';
        }
        
        // Проверка типа - массивы и объекты не конвертируем
        if (is_array($value) || (is_object($value) && !method_exists($value, '__toString'))) {
            return '';
        }
        
        // Преобразование в строку
        $value = (string)$value;
        
        // Удаление HTML тегов
        $value = strip_tags($value);
        
        // Обрезка пробелов
        $value = trim($value);
        
        return $value;
    }
    
    /**
     * Безопасное преобразование целого числа
     * 
     * @param mixed $value Входное значение
     * @return int|null Целое число или null если пусто
     */
    protected function safeInteger($value) {
        if ($value === null || $value === '') {
            return null;
        }
        
        return (int)$value;
    }
    
    /**
     * Безопасное преобразование десятичного числа
     * 
     * @param mixed $value Входное значение
     * @return float|null Десятичное число или null если пусто
     */
    protected function safeDecimal($value) {
        if ($value === null || $value === '') {
            return null;
        }
        
        return (float)$value;
    }
    
    /**
     * Построение BreadcrumbList объекта для Schema.org
     * 
     * @param array $breadcrumbs Массив навигационных крошек из BreadcrumbBuilder
     * @return array Schema.org BreadcrumbList объект
     */
    protected function buildBreadcrumbList($breadcrumbs) {
        $items = [];
        $position = 1;
        $total = count($breadcrumbs);
        
        foreach ($breadcrumbs as $crumb) {
            // Проверка наличия обязательных ключей и непустоты значений
            if (!isset($crumb['name']) || !isset($crumb['url']) || 
                empty(trim($crumb['name'])) || empty(trim($crumb['url']))) {
                continue; // Пропускаем некорректную крошку
            }
            
            $item = [
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $this->safeString($crumb['name'])
            ];
            
            // Все крошки кроме последней имеют item (URL)
            if ($position < $total) {
                $item['item'] = $crumb['url'];
            }
            
            $items[] = $item;
            $position++;
        }
        
        return [
            '@type' => 'BreadcrumbList',
            'itemListElement' => $items
        ];
    }
    
    /**
     * Построение базового Organization объекта
     * Используется в Equity, Debt классах как issuer
     * 
     * @param object $emitent Объект эмитента из БД
     * @param array $urls Дополнительные URL для sameAs
     * @return array Schema.org Organization объект
     */
    protected function buildOrganization($emitent, $urls = []) {
        // Проверка что передан объект
        if (!is_object($emitent)) {
            return []; // Возвращаем пустой массив если не объект
        }
        
        $org = [
            '@type' => 'Organization'
        ];
        
        // Проверка каждого свойства перед использованием
        if (property_exists($emitent, 'EMITENT_SHORT_NAME') && !empty($emitent->EMITENT_SHORT_NAME)) {
            $org['name'] = $this->safeString($emitent->EMITENT_SHORT_NAME);
        }
        
        // Полное юридическое название
        if (property_exists($emitent, 'EMITENT_FULL_NAME') && !empty($emitent->EMITENT_FULL_NAME)) {
            $org['legalName'] = $this->safeString($emitent->EMITENT_FULL_NAME);
        }
        
        // ИНН (taxID)
        if (property_exists($emitent, 'INN') && !empty($emitent->INN)) {
            $org['taxID'] = $this->safeString($emitent->INN);
        }
        
        // LEI код (если есть)
        if (property_exists($emitent, 'LEI') && !empty($emitent->LEI)) {
            $org['leiCode'] = $this->safeString($emitent->LEI);
        }
        
        // URL эмитента
        if (property_exists($emitent, 'EMITENT_URL') && !empty($emitent->EMITENT_URL)) {
            $org['url'] = $this->baseUrl . '/' . $this->safeString($emitent->EMITENT_URL) . '/';
        }
        
        // Дополнительные URL (sameAs)
        if (!empty($urls)) {
            $org['sameAs'] = array_values(array_filter($urls));
        }
        
        return $org;
    }
    
    /**
     * Финальная обёртка для JSON-LD разметки
     * Создаёт <script type="application/ld+json"> с @context и @graph
     * 
     * @param array $graph Массив Schema.org объектов
     * @return string HTML тег <script> с JSON-LD или пустая строка при ошибке
     */
    protected function buildJsonLd($graph) {
        if (empty($graph)) {
            return '';
        }
        
        $jsonLd = [
            '@context' => 'https://schema.org',
            '@graph' => $graph
        ];
        
        $json = json_encode(
            $jsonLd, 
            JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT | JSON_HEX_TAG
        );
        
        if ($json === false) {
            error_log('SchemaOrgBase: JSON encoding failed - ' . json_last_error_msg());
            return '';
        }
        
        return '<script type="application/ld+json">' . "\n" . $json . "\n" . '</script>';
    }
    
    /**
     * Проверка что строка не пустая
     * 
     * @param mixed $value Значение для проверки
     * @return bool true если строка валидна и не пуста
     */
    protected function isValidString($value) {
        // Оптимальный порядок: сначала быстрая проверка типа
        return is_string($value) && $value !== '' && trim($value) !== '';
    }
    
    /**
     * Проверка что число валидно и больше нуля
     * 
     * @param mixed $value Значение для проверки
     * @return bool true если число валидно и > 0
     */
    protected function isValidNumber($value) {
        return $value !== null && $value !== '' && is_numeric($value) && $value > 0;
    }
}