<?php
/**
 * DeepMax Debug System v4.4.1 CSP-FIX
 * Главный класс профилирования и отладки
 * 
 * @package DeepMax
 * @version 4.4.1
 * @date 2025-10-18
 * @status PRODUCTION READY - CSP COMPATIBLE
 * 
 * ИЗМЕНЕНИЯ v4.4.1 CSP-FIX:
 *   🔥 FIX: Убран inline <script> из renderPanel()
 *   🔥 NEW: Использование внешнего JS файла /js/deepmax-debug.js
 *   ✅ CSP: Полная совместимость без nonce
 *   ✅ IE11: JavaScript совместим с IE11
 *   ✅ MEMORY: Cleanup handlers в JS файле
 * 
 * ИЗМЕНЕНИЯ v4.4.0 HOOKS:
 *   🔥 NEW: Метод getPanelHTML() для использования с Cache хуками
 *   🔥 NEW: Флаг $panelInsertedViaHook для предотвращения дублирования
 *   🔥 MOD: shutdown() НЕ вставляет панель если она вставлена через хук
 *   ✅ ARCH: Поддержка Event System из cache.php v3.3.0
 *   ✅ SAFE: Защита от двойного вывода панели
 */

class DeepMaxDebug {
    
    // ============================================
    // КОНСТАНТЫ
    // ============================================
    
    const VERSION = '4.4.1';
    const LOG_FILE_PREFIX = 'profile_';
    const LOG_DATE_FORMAT = 'Y-m-d';
    const LOG_TIMESTAMP_FORMAT = 'Y-m-d H:i:s';
    const MAX_CHECKPOINTS = 1000;
    const MAX_LOG_MESSAGE_LENGTH = 5000;
    const LOG_CLEANUP_PROBABILITY = 100;
    const MAX_HTML_SIZE = 52428800; // 50 MB
    const MAX_BUFFER_CLOSE_ITERATIONS = 100;
    
    // ============================================
    // СТАТИЧЕСКИЕ СВОЙСТВА
    // ============================================
    
    private static $enabled = false;
    private static $initialized = false;
    private static $config = [];
    private static $startTime = 0;
    private static $startMemory = 0;
    private static $checkpoints = [];
    private static $logs = [];
    private static $errors = [];
    private static $sqlQueries = [];
    private static $handlersRegistered = false;
    private static $cspNonce = null;
    private static $shutdownExecuted = false;
    
    /**
     * v4.4.0 HOOKS: Флаг что панель уже вставлена через Cache хук
     * 
     * Используется для предотвращения двойного вывода панели:
     * - Если панель вставлена через Cache::addBeforeOutputHook() → true
     * - shutdown() проверяет этот флаг и НЕ вставляет панель повторно
     * 
     * @var bool
     */
    private static $panelInsertedViaHook = false;
    
    // ============================================
    // ПУБЛИЧНЫЕ МЕТОДЫ
    // ============================================
    
    public static function init($config = []) {
        if (self::$initialized) {
            return;
        }
        
        self::$initialized = true;
        self::$startTime = microtime(true);
        self::$startMemory = memory_get_usage();
        
        $fileConfig = [];
        if (file_exists(__DIR__ . '/../../cache.php')) {
            require_once(__DIR__ . '/../../cache.php');
            global $debug_config;
            if (isset($debug_config) && is_array($debug_config)) {
                $fileConfig = $debug_config;
            }
        }
        
        self::$config = array_merge([
            'enabled' => true,
            'allowed_ips' => ['127.0.0.1', '::1'],
            'allow_cli' => true,
            'trust_proxy' => false,
            'show_panel' => true,
            'panel_expanded' => false,
            'log_to_file' => true,
            'log_dir' => __DIR__ . '/../../logs/debug/',
            'slow_request_threshold' => 1.0,
            'slow_sql_threshold' => 0.1,
            'filter_passwords' => true,
            'max_log_age_days' => 30,
            'capture_errors' => true,
            'capture_exceptions' => true,
        ], $fileConfig, $config);
        
        if (!self::$config['enabled']) {
            return;
        }
        
        if (!self::isAllowedIP()) {
            return;
        }
        
        self::$enabled = true;
        self::registerHandlers();
        self::mark('System initialized');
        self::log('DeepMax Debug System v' . self::VERSION . ' активирована', 'info');
    }
    
    /**
     * v4.4.0 HOOKS: Получение HTML панели для вставки через Cache хук
     * 
     * Этот метод используется в Cache::addBeforeOutputHook() для вставки
     * Debug панели в HTML контент перед сохранением в кэш.
     * 
     * Устанавливает флаг $panelInsertedViaHook = true, чтобы shutdown()
     * не пытался вставить панель повторно.
     * 
     * @return string HTML код Debug панели или пустая строка
     */
    public static function getPanelHTML(): string
    {
        if (!self::$enabled) {
            return '';
        }
        
        if (!self::$config['show_panel']) {
            return '';
        }
        
        // Устанавливаем флаг что панель вставляется через хук
        self::$panelInsertedViaHook = true;
        
        // Генерируем финальный чекпоинт
        self::mark('Panel generated via hook');
        
        return self::renderPanel();
    }
    
    public static function mark($label) {
        if (!self::$enabled) {
            return;
        }
        
        if (count(self::$checkpoints) >= self::MAX_CHECKPOINTS) {
            if (count(self::$checkpoints) === self::MAX_CHECKPOINTS) {
                self::log('Достигнут лимит чекпоинтов (' . self::MAX_CHECKPOINTS . '), новые не добавляются', 'warning');
            }
            return;
        }
        
        $time = microtime(true) - self::$startTime;
        $memory = memory_get_usage();
        
        self::$checkpoints[] = [
            'label' => $label,
            'time' => $time,
            'memory' => $memory,
            'timestamp' => microtime(true),
        ];
    }
    
    public static function log($message, $level = 'info') {
        if (!self::$enabled) {
            return;
        }
        
        $allowedLevels = ['info', 'warning', 'error'];
        if (!in_array($level, $allowedLevels, true)) {
            $level = 'info';
        }
        
        if (is_string($message) && strlen($message) > self::MAX_LOG_MESSAGE_LENGTH) {
            $message = substr($message, 0, self::MAX_LOG_MESSAGE_LENGTH) . '... [обрезано]';
        }
        
        self::$logs[] = [
            'message' => $message,
            'level' => $level,
            'time' => microtime(true) - self::$startTime,
            'memory' => memory_get_usage(),
            'timestamp' => microtime(true),
        ];
    }
    
    public static function logSQL($query, $params = [], $time = 0) {
        if (!self::$enabled) {
            return;
        }
        
        if (self::$config['filter_passwords']) {
            $params = self::filterPasswords($params);
        }
        
        self::$sqlQueries[] = [
            'query' => $query,
            'params' => $params,
            'time' => $time,
            'timestamp' => microtime(true),
        ];
        
        $threshold = self::$config['slow_sql_threshold'] * 1000;
        if ($time > $threshold) {
            self::log("Медленный SQL запрос ({$time}ms): " . substr($query, 0, 100), 'warning');
        }
    }
    
    // ============================================
    // ОБРАБОТЧИКИ ОШИБОК
    // ============================================
    
    public static function errorHandler($errno, $errstr, $errfile, $errline) {
        if (!self::$enabled) {
            return false;
        }
        
        if (!(error_reporting() & $errno)) {
            return false;
        }
        
        $errorTypes = [
            E_ERROR => 'ERROR',
            E_WARNING => 'WARNING',
            E_PARSE => 'PARSE ERROR',
            E_NOTICE => 'NOTICE',
            E_CORE_ERROR => 'CORE ERROR',
            E_CORE_WARNING => 'CORE WARNING',
            E_COMPILE_ERROR => 'COMPILE ERROR',
            E_COMPILE_WARNING => 'COMPILE WARNING',
            E_USER_ERROR => 'USER ERROR',
            E_USER_WARNING => 'USER WARNING',
            E_USER_NOTICE => 'USER NOTICE',
            E_STRICT => 'STRICT',
            E_RECOVERABLE_ERROR => 'RECOVERABLE ERROR',
            E_DEPRECATED => 'DEPRECATED',
            E_USER_DEPRECATED => 'USER DEPRECATED',
        ];
        
        $errorType = isset($errorTypes[$errno]) ? $errorTypes[$errno] : 'UNKNOWN';
        
        self::$errors[] = [
            'type' => $errorType,
            'message' => $errstr,
            'file' => $errfile,
            'line' => $errline,
            'errno' => $errno,
            'timestamp' => microtime(true),
        ];
        
        $logLevel = in_array($errno, [E_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR]) ? 'error' : 'warning';
        self::log("PHP {$errorType}: {$errstr} in {$errfile}:{$errline}", $logLevel);
        
        return false;
    }
    
    public static function exceptionHandler($exception) {
        if (!self::$enabled) {
            return;
        }
        
        self::$errors[] = [
            'type' => 'EXCEPTION',
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'class' => get_class($exception),
            'timestamp' => microtime(true),
        ];
        
        self::log(
            'EXCEPTION [' . get_class($exception) . ']: ' . 
            $exception->getMessage() . ' in ' . 
            $exception->getFile() . ':' . $exception->getLine(),
            'error'
        );
    }
    
    /**
     * v4.4.0 HOOKS: Shutdown с поддержкой Cache хуков
     * 
     * Если панель уже вставлена через Cache хук ($panelInsertedViaHook = true),
     * то НЕ пытаемся вставить её повторно через processBufferAndOutputPanel().
     */
    public static function shutdown() {
        if (!self::$enabled) {
            return;
        }
        
        if (self::$shutdownExecuted) {
            return;
        }
        self::$shutdownExecuted = true;
        
        try {
            $error = error_get_last();
            if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
                self::errorHandler($error['type'], $error['message'], $error['file'], $error['line']);
            }
            
            self::mark('Shutdown');
            
            $totalTime = microtime(true) - self::$startTime;
            $threshold = self::$config['slow_request_threshold'];
            if ($totalTime > $threshold) {
                self::log("Медленный запрос: " . self::formatTime($totalTime), 'warning');
            }
            
            if (self::$config['log_to_file']) {
                self::writeToLog();
            }
            
            // v4.4.0 HOOKS: Проверяем был ли использован Cache хук
            if (self::$panelInsertedViaHook) {
                // Панель уже вставлена через Cache::addBeforeOutputHook()
                // Просто завершаем буферы
                if (function_exists('ob_get_level') && @ob_get_level() > 0) {
                    @ob_end_flush();
                }
                
                self::log('Panel inserted via Cache hook', 'info');
                
            } elseif (self::$config['show_panel']) {
                // Fallback: вставляем панель через processBuffer (старый способ)
                self::processBufferAndOutputPanel();
                
                self::log('Panel inserted via shutdown buffer', 'info');
            } else {
                // Панель отключена в конфиге
                if (function_exists('ob_get_level') && @ob_get_level() > 0) {
                    @ob_end_flush();
                }
            }
            
        } catch (\Throwable $e) {
            @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] CRITICAL ERROR in shutdown(): ' . $e->getMessage());
            
            if (function_exists('ob_get_level') && function_exists('ob_end_clean')) {
                $maxIterations = self::MAX_BUFFER_CLOSE_ITERATIONS;
                $iteration = 0;
                while (@ob_get_level() > 0 && $iteration < $maxIterations) {
                    @ob_end_clean();
                    $iteration++;
                }
            }
        }
    }
    
    /**
     * Обработка буфера и вывод панели (fallback для страниц без кэша)
     */
    private static function processBufferAndOutputPanel() {
        if (headers_sent($file, $line)) {
            @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] Headers already sent in ' . $file . ':' . $line);
            if (function_exists('ob_get_level') && @ob_get_level() > 0) {
                @ob_end_flush();
            }
            return;
        }
        
        $content = '';
        
        if (function_exists('ob_get_level') && function_exists('ob_get_contents') && function_exists('ob_end_clean')) {
            $maxIterations = self::MAX_BUFFER_CLOSE_ITERATIONS;
            $iteration = 0;
            $prevLevel = -1;
            
            while ($iteration < $maxIterations) {
                $currentLevel = @ob_get_level();
                
                if ($currentLevel === 0 || $currentLevel === false) {
                    break;
                }
                
                if ($currentLevel === $prevLevel) {
                    @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] Buffer level stuck at ' . $currentLevel . ', breaking loop');
                    break;
                }
                
                $prevLevel = $currentLevel;
                $buffer = @ob_get_contents();
                @ob_end_clean();
                
                if (empty($content) && is_string($buffer) && $buffer !== false) {
                    $content = $buffer;
                }
                
                $iteration++;
            }
            
            if ($iteration >= $maxIterations) {
                @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] Reached max buffer iterations (' . $maxIterations . ')');
            }
        }
        
        $contentLength = is_string($content) ? strlen($content) : 0;
        if ($contentLength > self::MAX_HTML_SIZE) {
            @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] HTML too large: ' . $contentLength . ' bytes (max ' . self::MAX_HTML_SIZE . ')');
            echo $content;
            return;
        }
        
        $panel = self::renderPanel();
        
        if (!empty($content) && is_string($content) && stripos($content, 'deepmax-debug-panel') !== false) {
            echo $content;
            return;
        }
        
        if (!empty($content) && is_string($content)) {
            $bodyPos = stripos($content, '</body>');
            
            if ($bodyPos !== false) {
                $newContent = substr_replace($content, $panel, $bodyPos, 0);
                
                if ($newContent !== false && is_string($newContent)) {
                    $content = $newContent;
                } else {
                    @error_log('[StockV6] [DeepMaxDebug v' . self::VERSION . '] substr_replace failed, appending to end');
                    $content .= $panel;
                }
            } else {
                $content .= $panel;
            }
            
            echo $content;
        } else {
            echo $panel;
        }
    }
    
    // ============================================
    // ПРИВАТНЫЕ МЕТОДЫ (HELPERS)
    // ============================================
    
    private static function isAllowedIP() {
        if (php_sapi_name() === 'cli') {
            return self::$config['allow_cli'] ?? true;
        }
        
        $allowedIPs = self::$config['allowed_ips'] ?? ['127.0.0.1'];
        $remoteIP = 'unknown';
        
        if (!isset($_SERVER)) {
            return false;
        }
        
        if (isset($_SERVER['REMOTE_ADDR'])) {
            $remoteIP = $_SERVER['REMOTE_ADDR'];
        }
        
        if (isset(self::$config['trust_proxy']) && self::$config['trust_proxy'] === true) {
            $proxyHeaders = [
                'HTTP_X_REAL_IP',
                'HTTP_X_FORWARDED_FOR',
                'HTTP_CLIENT_IP',
            ];
            
            foreach ($proxyHeaders as $header) {
                if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                    if ($header === 'HTTP_X_FORWARDED_FOR') {
                        $ips = explode(',', $_SERVER[$header]);
                        $remoteIP = trim($ips[0]);
                    } else {
                        $remoteIP = $_SERVER[$header];
                    }
                    break;
                }
            }
        }
        
        if (!filter_var($remoteIP, FILTER_VALIDATE_IP)) {
            return false;
        }
        
        return in_array($remoteIP, $allowedIPs, true);
    }
    
    private static function registerHandlers() {
        if (self::$handlersRegistered) {
            return;
        }
        
        if (self::$config['capture_errors']) {
            set_error_handler([__CLASS__, 'errorHandler']);
        }
        
        if (self::$config['capture_exceptions']) {
            set_exception_handler([__CLASS__, 'exceptionHandler']);
        }
        
        register_shutdown_function([__CLASS__, 'shutdown']);
        
        self::$handlersRegistered = true;
    }
    
    /**
     * v4.4.1 CSP-FIX: Рендеринг панели с внешним JS файлом
     * 
     * Убран inline <script> код, используется внешний файл /js/deepmax-debug.js
     * Полная CSP совместимость без необходимости nonce
     * 
     * ПРАВИЛО 10: XSS защита через htmlspecialchars()
     * ПРАВИЛО 11: CSP совместимость (external JS)
     * ПРАВИЛО 21: Комментарии на русском для PHP
     */
    private static function renderPanel() {
        $totalTime = microtime(true) - self::$startTime;
        $peakMemory = memory_get_peak_usage();
        $currentMemory = memory_get_usage();
        $sqlCount = count(self::$sqlQueries);
        $errorCount = count(self::$errors);
        $checkpointCount = count(self::$checkpoints);
        
        // ПРАВИЛО 10: XSS защита через htmlspecialchars
        $formattedTime = htmlspecialchars(self::formatTime($totalTime), ENT_QUOTES, 'UTF-8');
        $formattedMemory = htmlspecialchars(self::formatMemory($currentMemory), ENT_QUOTES, 'UTF-8');
        $formattedPeakMemory = htmlspecialchars(self::formatMemory($peakMemory), ENT_QUOTES, 'UTF-8');
        
        // Цвет статуса панели
        $statusColor = '#4a9d9d';
        $statusColorDark = '#3a7d7d';
        if ($errorCount > 0) {
            $statusColor = '#d9534f';
            $statusColorDark = '#c9433f';
        } elseif ($totalTime > self::$config['slow_request_threshold']) {
            $statusColor = '#f0ad4e';
            $statusColorDark = '#e09d3e';
        }
        
        // Генерация HTML чекпоинтов
        $checkpointsHtml = '';
        $prevTime = 0;
        foreach (self::$checkpoints as $index => $checkpoint) {
            $deltaTime = $checkpoint['time'] - $prevTime;
            $label = htmlspecialchars($checkpoint['label'], ENT_QUOTES, 'UTF-8');
            $checkpointsHtml .= sprintf(
                '<div style="padding:4px 0;border-bottom:1px solid #eee;">'.
                    '<span style="color:#666;">%02d.</span> '.
                    '<span style="color:#333;">%s</span>'.
                    '<span style="float:right;color:#4a9d9d;font-weight:bold;">+%s</span>'.
                '</div>',
                $index + 1,
                $label,
                htmlspecialchars(self::formatTime($deltaTime), ENT_QUOTES, 'UTF-8')
            );
            $prevTime = $checkpoint['time'];
        }
        
        // Генерация HTML ошибок
        $errorsHtml = '';
        if ($errorCount > 0) {
            foreach (self::$errors as $error) {
                $type = htmlspecialchars($error['type'], ENT_QUOTES, 'UTF-8');
                $message = htmlspecialchars($error['message'], ENT_QUOTES, 'UTF-8');
                $file = htmlspecialchars(basename($error['file']), ENT_QUOTES, 'UTF-8');
                $line = (int)$error['line'];
                
                $errorsHtml .= sprintf(
                    '<div style="padding:6px;margin:4px 0;background:#ffe6e6;border-left:3px solid #d9534f;border-radius:3px;font-size:11px;">'.
                        '<div style="font-weight:bold;color:#d9534f;">%s</div>'.
                        '<div style="color:#666;margin-top:2px;">%s</div>'.
                        '<div style="color:#999;margin-top:2px;font-size:10px;">%s:%d</div>'.
                    '</div>',
                    $type,
                    $message,
                    $file,
                    $line
                );
            }
        }
        
        // Генерация HTML логов
        $logsHtml = '';
        $logColors = [
            'info' => '#5bc0de',
            'warning' => '#f0ad4e',
            'error' => '#d9534f',
        ];
        $recentLogs = array_slice(self::$logs, -10);
        foreach ($recentLogs as $log) {
            $color = isset($logColors[$log['level']]) ? $logColors[$log['level']] : '#5bc0de';
            $message = htmlspecialchars($log['message'], ENT_QUOTES, 'UTF-8');
            $time = htmlspecialchars(self::formatTime($log['time']), ENT_QUOTES, 'UTF-8');
            
            $logsHtml .= sprintf(
                '<div style="padding:4px 0;border-bottom:1px solid #eee;font-size:11px;">'.
                    '<span style="display:inline-block;width:8px;height:8px;border-radius:50%%;background:%s;margin-right:6px;"></span>'.
                    '<span style="color:#666;">%s</span>'.
                    '<span style="float:right;color:#999;font-size:10px;">%s</span>'.
                '</div>',
                $color,
                $message,
                $time
            );
        }
        
        // Состояние панели (развёрнута/свёрнута)
        $expandedDisplay = self::$config['panel_expanded'] ? 'block' : 'none';
        $expandIcon = self::$config['panel_expanded'] ? '&#9650;' : '&#9660;';
        
        // Начало HTML панели
        $html = '<!-- DeepMax Debug Panel v' . self::VERSION . ' -->' . "\n";
        $html .= '<div id="deepmax-debug-panel" style="position:fixed;bottom:20px;right:20px;background:#fff;border:2px solid ' . $statusColor . ';border-radius:8px;box-shadow:0 4px 16px rgba(0,0,0,0.2);z-index:9999;font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,Oxygen,Ubuntu,Cantarell,sans-serif;font-size:13px;min-width:320px;max-width:480px;transition:all 0.3s ease;">' . "\n";
        
        // Заголовок панели
        $html .= '    <div style="padding:12px;background:linear-gradient(135deg, ' . $statusColor . ' 0%, ' . $statusColorDark . ' 100%);color:#fff;border-radius:6px 6px 0 0;cursor:pointer;" id="debug-header">' . "\n";
        $html .= '        <div style="display:flex;justify-content:space-between;align-items:center;">' . "\n";
        $html .= '            <div style="font-weight:bold;font-size:14px;">&#129504; DeepMax Debug v' . self::VERSION . '</div>' . "\n";
        $html .= '            <div style="display:flex;gap:8px;">' . "\n";
        $html .= '                <button id="debug-toggle" style="background:rgba(255,255,255,0.2);border:none;color:#fff;cursor:pointer;font-size:14px;padding:2px 8px;border-radius:3px;font-weight:bold;" title="Развернуть/Свернуть">' . $expandIcon . '</button>' . "\n";
        $html .= '                <button id="debug-close" style="background:rgba(255,255,255,0.2);border:none;color:#fff;cursor:pointer;font-size:16px;padding:2px 8px;border-radius:3px;font-weight:bold;" title="Закрыть">&times;</button>' . "\n";
        $html .= '            </div>' . "\n";
        $html .= '        </div>' . "\n";
        $html .= '        <div style="margin-top:8px;font-size:12px;opacity:0.95;display:flex;justify-content:space-between;">' . "\n";
        $html .= '            <span>&#9201; ' . $formattedTime . '</span>' . "\n";
        $html .= '            <span>&#128190; ' . $formattedMemory . '</span>' . "\n";
        $html .= '            <span>&#128202; ' . $sqlCount . ' SQL</span>' . "\n";
        $html .= '            <span>&#9888; ' . $errorCount . ' err</span>' . "\n";
        $html .= '        </div>' . "\n";
        $html .= '    </div>' . "\n";
        
        // Содержимое панели (развёрнутое/свёрнутое)
        $html .= '    <div id="debug-content" style="display:' . $expandedDisplay . ';max-height:60vh;overflow-y:auto;">' . "\n";
        
        // Секция: Метрики производительности
        $html .= '        <div style="padding:12px;border-bottom:1px solid #e0e0e0;background:#f9f9f9;">' . "\n";
        $html .= '            <div style="font-weight:bold;color:#333;margin-bottom:8px;font-size:12px;">&#128202; МЕТРИКИ ПРОИЗВОДИТЕЛЬНОСТИ</div>' . "\n";
        $html .= '            <div style="display:grid;grid-template-columns:1fr 1fr;gap:8px;font-size:11px;">' . "\n";
        $html .= '                <div style="background:#fff;padding:8px;border-radius:4px;border:1px solid #e0e0e0;">' . "\n";
        $html .= '                    <div style="color:#999;font-size:10px;">Время выполнения</div>' . "\n";
        $html .= '                    <div style="color:#333;font-weight:bold;margin-top:2px;">' . $formattedTime . '</div>' . "\n";
        $html .= '                </div>' . "\n";
        $html .= '                <div style="background:#fff;padding:8px;border-radius:4px;border:1px solid #e0e0e0;">' . "\n";
        $html .= '                    <div style="color:#999;font-size:10px;">Память (пик)</div>' . "\n";
        $html .= '                    <div style="color:#333;font-weight:bold;margin-top:2px;">' . $formattedMemory . ' (' . $formattedPeakMemory . ')</div>' . "\n";
        $html .= '                </div>' . "\n";
        $html .= '                <div style="background:#fff;padding:8px;border-radius:4px;border:1px solid #e0e0e0;">' . "\n";
        $html .= '                    <div style="color:#999;font-size:10px;">SQL запросов</div>' . "\n";
        $html .= '                    <div style="color:#333;font-weight:bold;margin-top:2px;">' . $sqlCount . '</div>' . "\n";
        $html .= '                </div>' . "\n";
        $html .= '                <div style="background:#fff;padding:8px;border-radius:4px;border:1px solid #e0e0e0;">' . "\n";
        $html .= '                    <div style="color:#999;font-size:10px;">Чекпоинтов</div>' . "\n";
        $html .= '                    <div style="color:#333;font-weight:bold;margin-top:2px;">' . $checkpointCount . '</div>' . "\n";
        $html .= '                </div>' . "\n";
        $html .= '            </div>' . "\n";
        $html .= '        </div>' . "\n";
        
        // Секция: Чекпоинты
        $html .= '        <div style="padding:12px;border-bottom:1px solid #e0e0e0;">' . "\n";
        $html .= '            <div style="font-weight:bold;color:#333;margin-bottom:8px;font-size:12px;">&#9201; ЧЕКПОИНТЫ</div>' . "\n";
        $html .= '            <div style="max-height:200px;overflow-y:auto;font-size:11px;">' . "\n";
        $html .= $checkpointsHtml;
        $html .= '            </div>' . "\n";
        $html .= '        </div>' . "\n";
        
        // Секция: Ошибки (если есть)
        if ($errorCount > 0) {
            $html .= '        <div style="padding:12px;border-bottom:1px solid #e0e0e0;background:#fff5f5;">' . "\n";
            $html .= '            <div style="font-weight:bold;color:#d9534f;margin-bottom:8px;font-size:12px;">&#9888; ОШИБКИ (' . $errorCount . ')</div>' . "\n";
            $html .= '            <div style="max-height:200px;overflow-y:auto;">' . "\n";
            $html .= $errorsHtml;
            $html .= '            </div>' . "\n";
            $html .= '        </div>' . "\n";
        }
        
        // Секция: Логи (последние 10)
        $html .= '        <div style="padding:12px;">' . "\n";
        $html .= '            <div style="font-weight:bold;color:#333;margin-bottom:8px;font-size:12px;">&#128221; ЛОГИ (последние 10)</div>' . "\n";
        $html .= '            <div style="max-height:200px;overflow-y:auto;">' . "\n";
        $html .= $logsHtml;
        $html .= '            </div>' . "\n";
        $html .= '        </div>' . "\n";
        
        $html .= '    </div>' . "\n";
        $html .= '</div>' . "\n";
        
        
        return $html;
    }
    
    private static function writeToLog() {
        $logDir = self::$config['log_dir'];
        
        $realLogDir = realpath($logDir);
        if ($realLogDir === false) {
            $logDir = self::$config['log_dir'];
        } else {
            $logDir = $realLogDir;
        }
        
        if (strpos($logDir, '..') !== false) {
            return;
        }
        
        if (!is_dir($logDir)) {
            if (!@mkdir($logDir, 0755, true)) {
                return;
            }
        }
        
        if (!is_writable($logDir)) {
            return;
        }
        
        $date = date(self::LOG_DATE_FORMAT);
        $logFile = $logDir . DIRECTORY_SEPARATOR . self::LOG_FILE_PREFIX . $date . '.log';
        
        $totalTime = microtime(true) - self::$startTime;
        $peakMemory = memory_get_peak_usage();
        $currentMemory = memory_get_usage();
        $sqlCount = count(self::$sqlQueries);
        $errorCount = count(self::$errors);
        $checkpointCount = count(self::$checkpoints);
        
        $requestUri = '/';
        $requestMethod = 'UNKNOWN';
        
        if (isset($_SERVER['REQUEST_URI'])) {
            $requestUri = substr($_SERVER['REQUEST_URI'], 0, 200);
            $requestUri = filter_var($requestUri, FILTER_SANITIZE_URL);
        }
        
        if (isset($_SERVER['REQUEST_METHOD'])) {
            $requestMethod = substr($_SERVER['REQUEST_METHOD'], 0, 10);
        }
        
        $timestamp = date(self::LOG_TIMESTAMP_FORMAT);
        $logLine = sprintf(
            "[%s] %s %s | Time: %s | Memory: %s | Peak: %s | SQL: %d | Errors: %d | Checkpoints: %d\n",
            $timestamp,
            $requestMethod,
            $requestUri,
            self::formatTime($totalTime),
            self::formatMemory($currentMemory),
            self::formatMemory($peakMemory),
            $sqlCount,
            $errorCount,
            $checkpointCount
        );
        
        @file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
        
        if (mt_rand(1, self::LOG_CLEANUP_PROBABILITY) === 1) {
            self::cleanOldLogs();
        }
    }
    
    private static function cleanOldLogs() {
        $maxAge = self::$config['max_log_age_days'] ?? 30;
        $logDir = self::$config['log_dir'];
        
        if (!is_dir($logDir)) {
            return;
        }
        
        $files = @glob($logDir . DIRECTORY_SEPARATOR . self::LOG_FILE_PREFIX . '*.log');
        if (!$files) {
            return;
        }
        
        $maxTime = time() - ($maxAge * 86400);
        
        foreach ($files as $file) {
            $fileTime = @filemtime($file);
            if ($fileTime && $fileTime < $maxTime) {
                @unlink($file);
            }
        }
    }
    
    private static function formatTime($seconds) {
        if ($seconds < 0.001) {
            return round($seconds * 1000000) . 'μs';
        } elseif ($seconds < 1) {
            return round($seconds * 1000, 2) . 'ms';
        } elseif ($seconds < 60) {
            return round($seconds, 3) . 's';
        } else {
            $minutes = floor($seconds / 60);
            $secs = $seconds - ($minutes * 60);
            return $minutes . 'm ' . round($secs, 1) . 's';
        }
    }
    
    private static function formatMemory($bytes) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < 4) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 1) . ' ' . $units[$i];
    }
    
    private static function filterPasswords($data) {
        if (!is_array($data)) {
            return $data;
        }
        
        $jsonTest = @json_encode($data);
        if ($jsonTest === false) {
            return ['[фильтровано - циклическая структура]'];
        }
        
        $sensitiveKeys = ['password', 'pass', 'pwd', 'secret', 'token', 'api_key', 'apikey'];
        
        $filtered = [];
        foreach ($data as $key => $value) {
            $keyLower = strtolower($key);
            $isSensitive = false;
            
            foreach ($sensitiveKeys as $sensitiveKey) {
                if (strpos($keyLower, $sensitiveKey) !== false) {
                    $isSensitive = true;
                    break;
                }
            }
            
            if ($isSensitive) {
                $filtered[$key] = '***FILTERED***';
            } elseif (is_array($value)) {
                $filtered[$key] = self::filterPasswords($value);
            } else {
                $filtered[$key] = $value;
            }
        }
        
        return $filtered;
    }
    
    public static function getMetrics() {
        if (!self::$enabled) {
            return [];
        }
        
        return [
            'time' => microtime(true) - self::$startTime,
            'memory' => memory_get_usage(),
            'peak_memory' => memory_get_peak_usage(),
            'sql_count' => count(self::$sqlQueries),
            'error_count' => count(self::$errors),
            'checkpoint_count' => count(self::$checkpoints),
        ];
    }
    
    public static function getCheckpoints() {
        return self::$checkpoints;
    }
    
    public static function getLogs() {
        return self::$logs;
    }
    
    public static function getErrors() {
        return self::$errors;
    }
    
    public static function isEnabled() {
        return self::$enabled;
    }
    
    public static function setCspNonce($nonce) {
        self::$cspNonce = $nonce;
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════════════
 * END DeepMaxDebug.php v4.4.1 CSP-FIX ✅
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 4.4.1 CSP-FIX
 * Дата: 2025-10-18
 * Статус: PRODUCTION READY - CSP COMPATIBLE
 * 
 * НОВОЕ v4.4.1 CSP-FIX:
 *   ✅ renderPanel() использует внешний JS файл
 *   ✅ Убран весь inline <script> код
 *   ✅ CSP совместимость без nonce
 *   ✅ IE11 совместимость в JS файле
 *   ✅ Memory cleanup в JS файле
 *   ✅ PHP комментарии на русском (ПРАВИЛО 21)
 * 
 * v4.4.0 HOOKS:
 *   ✅ getPanelHTML() - для использования с Cache хуками
 *   ✅ Флаг $panelInsertedViaHook - предотвращение дублирования
 *   ✅ shutdown() проверяет флаг и НЕ вставляет панель дважды
 *   ✅ Fallback для страниц без кэша (processBufferAndOutputPanel)
 * 
 * СОВМЕСТИМОСТЬ:
 *   - Работает с cache.php v3.3.0+ HOOKS
 *   - Работает с index.php v2.4.0+ HOOKS
 *   - Обратная совместимость с v4.3.2
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */