<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС РЕНДЕРИНГА ХЛЕБНЫХ КРОШЕК
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 1.0.0 - Первая версия после рефакторинга
 * 
 * CHANGELOG v1.0.0 (2025-11-02):
 *   ✨ NEW: Выделение логики рендеринга HTML из BreadcrumbBuilder
 *   ✨ NEW: Метод renderHtml() - генерация HTML разметки
 *   ✨ NEW: Метод escapeBreadscrumbs() - экранирование данных
 *   ✨ NEW: Метод buildHtmlMarkup() - построение HTML структуры
 *   📐 АРХИТЕКТУРА: Single Responsibility Principle
 * 
 * НАЗНАЧЕНИЕ:
 *   - Генерация HTML разметки хлебных крошек
 *   - Микроразметка Schema.org для навигации
 *   - Экранирование данных для безопасного вывода
 *   - Адаптация под desktop/mobile (будущее расширение)
 * 
 * ТРЕБОВАНИЯ:
 *   - PHP >= 7.4
 *   - Работает с данными из BreadcrumbBuilder::getBreadcrumbs()
 * 
 * @version 1.0.0
 * @date 2025-11-02
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * @license Proprietary
 * ═══════════════════════════════════════════════════════════════════════════
 */

declare(strict_types=1);

class BreadcrumbRenderer
{
    /**
     * Рендеринг HTML разметки breadcrumbs с микроразметкой Schema.org
     * 
     * @param array<int, array<string, string>> $breadcrumbs Массив крошек
     * @param bool $isMobile Мобильная версия (для будущего использования)
     * @return string HTML код или пустая строка
     */
    public function renderHtml(array $breadcrumbs, bool $isMobile = false): string
    {
        // Если меньше 2 крошек - не показываем навигацию
        if (count($breadcrumbs) <= 1) {
            return '';
        }
        
        // Экранирование данных для безопасности
        $escapedCrumbs = $this->escapeBreadscrumbs($breadcrumbs);
        
        return $this->buildHtmlMarkup($escapedCrumbs);
    }
    
    /**
     * Экранирование данных breadcrumbs для безопасного вывода в HTML
     * 
     * @param array<int, array<string, string>> $crumbs
     * @return array<int, array<string, string>>
     */
    private function escapeBreadscrumbs(array $crumbs): array
    {
        $escaped = [];
        
        foreach ($crumbs as $crumb) {
            $escaped[] = [
                'name' => htmlspecialchars($crumb['name'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'),
                'url' => htmlspecialchars($crumb['url'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8')
            ];
        }
        
        return $escaped;
    }
    
    /**
     * Построение HTML разметки breadcrumbs
     * 
     * @param array<int, array<string, string>> $crumbs Экранированные крошки
     * @return string HTML код
     */
    private function buildHtmlMarkup(array $crumbs): string
    {
        $html = '<nav aria-label="Breadcrumb" class="breadcrumbs">' . "\n";
        $html .= '  <ol itemscope itemtype="https://schema.org/BreadcrumbList">' . "\n";
        
        $total = count($crumbs);
        
        foreach ($crumbs as $index => $crumb) {
            $position = $index + 1;
            $isLast = ($position === $total);
            
            $html .= '    <li itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"';
            
            if ($isLast) {
                $html .= ' class="active"';
            }
            
            $html .= '>' . "\n";
            
            if ($isLast) {
                $html .= '      <span itemprop="name">' . $crumb['name'] . '</span>' . "\n";
            } else {
                $html .= '      <a itemprop="item" href="' . $crumb['url'] . '">' . "\n";
                $html .= '        <span itemprop="name">' . $crumb['name'] . '</span>' . "\n";
                $html .= '      </a>' . "\n";
            }
            
            $html .= '      <meta itemprop="position" content="' . $position . '">' . "\n";
            $html .= '    </li>' . "\n";
        }
        
        $html .= '  </ol>' . "\n";
        $html .= '  <div class="breadcrumb-indicator"></div>' . "\n";
        $html .= '</nav>';
        
        return $html;
    }
}
