<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КЛАСС ПОСТРОЕНИЯ ХЛЕБНЫХ КРОШЕК
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * ВЕРСИЯ 6.2.0 - FEATURE: Поддержка каталогов инструментов
 * 
 * CHANGELOG v6.2.0 (2025-10-22):
 *   ✨ NEW: Поддержка страниц каталогов (списков инструментов)
 *   ✨ NEW: 6 новых констант PAGE_TYPE_CATALOG_* для каталогов
 *   ✨ NEW: Определение pageType для каталогов в tryBuildInstrumentPath()
 *   ✨ NEW: Маппинг Schema.org классов для каталогов в loadSchemaOrgClass()
 *   📝 DOCS: Обновлена документация по типам страниц
 * 
 * CHANGELOG v6.1.1 (2025-10-14):
 *   🔧 FIX: Исправлены названия QUIK таблиц в $instrumentTypes
 *       - Type 2 (расписки): QUIK_Receipts → QUIK_Akcii
 *       - Type 3 (еврооблигации): QUIK_Eurobonds → QUIK_Obligacii
 *       - Type 4 (паи): QUIK_Funds → QUIK_Akcii
 *       - Type 5 (ипотечные): QUIK_Mortgage_Notes → QUIK_Obligacii
 *   🔧 FIX: Упрощён getQuikFieldName() - только 2 таблицы вместо 6
 *   🔧 FIX: validateQuikTable() - только реально существующие таблицы
 *   📝 DOCS: Обновлены комментарии под реальную структуру БД
 * 
 * CHANGELOG v6.1.0 (2025-10-13):
 *   🔒 FIX: Thread-safe кэш с mutex блокировками
 *   🔒 FIX: Улучшенный exception handling в конструкторе
 *   🔒 FIX: Автоматическая очистка устаревшего кэша по TTL
 *   🔒 FIX: Валидация SITE_URL на корректность
 *   🔒 FIX: Единый подход к error handling (только exceptions)
 *   ✨ NEW: Метод isCacheHealthy() для мониторинга кэша
 *   ✨ NEW: Публичные геттеры: getPageType(), getSupertypeId(), isBuilt()
 *   ✨ NEW: Защита от circular references в URL
 *   ⚡ PERF: Оптимизированное логирование без конкатенации
 *   📝 DOCS: Обновлена документация по exception handling
 * 
 * ИЗМЕНЕНИЯ В v6.0:
 *   🔥 BREAKING: Требуется PHP >= 7.4
 *   ✨ NEW: Метод loadSchemaOrgClass() - убрано дублирование
 *   🔧 FIX: Улучшена UTF-8 валидация
 *   ⚡ PERF: Оптимизирован кэш (100 вместо 1000)
 * 
 * НАЗНАЧЕНИЕ:
 *   - Построение навигационных цепочек для всех типов страниц
 *   - HTML генерация с микроразметкой Schema.org
 *   - Базовая JSON-LD для BreadcrumbList
 *   - Автоматическая маршрутизация к SchemaOrg классам
 *   - Thread-safe кэширование запросов к БД
 * 
 * ТРЕБОВАНИЯ:
 *   - PHP >= 7.4
 *   - mbstring extension
 *   - mysqli extension
 *   - SITE_URL constant defined (valid URL)
 * 
 * @version 6.2.0
 * @date 2025-10-22
 * @author DeepMax Development Team
 * @status PRODUCTION READY ✅
 * @license Proprietary
 * ═══════════════════════════════════════════════════════════════════════════
 */

declare(strict_types=1);

class BreadcrumbBuilder
{
    // ═══════════════════════════════════════════════════════════════════════
    // КОНСТАНТЫ ТИПОВ ИНСТРУМЕНТОВ
    // ═══════════════════════════════════════════════════════════════════════
    
    public const INSTRUMENT_TYPE_STOCKS = 1;
    public const INSTRUMENT_TYPE_RECEIPTS = 2;
    public const INSTRUMENT_TYPE_EUROBONDS = 3;
    public const INSTRUMENT_TYPE_FUNDS = 4;
    public const INSTRUMENT_TYPE_MORTGAGE_NOTES = 5;
    public const INSTRUMENT_TYPE_BONDS = 6;
    
    // ═══════════════════════════════════════════════════════════════════════
    // КОНСТАНТЫ ТИПОВ СТРАНИЦ
    // ═══════════════════════════════════════════════════════════════════════
    
    public const PAGE_TYPE_EQUITY = 'equity';
    public const PAGE_TYPE_DEBT = 'debt';
    public const PAGE_TYPE_EMITENT = 'emitent';
    public const PAGE_TYPE_SECTOR = 'sector';
    
    // Константы для каталогов (v6.2.0)
    public const PAGE_TYPE_CATALOG_STOCKS = 'catalog_stocks';
    public const PAGE_TYPE_CATALOG_RECEIPTS = 'catalog_receipts';
    public const PAGE_TYPE_CATALOG_EUROBONDS = 'catalog_eurobonds';
    public const PAGE_TYPE_CATALOG_FUNDS = 'catalog_funds';
    public const PAGE_TYPE_CATALOG_MORTGAGE_NOTES = 'catalog_mortgage_notes';
    public const PAGE_TYPE_CATALOG_BONDS = 'catalog_bonds';
    public const PAGE_TYPE_CATALOG_SECTORS = 'catalog_sectors';
    public const PAGE_TYPE_CATALOG_EMITENTS = 'catalog_emitents';
    
    // ═══════════════════════════════════════════════════════════════════════
    // КОНСТАНТЫ КЭШИРОВАНИЯ И ЛИМИТОВ
    // ═══════════════════════════════════════════════════════════════════════
    
    private const CACHE_NOT_FOUND = '__NOT_FOUND__';
    private const CACHE_VERSION = 9; // Увеличена версия для v6.2.0
    private const CACHE_TTL = 3600;
    private const CACHE_CLEANUP_PROBABILITY = 0.01; // 1% вероятность очистки
    private const MAX_CACHE_SIZE = 100;
    private const MAX_NOT_FOUND_CACHE = 50;
    private const CACHE_EVICTION_PERCENTAGE = 0.1;
    
    private const MAX_URL_LENGTH = 2000;
    private const MAX_BREADCRUMB_DEPTH = 10;
    private const MAX_CRUMB_NAME_LENGTH = 250;
    private const ELLIPSIS_LENGTH = 3;
    private const MAX_LOG_ERRORS_PER_MINUTE = 10;
    private const MAX_CIRCULAR_REFERENCE_DEPTH = 5;
    
    // ═══════════════════════════════════════════════════════════════════════
    // КОНФИГУРАЦИЯ ТИПОВ ИНСТРУМЕНТОВ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * ВАЖНО: В БД существуют только 2 QUIK таблицы:
     *   - QUIK_Akcii: для equity инструментов (types 1, 2, 4)
     *   - QUIK_Obligacii: для debt инструментов (types 3, 5, 6)
     * 
     * @var array<int, array<string, string>>
     */
    private static array $instrumentTypes = [
        self::INSTRUMENT_TYPE_STOCKS => [
            'name' => 'Акции',
            'slug' => 'stocks',
            'plural' => 'Каталог акций',
            'quik_table' => 'QUIK_Akcii',
            'group' => self::PAGE_TYPE_EQUITY,
            'catalog_type' => self::PAGE_TYPE_CATALOG_STOCKS
        ],
        self::INSTRUMENT_TYPE_RECEIPTS => [
            'name' => 'Депозитарные расписки',
            'slug' => 'receipts',
            'plural' => 'Все Депозитарные расписки',
            'quik_table' => 'QUIK_Akcii',  // v6.1.1: ИСПРАВЛЕНО с QUIK_Receipts
            'group' => self::PAGE_TYPE_EQUITY,
            'catalog_type' => self::PAGE_TYPE_CATALOG_RECEIPTS
        ],
        self::INSTRUMENT_TYPE_EUROBONDS => [
            'name' => 'Еврооблигации',
            'slug' => 'eurobonds',
            'plural' => 'Все Еврооблигации',
            'quik_table' => 'QUIK_Obligacii',  // v6.1.1: ИСПРАВЛЕНО с QUIK_Eurobonds
            'group' => self::PAGE_TYPE_DEBT,
            'catalog_type' => self::PAGE_TYPE_CATALOG_EUROBONDS
        ],
        self::INSTRUMENT_TYPE_FUNDS => [
            'name' => 'Инвестиционные паи',
            'slug' => 'funds',
            'plural' => 'Все Инвестиционные паи',
            'quik_table' => 'QUIK_Akcii',  // v6.1.1: ИСПРАВЛЕНО с QUIK_Funds
            'group' => self::PAGE_TYPE_EQUITY,
            'catalog_type' => self::PAGE_TYPE_CATALOG_FUNDS
        ],
        self::INSTRUMENT_TYPE_MORTGAGE_NOTES => [
            'name' => 'Ипотечные сертификаты участия',
            'slug' => 'mortgage-notes',
            'plural' => 'Все Ипотечные сертификаты участия',
            'quik_table' => 'QUIK_Obligacii',  // v6.1.1: ИСПРАВЛЕНО с QUIK_Mortgage_Notes
            'group' => self::PAGE_TYPE_DEBT,
            'catalog_type' => self::PAGE_TYPE_CATALOG_MORTGAGE_NOTES
        ],
        self::INSTRUMENT_TYPE_BONDS => [
            'name' => 'Облигации',
            'slug' => 'bonds',
            'plural' => 'Каталог облигаций',
            'quik_table' => 'QUIK_Obligacii',
            'group' => self::PAGE_TYPE_DEBT,
            'catalog_type' => self::PAGE_TYPE_CATALOG_BONDS
        ]
    ];
    
    // ═══════════════════════════════════════════════════════════════════════
    // СВОЙСТВА ЭКЗЕМПЛЯРА
    // ═══════════════════════════════════════════════════════════════════════
    
    private mysqli $db;
    private string $baseUrl;
    private bool $isBuilt = false;
    private ?string $lastUrl = null;
    private int $circularReferenceDepth = 0;
    
    /** @var array<int, array<string, string>> */
    private array $breadcrumbs = [];
    
    /** @var array<string, bool> */
    private array $usedUrls = [];
    
    private ?string $pageType = null;
    private ?int $supertypeId = null;
    
    // ═══════════════════════════════════════════════════════════════════════
    // СТАТИЧЕСКИЕ СВОЙСТВА КЭША С THREAD-SAFETY
    // ═══════════════════════════════════════════════════════════════════════
    
    /** @var array<string, array<string, mixed>> */
    private static array $queryCache = [];
    
    /** @var resource|false|null Файловый мьютекс для thread-safety */
    private static $cacheMutex = null;
    
    private static bool $cacheEnabled = true;
    private static int $cacheHits = 0;
    private static int $cacheMisses = 0;
    private static int $notFoundCount = 0;
    private static int $errorLogCount = 0;
    private static int $errorLogResetTime = 0;
    private static int $lastCacheCleanup = 0;
    
    // ═══════════════════════════════════════════════════════════════════════
    // КОНСТРУКТОР
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Конструктор класса с улучшенной обработкой исключений
     * 
     * @param mysqli|object $database Подключение к БД
     * 
     * @throws RuntimeException Если отсутствует расширение mbstring
     * @throws RuntimeException Если некорректная конфигурация
     * @throws InvalidArgumentException Если некорректное подключение к БД
     * @throws RuntimeException Если SITE_URL не определена или некорректна
     */
    public function __construct($database)
    {
        try {
            // Проверка зависимостей
            $this->validateDependencies();
            
            // Валидация конфигурации
            $this->validateConfiguration();
            
            // Инициализация подключения к БД
            $this->db = $this->extractDatabaseConnection($database);
            
            // Валидация и инициализация SITE_URL
            $this->initializeBaseUrl();
            
            // Инициализация мьютекса для thread-safety
            $this->initializeCacheMutex();
            
        } catch (RuntimeException $e) {
            // Логируем и пробрасываем дальше с контекстом
            $this->logError('Constructor failed: ' . $e->getMessage());
            throw new RuntimeException(
                'BreadcrumbBuilder initialization failed: ' . $e->getMessage(),
                0,
                $e
            );
        } catch (InvalidArgumentException $e) {
            $this->logError('Invalid constructor argument: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Проверка наличия необходимых PHP расширений
     * 
     * @return void
     * @throws RuntimeException
     */
    private function validateDependencies(): void
    {
        if (!function_exists('mb_strlen')) {
            throw new RuntimeException('PHP mbstring extension is required');
        }
        
        if (!class_exists('mysqli')) {
            throw new RuntimeException('PHP mysqli extension is required');
        }
    }
    
    /**
     * Валидация конфигурации класса
     * 
     * @return void
     * @throws RuntimeException
     */
    private function validateConfiguration(): void
    {
        // Проверка корректности констант
        if (self::ELLIPSIS_LENGTH >= self::MAX_CRUMB_NAME_LENGTH) {
            throw new RuntimeException(
                'ELLIPSIS_LENGTH (' . self::ELLIPSIS_LENGTH . ') must be less than MAX_CRUMB_NAME_LENGTH (' . self::MAX_CRUMB_NAME_LENGTH . ')'
            );
        }
        
        if (self::MAX_CACHE_SIZE <= 0) {
            throw new RuntimeException('MAX_CACHE_SIZE must be positive integer');
        }
        
        if (self::CACHE_TTL <= 0) {
            throw new RuntimeException('CACHE_TTL must be positive integer');
        }
        
        // Проверка конфигурации типов инструментов
        $requiredKeys = ['name', 'slug', 'plural', 'quik_table', 'group'];
        
        foreach (self::$instrumentTypes as $typeId => $config) {
            foreach ($requiredKeys as $key) {
                if (!isset($config[$key]) || !is_string($config[$key]) || $config[$key] === '') {
                    throw new RuntimeException(
                        "Missing or invalid required key '{$key}' in configuration for instrument type {$typeId}"
                    );
                }
            }
        }
    }
    
    /**
     * Извлечение подключения к БД из разных типов входных данных
     * 
     * @param mysqli|object $database
     * @return mysqli
     * @throws InvalidArgumentException
     */
    private function extractDatabaseConnection($database): mysqli
    {
        // Если это объект с property connection
        if (is_object($database) && property_exists($database, 'connection')) {
            $database = $database->connection;
        }
        
        // Проверка что это mysqli
        if (!($database instanceof mysqli)) {
            throw new InvalidArgumentException(
                'Database must be mysqli instance, ' . gettype($database) . ' given'
            );
        }
        
        // Проверка что соединение активно
        if (!@$database->ping()) {
            throw new InvalidArgumentException('Database connection is not active');
        }
        
        return $database;
    }
    
    /**
     * Инициализация и валидация базового URL
     * 
     * @return void
     * @throws RuntimeException
     */
    private function initializeBaseUrl(): void
    {
        if (!defined('SITE_URL')) {
            throw new RuntimeException('SITE_URL constant is not defined');
        }
        
        $siteUrl = SITE_URL;
        
        if (!is_string($siteUrl) || $siteUrl === '') {
            throw new RuntimeException('SITE_URL must be non-empty string');
        }
        
        // Валидация что это корректный URL
        $parsedUrl = parse_url($siteUrl);
        
        if ($parsedUrl === false || !isset($parsedUrl['scheme']) || !isset($parsedUrl['host'])) {
            throw new RuntimeException('SITE_URL is not a valid URL: ' . $siteUrl);
        }
        
        // Проверка на допустимые схемы
        if (!in_array(strtolower($parsedUrl['scheme']), ['http', 'https'], true)) {
            throw new RuntimeException('SITE_URL must use http or https scheme, got: ' . $parsedUrl['scheme']);
        }
        
        // Проверка на XSS в URL
        if (preg_match('/<|>|javascript:/i', $siteUrl)) {
            throw new RuntimeException('SITE_URL contains potentially dangerous content');
        }
        
        $this->baseUrl = rtrim($siteUrl, '/');
    }
    
    /**
     * Инициализация файлового мьютекса для thread-safe кэша
     * 
     * @return void
     */
    private function initializeCacheMutex(): void
    {
        if (self::$cacheMutex === null) {
            $lockFile = __DIR__ . '/../cache/breadcrumb_cache.lock';
            
            // Создаём lock файл если не существует
            if (!file_exists($lockFile)) {
                @touch($lockFile);
            }
            
            if (file_exists($lockFile) && is_writable($lockFile)) {
                self::$cacheMutex = @fopen($lockFile, 'c+');
            } else {
                // Если не удалось создать мьютекс - логируем но продолжаем работу
                $this->logError('Failed to initialize cache mutex, thread-safety disabled');
                self::$cacheMutex = false;
            }
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // ПУБЛИЧНЫЕ ГЕТТЕРЫ СОСТОЯНИЯ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Получить тип текущей страницы
     * 
     * @return string|null equity|debt|emitent|sector или null
     */
    public function getPageType(): ?string
    {
        return $this->pageType;
    }
    
    /**
     * Получить ID типа инструмента
     * 
     * @return int|null
     */
    public function getSupertypeId(): ?int
    {
        return $this->supertypeId;
    }
    
    /**
     * Проверить построены ли breadcrumbs
     * 
     * @return bool
     */
    public function isBuilt(): bool
    {
        return $this->isBuilt;
    }
    
    /**
     * Получить последний обработанный URL
     * 
     * @return string|null
     */
    public function getLastUrl(): ?string
    {
        return $this->lastUrl;
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // ОСНОВНЫЕ ПУБЛИЧНЫЕ МЕТОДЫ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Построить цепочку breadcrumbs из URL
     * 
     * @param string $currentUrl Текущий URL страницы
     * @return self Для цепочки вызовов
     * @throws InvalidArgumentException Если URL некорректен
     */
    public function build(string $currentUrl): self
    {
        // Если уже построено для этого URL - вернуть существующее
        if ($this->isBuilt && $this->lastUrl === $currentUrl) {
            return $this;
        }
        
        // Защита от circular references
        $this->circularReferenceDepth++;
        
        if ($this->circularReferenceDepth > self::MAX_CIRCULAR_REFERENCE_DEPTH) {
            throw new RuntimeException(
                'Circular reference detected: build() called more than ' . 
                self::MAX_CIRCULAR_REFERENCE_DEPTH . ' times'
            );
        }
        
        try {
            // Сброс состояния для пересборки
            $this->reset();
            
            // Проверка длины URL
            if (strlen($currentUrl) > self::MAX_URL_LENGTH) {
                throw new InvalidArgumentException(
                    'URL exceeds maximum length: ' . strlen($currentUrl) . ' > ' . self::MAX_URL_LENGTH
                );
            }
            
            // Парсинг URL
            $path = parse_url($currentUrl, PHP_URL_PATH);
            if ($path === false || $path === null) {
                throw new InvalidArgumentException('Failed to parse URL: ' . $currentUrl);
            }
            
            $path = trim($path, '/');
            
            // Главная страница (всегда первая крошка)
            $this->addCrumb('Главная', $this->baseUrl . '/');
            
            // Если путь пустой - только главная
            if ($path === '') {
                return $this;
            }
            
            // Специальная страница "Сектора"
            if ($path === 'sectors') {
                $this->addCrumb('Сектора экономики', $this->baseUrl . '/sectors/');
                $this->pageType = self::PAGE_TYPE_CATALOG_SECTORS;
                return $this;
            }
            
            // Специальная страница "Эмитенты"
            if ($path === 'emitents') {
                $this->addCrumb('Каталог эмитентов', $this->baseUrl . '/emitents/');
                $this->pageType = self::PAGE_TYPE_CATALOG_EMITENTS;
                return $this;
            }
            
            // Проверка на страницы инструментов
            if ($this->tryBuildInstrumentPath($path)) {
                return $this;
            }
            
            // Динамические страницы (сектора, эмитенты)
            $decoded = @urldecode($path);
            if ($decoded !== false && $decoded !== null) {
                $this->buildDynamicBreadcrumb($decoded);
            }
            
            return $this;
            
        } finally {
            // Всегда уменьшаем счётчик после выполнения
            $this->circularReferenceDepth--;
        }
    }
    
    /**
     * Сброс состояния для пересборки
     * 
     * @return void
     */
    private function reset(): void
    {
        $this->breadcrumbs = [];
        $this->usedUrls = [];
        $this->pageType = null;
        $this->supertypeId = null;
        $this->isBuilt = true;
        
        // Вероятностная очистка устаревшего кэша
        $this->probabilisticCacheCleanup();
    }
    
    /**
     * Вероятностная очистка устаревшего кэша (1% вероятность)
     * 
     * @return void
     */
    private function probabilisticCacheCleanup(): void
    {
        $random = mt_rand(1, 100) / 100;
        
        if ($random <= self::CACHE_CLEANUP_PROBABILITY) {
            $this->cleanupExpiredCache();
        }
    }
    
    /**
     * Очистка устаревших записей из кэша
     * 
     * @return void
     */
    private function cleanupExpiredCache(): void
    {
        // Защита от слишком частой очистки
        $currentTime = time();
        
        if ($currentTime - self::$lastCacheCleanup < 60) {
            return;
        }
        
        self::$lastCacheCleanup = $currentTime;
        
        $this->acquireCacheLock();
        
        try {
            $removed = 0;
            
            foreach (self::$queryCache as $key => $cached) {
                if (!isset($cached['expires']) || $cached['expires'] <= $currentTime) {
                    unset(self::$queryCache[$key]);
                    $removed++;
                }
            }
            
            if ($removed > 0) {
                $this->logError("Cache cleanup: removed {$removed} expired entries");
            }
            
        } finally {
            $this->releaseCacheLock();
        }
    }
    
    /**
     * Попытка построить путь для страницы инструмента
     * 
     * @param string $path Путь из URL
     * @return bool True если это страница инструмента
     */
    private function tryBuildInstrumentPath(string $path): bool
    {
        foreach (self::$instrumentTypes as $typeId => $typeConfig) {
            $slug = $typeConfig['slug'];
            
            // Проверка что путь начинается со slug инструмента
            if (strpos($path, $slug . '/') === 0) {
                $code = str_replace($slug . '/', '', $path);
                $code = trim($code, '/');
                
                $decoded = @urldecode($code);
                if ($decoded === null || $decoded === false) {
                    return true;
                }
                $code = $decoded;
                
                if ($code !== '') {
                    // Страница отдельного инструмента
                    $this->buildInstrumentBreadcrumb($code, $typeId);
                } else {
                    // Страница каталога (список инструментов) - v6.2.0
                    $this->addCrumb($typeConfig['plural'], $this->baseUrl . '/' . $slug . '/');
                    $this->pageType = $typeConfig['catalog_type'];
                    $this->supertypeId = $typeId;
                }
                
                return true;
            }
            
            // Проверка точного совпадения со slug (список всех инструментов)
            if ($path === $slug) {
                // Страница каталога (список инструментов) - v6.2.0
                $this->addCrumb($typeConfig['plural'], $this->baseUrl . '/' . $slug . '/');
                $this->pageType = $typeConfig['catalog_type'];
                $this->supertypeId = $typeId;
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Получить массив breadcrumbs
     * 
     * @return array<int, array<string, string>>
     */
    public function getBreadcrumbs(): array
    {
        return $this->breadcrumbs;
    }
    
    
    /**
     * Генерация расширенной JSON-LD разметки с автоматической маршрутизацией
     * 
     * @param array<string, mixed> $page_data Данные страницы
     * @return string JSON-LD script tag или HTML комментарий
     * @throws InvalidArgumentException Если page_data некорректен
     */
    public function getJsonLdEnhanced(array $page_data): string
    {
        // Тип страницы не определён
        if ($this->pageType === null) {
            return '<!-- BreadcrumbBuilder: pageType не определён -->';
        }
        
        // Проверка что массив не пустой (но может содержать 0 и false)
        if (count($page_data) === 0) {
            return '<!-- BreadcrumbBuilder: нет данных страницы -->';
        }
        
        // Загрузка соответствующего Schema.org класса
        try {
            $schema = $this->loadSchemaOrgClass($this->pageType);
            
            if ($schema === null) {
                return '<!-- BreadcrumbBuilder: не удалось загрузить Schema.org класс для типа: ' 
                       . htmlspecialchars($this->pageType, ENT_QUOTES, 'UTF-8') . ' -->';
            }
            
            // Генерация JSON-LD разметки
            $result = $schema->buildEnhancedJsonLd($page_data, $this->breadcrumbs);
            return $result;
            
        } catch (Exception $e) {
            $this->logError('Schema.org generation error: ' . $e->getMessage());
            return '<!-- BreadcrumbBuilder: ошибка генерации Schema.org -->';
        }
    }
    
    /**
     * Загрузка и инициализация Schema.org класса
     * 
     * @param string $type Тип класса
     * @return object|null Экземпляр класса или null
     * @throws RuntimeException Если файлы не найдены или класс не создан
     */
    private function loadSchemaOrgClass(string $type): ?object
    {
        // Маппинг типов на классы (v6.2.0: добавлены каталоги)
        $classMap = [
            self::PAGE_TYPE_EQUITY => 'SchemaOrgEquity',
            self::PAGE_TYPE_DEBT => 'SchemaOrgDebt',
            self::PAGE_TYPE_EMITENT => 'SchemaOrgEmitent',
            self::PAGE_TYPE_SECTOR => 'SchemaOrgSector',
            self::PAGE_TYPE_CATALOG_STOCKS => 'SchemaOrgStocksCatalog',
            self::PAGE_TYPE_CATALOG_RECEIPTS => 'SchemaOrgReceiptsCatalog',
            self::PAGE_TYPE_CATALOG_EUROBONDS => 'SchemaOrgEurobondsCatalog',
            self::PAGE_TYPE_CATALOG_FUNDS => 'SchemaOrgFundsCatalog',
            self::PAGE_TYPE_CATALOG_MORTGAGE_NOTES => 'SchemaOrgMortgageNotesCatalog',
            self::PAGE_TYPE_CATALOG_BONDS => 'SchemaOrgBondsCatalog',
            self::PAGE_TYPE_CATALOG_SECTORS => 'SchemaOrgSectorsCatalog',
            self::PAGE_TYPE_CATALOG_EMITENTS => 'SchemaOrgEmitentsCatalog'
        ];
        
        // Проверка что тип существует
        if (!isset($classMap[$type])) {
            throw new InvalidArgumentException('Unknown Schema.org type: ' . $type);
        }
        
        $className = $classMap[$type];
        $baseFile = __DIR__ . '/SchemaOrg/SchemaOrgBase.php';
        $classFile = __DIR__ . '/SchemaOrg/' . $className . '.php';
        
        // Проверка существования файлов
        if (!file_exists($baseFile)) {
            throw new RuntimeException('Schema.org base file not found: ' . $baseFile);
        }
        
        if (!file_exists($classFile)) {
            throw new RuntimeException('Schema.org class file not found: ' . $classFile);
        }
        
        // Подключение файлов
        require_once $baseFile;
        require_once $classFile;
        
        // Проверка что класс существует
        if (!class_exists($className)) {
            throw new RuntimeException('Schema.org class not found after require: ' . $className);
        }
        
        // Создание экземпляра
        return new $className($this->db);
    }
    
    /**
     * Преобразование объекта в строку (вызывается при echo)
     * 
     * @return string Пустая строка (HTML генерируется через BreadcrumbRenderer)
     */
    public function __toString(): string
    {
        // HTML генерация вынесена в BreadcrumbRenderer
        // Этот метод оставлен для совместимости, но возвращает пустую строку
        return '';
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // ПОСТРОЕНИЕ BREADCRUMBS ДЛЯ ИНСТРУМЕНТОВ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Построить breadcrumb цепочку для страницы инструмента
     * 
     * @param string $code Торговый код или ISIN инструмента
     * @param int $supertypeId ID типа инструмента
     * @return void
     */
    private function buildInstrumentBreadcrumb(string $code, int $supertypeId): void
    {
        // Проверка что тип существует
        if (!isset(self::$instrumentTypes[$supertypeId])) {
            $this->logError('Unknown instrument supertype: ' . $supertypeId);
            return;
        }
        
        // Сохранение типа страницы для Schema.org
        $this->supertypeId = $supertypeId;
        $this->pageType = self::$instrumentTypes[$supertypeId]['group'];
        
        // Валидация кода инструмента
        if (!$this->isValidInstrumentCode($code)) {
            return;
        }
        
        $typeConfig = self::$instrumentTypes[$supertypeId];
        
        // Попытка получить данные из кэша или БД
        $cacheKey = "instrument_{$supertypeId}_{$code}";
        $row = $this->getCachedOrFetch($cacheKey, function() use ($code, $supertypeId) {
            return $this->fetchInstrumentData($code, $supertypeId, true);
        });
        
        // Данные не найдены
        if (!is_array($row)) {
            return;
        }
        
        // Построение цепочки
        $this->addSectorCrumbsIfAvailable($row);
        $this->addEmitentCrumbIfAvailable($row);
        $this->addCrumb($typeConfig['name'], $this->baseUrl . '/' . $typeConfig['slug'] . '/');
        
        if ($this->isValidString($row, 'Instrument_TRADE_CODE')) {
            $crumbName = $this->buildInstrumentCrumbName($row);
            $this->addCrumb($crumbName, $this->baseUrl . '/' . $typeConfig['slug'] . '/' . $code . '/');
        }
    }
    
    /**
     * Валидация кода инструмента
     * 
     * @param string $code Код для проверки
     * @return bool
     */
    private function isValidInstrumentCode(string $code): bool
    {
        $codeLength = mb_strlen($code, 'UTF-8');
        
        if ($codeLength === 0 || $codeLength > 255) {
            $this->logError('Invalid instrument code length');
            return false;
        }
        
        if (strpos($code, "\x00") !== false) {
            $this->logError('Instrument code contains null byte');
            return false;
        }
        
        if (preg_match('/[<>\'"\\\\;]/u', $code)) {
            $this->logError('Instrument code contains dangerous characters');
            return false;
        }
        
        return true;
    }
    
    /**
     * Добавить крошки сектора если данные доступны
     * 
     * @param array<string, mixed> $row Данные из БД
     * @return void
     */
    private function addSectorCrumbsIfAvailable(array $row): void
    {
        if ($this->isValidString($row, 'SECTOR_NAME') && $this->isValidString($row, 'SECTOR_ECONOMIKI_URL')) {
            $this->addCrumb('Сектора экономики', $this->baseUrl . '/sectors/');
            $this->addCrumb($row['SECTOR_NAME'], $this->baseUrl . '/' . $row['SECTOR_ECONOMIKI_URL'] . '/');
        }
    }
    
    /**
     * Добавить крошку эмитента если данные доступны
     * 
     * @param array<string, mixed> $row Данные из БД
     * @return void
     */
    private function addEmitentCrumbIfAvailable(array $row): void
    {
        if ($this->isValidString($row, 'EMITENT_SHORT_NAME') && $this->isValidString($row, 'EMITENT_URL')) {
            $this->addCrumb($row['EMITENT_SHORT_NAME'], $this->baseUrl . '/' . $row['EMITENT_URL'] . '/');
        }
    }
    
    /**
     * Построить название крошки для инструмента на основе группы
     * 
     * @param array<string, mixed> $row Данные инструмента из БД
     * @return string Форматированное название
     */
    private function buildInstrumentCrumbName(array $row): string
    {
        $supertypeId = (int)($row['Instrument_SUPERTYPE_ID'] ?? 0);
        
        if (!isset(self::$instrumentTypes[$supertypeId])) {
            return $this->isValidString($row, 'Instrument_TRADE_CODE') 
                ? trim($row['Instrument_TRADE_CODE']) 
                : 'Инструмент';
        }
        
        $group = self::$instrumentTypes[$supertypeId]['group'];
        
        return $group === self::PAGE_TYPE_DEBT
            ? $this->buildDebtCrumbName($row)
            : $this->buildEquityCrumbName($row);
    }
    
    /**
     * Построить название для ДОЛЕВЫХ инструментов
     * 
     * @param array<string, mixed> $row Данные инструмента
     * @return string Форматированное название
     */
    private function buildEquityCrumbName(array $row): string
    {
        $parts = array_filter([
            $this->isValidString($row, 'Instrument_TYPE') ? trim($row['Instrument_TYPE']) : null,
            $this->isValidString($row, 'Instrument_TRADE_CODE') ? trim($row['Instrument_TRADE_CODE']) : null,
            $this->isValidString($row, 'QuikName') ? trim($row['QuikName']) : null
        ]);
        
        if (empty($parts)) {
            return $this->isValidString($row, 'Instrument_TRADE_CODE') 
                ? trim($row['Instrument_TRADE_CODE']) 
                : 'Инструмент';
        }
        
        return $this->truncateIfNeeded(implode(' ', $parts));
    }
    
    /**
     * Построить название для ДОЛГОВЫХ инструментов
     * 
     * @param array<string, mixed> $row Данные инструмента
     * @return string Форматированное название
     */
    private function buildDebtCrumbName(array $row): string
    {
        $parts = [];
        
        if ($this->isValidString($row, 'QuikName')) {
            $parts[] = trim($row['QuikName']);
        } elseif ($this->isValidString($row, 'Instrument_TYPE')) {
            $this->logError('QUIK data missing for debt instrument');
            $parts[] = trim($row['Instrument_TYPE']);
        }
        
        if ($this->isValidString($row, 'Instrument_TRADE_CODE')) {
            $parts[] = trim($row['Instrument_TRADE_CODE']);
        }
        
        if (empty($parts)) {
            return $this->isValidString($row, 'Instrument_TRADE_CODE') 
                ? trim($row['Instrument_TRADE_CODE']) 
                : 'Инструмент';
        }
        
        return $this->truncateIfNeeded(implode(' ', $parts));
    }
    
    /**
     * Обрезать строку если она превышает максимальную длину
     * 
     * @param string $text Текст для обрезки
     * @return string Обрезанный текст с многоточием или оригинал
     */
    private function truncateIfNeeded(string $text): string
    {
        $length = mb_strlen($text, 'UTF-8');
        
        if ($length <= self::MAX_CRUMB_NAME_LENGTH) {
            return $text;
        }
        
        $maxLength = self::MAX_CRUMB_NAME_LENGTH - self::ELLIPSIS_LENGTH;
        return mb_substr($text, 0, $maxLength, 'UTF-8') . '...';
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // ПОСТРОЕНИЕ BREADCRUMBS ДЛЯ СЕКТОРОВ И ЭМИТЕНТОВ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Построить breadcrumb цепочку для страницы сектора или эмитента
     * 
     * @param string $slug URL slug
     * @return void
     */
    private function buildDynamicBreadcrumb(string $slug): void
    {
        // Валидация slug
        if (!$this->isValidSlug($slug)) {
            return;
        }
        
        // Попытка найти сектор
        $cacheKey = "sector_{$slug}";
        $sectorData = $this->getCachedOrFetch($cacheKey, function() use ($slug) {
            return $this->fetchSectorData($slug);
        });
        
        if (is_array($sectorData)) {
            $this->pageType = self::PAGE_TYPE_SECTOR;
            $this->addCrumb('Сектора экономики', $this->baseUrl . '/sectors/');
            
            if ($this->isValidString($sectorData, 'SECTOR_NAME')) {
                $this->addCrumb($sectorData['SECTOR_NAME'], $this->baseUrl . '/' . $slug . '/');
            }
            
            return;
        }
        
        // Попытка найти эмитента
        $cacheKey = "emitent_{$slug}";
        $emitentData = $this->getCachedOrFetch($cacheKey, function() use ($slug) {
            return $this->fetchEmitentData($slug);
        });
        
        if (is_array($emitentData)) {
            $this->pageType = self::PAGE_TYPE_EMITENT;
            $this->addSectorCrumbsIfAvailable($emitentData);
            
            if ($this->isValidString($emitentData, 'EMITENT_SHORT_NAME')) {
                $this->addCrumb($emitentData['EMITENT_SHORT_NAME'], $this->baseUrl . '/' . $slug . '/');
            }
        }
    }
    
    /**
     * Валидация URL slug
     * 
     * @param string $slug Slug для проверки
     * @return bool
     */
    private function isValidSlug(string $slug): bool
    {
        $slugLength = mb_strlen($slug, 'UTF-8');
        
        if ($slugLength === 0 || $slugLength > 255) {
            $this->logError('Invalid slug length');
            return false;
        }
        
        if (strpos($slug, "\x00") !== false) {
            $this->logError('Slug contains null byte');
            return false;
        }
        
        // Улучшенная UTF-8 валидация
        if (!mb_check_encoding($slug, 'UTF-8')) {
            $this->logError('Slug is not valid UTF-8');
            return false;
        }
        
        // Разрешены: латиница, кириллица, цифры, дефис
        if (!preg_match('/^[\p{L}\p{N}\-]+$/u', $slug)) {
            $this->logError('Slug contains invalid characters');
            return false;
        }
        
        return true;
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // РАБОТА С БАЗОЙ ДАННЫХ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Получить данные инструмента из БД
     * 
     * @param string $code Торговый код или ISIN
     * @param int $supertypeId ID типа инструмента
     * @param bool $useQuik Включать ли данные из QUIK таблиц
     * @return array<string, mixed>|false Данные инструмента или false
     */
    private function fetchInstrumentData(string $code, int $supertypeId, bool $useQuik = true)
    {
        if (!isset(self::$instrumentTypes[$supertypeId])) {
            return false;
        }
        
        $typeConfig = self::$instrumentTypes[$supertypeId];
        
        // Базовый SQL
        $baseSql = "SELECT 
                    i.Instrument_TRADE_CODE,
                    i.Instrument_TYPE,
                    i.Instrument_SUPERTYPE_ID,
                    e.EMITENT_SHORT_NAME,
                    e.EMITENT_URL,
                    s.SECTOR_NAME,
                    s.SECTOR_ECONOMIKI_URL
                FROM Instrument i
                LEFT JOIN Emitent e ON i.Emitent_Id = e.Id
                LEFT JOIN SECTOR_ECONOMIKI s ON e.ID_SECTOR_ECONOMIKI = s.Id
                WHERE (i.Instrument_TRADE_CODE = ? OR i.Instrument_ISIN = ?)
                AND i.Instrument_SUPERTYPE_ID = ?
                LIMIT 1";
        
        $sql = $baseSql;
        
        // Добавление QUIK данных если требуется
        if ($useQuik) {
            $quikTable = $this->validateQuikTable($typeConfig['quik_table']);
            
            if ($quikTable) {
                $quikField = $this->getQuikFieldName($supertypeId);
                
                if ($quikField) {
                    $sql = "SELECT 
                                i.Instrument_TRADE_CODE,
                                i.Instrument_TYPE,
                                i.Instrument_SUPERTYPE_ID,
                                e.EMITENT_SHORT_NAME,
                                e.EMITENT_URL,
                                s.SECTOR_NAME,
                                s.SECTOR_ECONOMIKI_URL,
                                q.{$quikField} AS QuikName
                            FROM Instrument i
                            LEFT JOIN Emitent e ON i.Emitent_Id = e.Id
                            LEFT JOIN SECTOR_ECONOMIKI s ON e.ID_SECTOR_ECONOMIKI = s.Id
                            LEFT JOIN {$quikTable} q ON (q.SecCode = i.Instrument_TRADE_CODE OR q.ISIN = i.Instrument_ISIN)
                            WHERE (i.Instrument_TRADE_CODE = ? OR i.Instrument_ISIN = ?)
                            AND i.Instrument_SUPERTYPE_ID = ?
                            LIMIT 1";
                }
            }
        }
        
        // Выполнение запроса с fallback
        try {
            $stmt = $this->db->prepare($sql);
            
            if (!$stmt) {
                if ($useQuik) {
                    return $this->fetchInstrumentData($code, $supertypeId, false);
                }
                return false;
            }
            
            if (!$stmt->bind_param("ssi", $code, $code, $supertypeId)) {
                $stmt->close();
                return false;
            }
            
            if (!$stmt->execute()) {
                $stmt->close();
                
                if ($useQuik) {
                    return $this->fetchInstrumentData($code, $supertypeId, false);
                }
                
                return false;
            }
            
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            
            return $row ?: false;
            
        } catch (mysqli_sql_exception $e) {
            $this->logError('Database error: ' . $e->getMessage());
            
            if ($useQuik) {
                return $this->fetchInstrumentData($code, $supertypeId, false);
            }
            
            return false;
        }
    }
    
    /**
     * Получить имя поля из QUIK таблицы
     * 
     * ЛОГИКА v6.1.1:
     *   - QUIK_Akcii (types 1,2,4): используем SecName
     *   - QUIK_Obligacii (type 6): используем ShortName
     *   - QUIK_Obligacii (types 3,5): используем SecName
     * 
     * @param int $supertypeId ID типа инструмента
     * @return string|false Имя поля или false
     */
    private function getQuikFieldName(int $supertypeId)
    {
        // Для облигаций (type 6) используем ShortName
        if ($supertypeId === self::INSTRUMENT_TYPE_BONDS) {
            return 'ShortName';
        }
        
        // Для всех остальных типов (1,2,3,4,5) используем SecName
        return 'SecName';
    }
    
    /**
     * Валидация имени QUIK таблицы
     * 
     * ВАЖНО v6.1.1: В БД существуют только 2 QUIK таблицы:
     *   - QUIK_Akcii: для equity инструментов (types 1, 2, 4)
     *   - QUIK_Obligacii: для debt инструментов (types 3, 5, 6)
     * 
     * @param string $tableName Имя таблицы
     * @return string|false Валидное имя или false
     */
    private function validateQuikTable(string $tableName)
    {
        // Только 2 реально существующие таблицы
        $allowedTables = [
            'QUIK_Akcii',      // Types 1, 2, 4: акции, расписки, паи
            'QUIK_Obligacii'   // Types 3, 5, 6: еврооблигации, ипотечные, облигации
        ];
        
        if (!in_array($tableName, $allowedTables, true)) {
            $this->logError('Invalid QUIK table name: ' . $tableName);
            return false;
        }
        
        // Проверка существования таблицы в БД
        try {
            $result = $this->db->query("SHOW TABLES LIKE '{$tableName}'");
            
            if (!$result || $result->num_rows === 0) {
                $this->logError('QUIK table does not exist: ' . $tableName);
                return false;
            }
            
            return $tableName;
            
        } catch (mysqli_sql_exception $e) {
            $this->logError('Error checking QUIK table: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Получить данные сектора из БД
     * 
     * @param string $slug URL slug сектора
     * @return array<string, mixed>|false Данные сектора или false
     */
    private function fetchSectorData(string $slug)
    {
        $sql = "SELECT Id, SECTOR_NAME, SECTOR_ECONOMIKI_URL 
                FROM SECTOR_ECONOMIKI 
                WHERE SECTOR_ECONOMIKI_URL = ? 
                LIMIT 1";
        
        try {
            $stmt = $this->db->prepare($sql);
            
            if (!$stmt) {
                return false;
            }
            
            $stmt->bind_param("s", $slug);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            
            return $row ?: false;
            
        } catch (mysqli_sql_exception $e) {
            $this->logError('Database error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Получить данные эмитента из БД
     * 
     * @param string $slug URL slug эмитента
     * @return array<string, mixed>|false Данные эмитента или false
     */
    private function fetchEmitentData(string $slug)
    {
        $sql = "SELECT e.EMITENT_SHORT_NAME, e.EMITENT_URL, s.SECTOR_NAME, s.SECTOR_ECONOMIKI_URL
                FROM Emitent e
                LEFT JOIN SECTOR_ECONOMIKI s ON e.ID_SECTOR_ECONOMIKI = s.Id
                WHERE e.EMITENT_URL = ? 
                LIMIT 1";
        
        try {
            $stmt = $this->db->prepare($sql);
            
            if (!$stmt) {
                return false;
            }
            
            $stmt->bind_param("s", $slug);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            
            return $row ?: false;
            
        } catch (mysqli_sql_exception $e) {
            $this->logError('Database error: ' . $e->getMessage());
            return false;
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // THREAD-SAFE КЭШИРОВАНИЕ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Получить блокировку кэша для thread-safety
     * 
     * @return bool Успешность получения блокировки
     */
    private function acquireCacheLock(): bool
    {
        if (self::$cacheMutex === false || self::$cacheMutex === null) {
            return false;
        }
        
        return @flock(self::$cacheMutex, LOCK_EX);
    }
    
    /**
     * Освободить блокировку кэша
     * 
     * @return bool Успешность освобождения
     */
    private function releaseCacheLock(): bool
    {
        if (self::$cacheMutex === false || self::$cacheMutex === null) {
            return false;
        }
        
        return @flock(self::$cacheMutex, LOCK_UN);
    }
    
    /**
     * Получить данные из кэша или выполнить callback
     * 
     * @param string $key Ключ кэша
     * @param callable $fetchCallback Функция для получения данных
     * @return mixed Данные из кэша или результат callback
     */
    private function getCachedOrFetch(string $key, callable $fetchCallback)
    {
        $cached = $this->getCachedData($key);
        
        if ($cached === null) {
            $data = $fetchCallback();
            $this->setCachedData($key, $data ?: self::CACHE_NOT_FOUND);
            return $data;
        }
        
        if ($cached === self::CACHE_NOT_FOUND) {
            return false;
        }
        
        return $cached;
    }
    
    /**
     * Получить данные из кэша с thread-safety
     * 
     * @param string $key Ключ кэша
     * @return mixed|null Данные или null если не найдено
     */
    private function getCachedData(string $key)
    {
        if (!self::$cacheEnabled) {
            self::$cacheMisses++;
            return null;
        }
        
        $this->acquireCacheLock();
        
        try {
            if (isset(self::$queryCache[$key])) {
                $cached = self::$queryCache[$key];
                
                // Проверка версии кэша
                if (!isset($cached['version']) || $cached['version'] !== self::CACHE_VERSION) {
                    unset(self::$queryCache[$key]);
                    self::$cacheMisses++;
                    return null;
                }
                
                // Проверка TTL
                if ($cached['expires'] > time()) {
                    self::$cacheHits++;
                    return $cached['data'];
                }
                
                // Удаление устаревших данных
                unset(self::$queryCache[$key]);
            }
            
            self::$cacheMisses++;
            return null;
            
        } finally {
            $this->releaseCacheLock();
        }
    }
    
    /**
     * Сохранить данные в кэш с thread-safety
     * 
     * @param string $key Ключ кэша
     * @param mixed $data Данные для сохранения
     * @return void
     */
    private function setCachedData(string $key, $data): void
    {
        if (!self::$cacheEnabled) {
            return;
        }
        
        $this->acquireCacheLock();
        
        try {
            // Ограничение кэша NOT_FOUND записей
            if ($data === self::CACHE_NOT_FOUND) {
                if (self::$notFoundCount >= self::MAX_NOT_FOUND_CACHE) {
                    return;
                }
                self::$notFoundCount++;
            }
            
            // Очистка старого кэша при достижении лимита
            if (count(self::$queryCache) >= self::MAX_CACHE_SIZE) {
                $this->evictOldestCacheEntries();
            }
            
            // Сохранение в кэш
            self::$queryCache[$key] = [
                'version' => self::CACHE_VERSION,
                'data' => $data,
                'expires' => time() + self::CACHE_TTL,
                'created' => time()
            ];
            
        } finally {
            $this->releaseCacheLock();
        }
    }
    
    /**
     * Удалить старые записи из кэша
     * 
     * @return void
     */
    private function evictOldestCacheEntries(): void
    {
        // Сортировка по времени создания
        uasort(self::$queryCache, function($a, $b) {
            return ($a['created'] ?? 0) <=> ($b['created'] ?? 0);
        });
        
        $toRemove = (int)(self::MAX_CACHE_SIZE * self::CACHE_EVICTION_PERCENTAGE);
        $removed = 0;
        
        foreach (self::$queryCache as $key => $value) {
            if ($removed >= $toRemove) {
                break;
            }
            
            unset(self::$queryCache[$key]);
            $removed++;
        }
    }
    
    /**
     * Проверить здоровье кэша
     * 
     * @return array<string, mixed> Статус кэша
     */
    public function isCacheHealthy(): array
    {
        $total = self::$cacheHits + self::$cacheMisses;
        $hitRate = $total > 0 ? round((self::$cacheHits / $total) * 100, 2) : 0;
        $cacheSize = count(self::$queryCache);
        $cacheUtilization = self::MAX_CACHE_SIZE > 0 
            ? round(($cacheSize / self::MAX_CACHE_SIZE) * 100, 2) 
            : 0;
        
        $isHealthy = $hitRate >= 30 && $cacheUtilization < 90;
        
        return [
            'healthy' => $isHealthy,
            'hit_rate_percent' => $hitRate,
            'cache_size' => $cacheSize,
            'max_cache_size' => self::MAX_CACHE_SIZE,
            'utilization_percent' => $cacheUtilization,
            'mutex_available' => (self::$cacheMutex !== false && self::$cacheMutex !== null),
            'recommendations' => $this->getCacheRecommendations($hitRate, $cacheUtilization)
        ];
    }
    
    /**
     * Получить рекомендации по кэшу
     * 
     * @param float $hitRate Hit rate в процентах
     * @param float $utilization Использование кэша в процентах
     * @return array<string> Массив рекомендаций
     */
    private function getCacheRecommendations(float $hitRate, float $utilization): array
    {
        $recommendations = [];
        
        if ($hitRate < 30) {
            $recommendations[] = 'Low cache hit rate - consider increasing CACHE_TTL';
        }
        
        if ($utilization > 90) {
            $recommendations[] = 'Cache almost full - consider increasing MAX_CACHE_SIZE';
        }
        
        if (self::$cacheMutex === false || self::$cacheMutex === null) {
            $recommendations[] = 'Cache mutex unavailable - thread-safety disabled';
        }
        
        if (empty($recommendations)) {
            $recommendations[] = 'Cache is healthy';
        }
        
        return $recommendations;
    }
    
    /**
     * Очистить кэш
     * 
     * @param bool $resetStats Сбросить статистику кэша
     * @return void
     */
    public static function clearCache(bool $resetStats = false): void
    {
        if (self::$cacheMutex !== false && self::$cacheMutex !== null) {
            @flock(self::$cacheMutex, LOCK_EX);
        }
        
        try {
            self::$queryCache = [];
            self::$notFoundCount = 0;
            self::$lastCacheCleanup = time();
            
            if ($resetStats) {
                self::$cacheHits = 0;
                self::$cacheMisses = 0;
            }
        } finally {
            if (self::$cacheMutex !== false && self::$cacheMutex !== null) {
                @flock(self::$cacheMutex, LOCK_UN);
            }
        }
    }
    
    /**
     * Получить статистику кэша
     * 
     * @return array<string, int|float> Статистика
     */
    public static function getCacheStats(): array
    {
        $total = self::$cacheHits + self::$cacheMisses;
        $hitRate = $total > 0 ? round((self::$cacheHits / $total) * 100, 2) : 0;
        
        return [
            'hits' => self::$cacheHits,
            'misses' => self::$cacheMisses,
            'total_requests' => $total,
            'hit_rate_percent' => $hitRate,
            'cache_size' => count(self::$queryCache),
            'not_found_count' => self::$notFoundCount,
            'last_cleanup' => self::$lastCacheCleanup
        ];
    }
    
    /**
     * Включить/отключить кэширование
     * 
     * @param bool $enabled
     * @return void
     */
    public static function setCacheEnabled(bool $enabled): void
    {
        self::$cacheEnabled = $enabled;
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // ВСПОМОГАТЕЛЬНЫЕ МЕТОДЫ
    // ═══════════════════════════════════════════════════════════════════════
    
    /**
     * Добавить крошку в цепочку
     * 
     * @param string $name Название крошки
     * @param string $url URL крошки
     * @return void
     * @throws InvalidArgumentException Если параметры некорректны
     */
    private function addCrumb(string $name, string $url): void
    {
        // Проверка глубины
        if (count($this->breadcrumbs) >= self::MAX_BREADCRUMB_DEPTH) {
            throw new RuntimeException(
                'Breadcrumb depth limit reached: ' . self::MAX_BREADCRUMB_DEPTH
            );
        }
        
        // Проверка что URL принадлежит сайту
        if (strpos($url, $this->baseUrl) !== 0) {
            throw new InvalidArgumentException('Breadcrumb URL must start with base URL: ' . $url);
        }
        
        // Проверка длины URL
        if (strlen($url) > self::MAX_URL_LENGTH) {
            throw new InvalidArgumentException('Breadcrumb URL too long: ' . strlen($url));
        }
        
        // Обрезка названия если слишком длинное
        if (mb_strlen($name, 'UTF-8') > self::MAX_CRUMB_NAME_LENGTH) {
            $name = $this->truncateIfNeeded($name);
        }
        
        // Проверка что название не пустое
        $trimmedName = trim($name);
        if ($trimmedName === '' || $trimmedName === '...') {
            throw new InvalidArgumentException('Breadcrumb name cannot be empty or only ellipsis');
        }
        
        // Проверка дублирования URL
        if (isset($this->usedUrls[$url])) {
            return; // Тихо пропускаем дубликаты
        }
        
        // Добавление крошки
        $this->breadcrumbs[] = [
            'name' => $name,
            'url' => $url
        ];
        
        $this->usedUrls[$url] = true;
        $this->lastUrl = $url;
    }
    
    /**
     * Проверка что значение массива является валидной строкой
     * 
     * @param array<string, mixed> $row Массив данных
     * @param string $key Ключ для проверки
     * @return bool
     */
    private function isValidString(array $row, string $key): bool
    {
        return isset($row[$key]) 
            && $row[$key] !== null 
            && is_string($row[$key]) 
            && $row[$key] !== '';
    }
    
    /**
     * Логирование ошибок с rate limiting и отложенным форматированием
     * 
     * @param string $message Сообщение об ошибке
     * @return void
     */
    private function logError(string $message): void
    {
        $currentTime = time();
        
        // Сброс счётчика каждую минуту
        if ($currentTime > self::$errorLogResetTime) {
            self::$errorLogCount = 0;
            self::$errorLogResetTime = $currentTime + 60;
        }
        
        // Rate limiting
        if (self::$errorLogCount >= self::MAX_LOG_ERRORS_PER_MINUTE) {
            return;
        }
        
        self::$errorLogCount++;
        
        // Отложенное форматирование - конкатенация только если логирование разрешено
        @error_log('BreadcrumbBuilder v6.2.0: ' . $message);
    }
    
    /**
     * Деструктор - освобождение ресурсов
     * 
     * @return void
     */
    public function __destruct()
    {
        // Освобождение мьютекса если был захвачен
        if (self::$cacheMutex !== false && self::$cacheMutex !== null) {
            @flock(self::$cacheMutex, LOCK_UN);
        }
    }
}