<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * МОДУЛЬ ОПРЕДЕЛЕНИЯ ТИПА СТРАНИЦЫ И ПОЛУЧЕНИЯ МЕТА-ДАННЫХ
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Назначение:
 *   - Определяет тип запрашиваемой страницы на основе URL
 *   - Получает данные из БД для каждого типа страницы
 *   - Формирует SEO мета-данные (title, description, keywords, Open Graph)
 * 
 * Поддерживаемые типы страниц:
 *   1. BOND - Облигации (/bonds/RU000A102VD0/)
 *   2. STOCK - Акции (/stocks/GAZP/)
 *   3. RECEIPT - Депозитарные расписки (/receipts/US12345/)
 *   4. EUROBOND - Еврооблигации (/eurobonds/XS12345/)
 *   5. FUND - Инвестиционные паи (/funds/1234-56/)
 *   6. MORTGAGE_NOTE - Ипотечные сертификаты (/mortgage-notes/5678/)
 *   7. CATALOG_STOCKS - Каталог акций (/stocks/)
 *   8. MENU - Страницы из таблицы MENU (/bonds/, /about/, и т.д.)
 *   9. SECTOR - Сектора экономики (/agro-sector/)
 *   10. EMITENT - Эмитенты (/gazprom/)
 * 
 * Используемые таблицы БД:
 *   - Instrument - данные инструментов
 *   - QUIK_Obligacii, QUIK_Akcii, QUIK_Receipts и т.д. - текущие данные из QUIK
 *   - Emitent - данные эмитентов
 *   - SECTOR_ECONOMIKI - данные секторов экономики
 *   - MENU - страницы меню
 * 
 * Безопасность:
 *   - Все SQL запросы используют prepared statements для защиты от инъекций
 * 
 * @author DeepMax Development Team
 * @version 2.1.1 RESTORED (FIX: Восстановлена рабочая логика - /stocks/ через MENU)
 * ═══════════════════════════════════════════════════════════════════════════
 */

/**
 * ───────────────────────────────────────────────────────────────────────────
 * ФУНКЦИЯ: getPageTypeAndMeta()
 * ───────────────────────────────────────────────────────────────────────────
 * 
 * Определяет тип страницы и получает необходимые данные из БД
 * 
 * @param array $url_parts - Массив частей URL [0 => '', 1 => 'bonds', 2 => 'RU000...']
 * @return array [type, instrument, security, emitent] или [type, menu_object]
 * 
 * Структура возвращаемого массива:
 *   Для инструментов: ['BOND', Instrument Object, Security Object, Emitent Object]
 *   Для меню: ['MENU', Menu Object]
 *   Для сектора: ['SECTOR', Sector Object]
 *   Для эмитента: ['EMITENT', Emitent Object]
 *   Для каталога акций: ['CATALOG_STOCKS', null]
 * ───────────────────────────────────────────────────────────────────────────
 */
function getPageTypeAndMeta($url_parts){
   global $db;

   // ═══════════════════════════════════════════════════════════════════════
   // 6. ОБЛИГАЦИИ (BONDS)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /bonds/RU000A102VD0/
   // Таблица QUIK: QUIK_Obligacii
   // Instrument SUPERTYPE_ID: 6
   // Поиск: только по ISIN (оригинальная логика)
   // ═══════════════════════════════════════════════════════════════════════
   if($url_parts[1] == 'bonds' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      // Получаем данные инструмента из таблицы Instrument
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      }
      
      // Получаем текущие данные из QUIK (цены, параметры)
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Obligacii WHERE ISIN=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      // Получаем данные эмитента (по LIKE - может быть несколько ISIN)
      $emitent = [];
      $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE ISIN LIKE ? LIMIT 1');
      $like_pattern = '%' . $url_parts[2] . '%';
      $stmt->bind_param("s", $like_pattern);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/emitent.php');
         $emitent = $result->fetch_object('Emitent');
      }
      
      return ['BOND', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // 1. АКЦИИ (STOCKS)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /stocks/GAZP/
   // Таблица QUIK: QUIK_Akcii
   // Instrument SUPERTYPE_ID: 1
   // Поиск: сначала по TRADE_CODE, если не найдено - по ISIN
   // ═══════════════════════════════════════════════════════════════════════
   elseif($url_parts[1] == 'stocks' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_TRADE_CODE=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      } else {
         $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
         $stmt->bind_param("s", $url_parts[2]);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/instrument.php');
            $instrument = $result->fetch_object('Instrument');
         }
      }
      
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Akcii WHERE SecCode=? OR ISIN=? LIMIT 1');
      $stmt->bind_param("ss", $url_parts[2], $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      $emitent = [];
      if(!empty($instrument) && !empty($instrument->Emitent_Id)){
         $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE Id=? LIMIT 1');
         $stmt->bind_param("i", $instrument->Emitent_Id);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/emitent.php');
            $emitent = $result->fetch_object('Emitent');
         }
      }
      
      return ['STOCK', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // 2. ДЕПОЗИТАРНЫЕ РАСПИСКИ (RECEIPTS)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /receipts/US12345/
   // Таблица QUIK: QUIK_Akcii
   // Instrument SUPERTYPE_ID: 2
   // Поиск: сначала по TRADE_CODE, если не найдено - по ISIN
   // ═══════════════════════════════════════════════════════════════════════
   elseif($url_parts[1] == 'receipts' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_TRADE_CODE=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      } else {
         $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
         $stmt->bind_param("s", $url_parts[2]);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/instrument.php');
            $instrument = $result->fetch_object('Instrument');
         }
      }
      
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Akcii WHERE SecCode=? OR ISIN=? LIMIT 1');
      $stmt->bind_param("ss", $url_parts[2], $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      $emitent = [];
      if(!empty($instrument) && !empty($instrument->Emitent_Id)){
         $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE Id=? LIMIT 1');
         $stmt->bind_param("i", $instrument->Emitent_Id);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/emitent.php');
            $emitent = $result->fetch_object('Emitent');
         }
      }
      
      return ['RECEIPT', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // 3. ЕВРООБЛИГАЦИИ (EUROBONDS)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /eurobonds/XS12345/
   // Таблица QUIK: QUIK_Obligacii
   // Instrument SUPERTYPE_ID: 3
   // Поиск: сначала по TRADE_CODE, если не найдено - по ISIN
   // ═══════════════════════════════════════════════════════════════════════
   elseif($url_parts[1] == 'eurobonds' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_TRADE_CODE=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      } else {
         $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
         $stmt->bind_param("s", $url_parts[2]);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/instrument.php');
            $instrument = $result->fetch_object('Instrument');
         }
      }
      
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Obligacii WHERE SecCode=? OR ISIN=? LIMIT 1');
      $stmt->bind_param("ss", $url_parts[2], $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      $emitent = [];
      if(!empty($instrument) && !empty($instrument->Emitent_Id)){
         $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE Id=? LIMIT 1');
         $stmt->bind_param("i", $instrument->Emitent_Id);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/emitent.php');
            $emitent = $result->fetch_object('Emitent');
         }
      }
      
      return ['EUROBOND', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // 4. ИНВЕСТИЦИОННЫЕ ПАИ (FUNDS)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /funds/1234-56/
   // Таблица QUIK: QUIK_Akcii
   // Instrument SUPERTYPE_ID: 4
   // Поиск: сначала по TRADE_CODE, если не найдено - по ISIN
   // ═══════════════════════════════════════════════════════════════════════
   elseif($url_parts[1] == 'funds' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_TRADE_CODE=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      } else {
         $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
         $stmt->bind_param("s", $url_parts[2]);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/instrument.php');
            $instrument = $result->fetch_object('Instrument');
         }
      }
      
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Akcii WHERE SecCode=? OR ISIN=? LIMIT 1');
      $stmt->bind_param("ss", $url_parts[2], $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      $emitent = [];
      if(!empty($instrument) && !empty($instrument->Emitent_Id)){
         $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE Id=? LIMIT 1');
         $stmt->bind_param("i", $instrument->Emitent_Id);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/emitent.php');
            $emitent = $result->fetch_object('Emitent');
         }
      }
      
      return ['FUND', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // 5. ИПОТЕЧНЫЕ СЕРТИФИКАТЫ (MORTGAGE_NOTES)
   // ═══════════════════════════════════════════════════════════════════════
   // URL: /mortgage-notes/5678/
   // Таблица QUIK: QUIK_Obligacii
   // Instrument SUPERTYPE_ID: 5
   // Поиск: сначала по TRADE_CODE, если не найдено - по ISIN
   // ═══════════════════════════════════════════════════════════════════════
   elseif($url_parts[1] == 'mortgage-notes' && isset($url_parts[2]) && !empty($url_parts[2])){
      
      $instrument = [];
      $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_TRADE_CODE=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/instrument.php');
         $instrument = $result->fetch_object('Instrument');
      } else {
         $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=? LIMIT 1');
         $stmt->bind_param("s", $url_parts[2]);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/instrument.php');
            $instrument = $result->fetch_object('Instrument');
         }
      }
      
      $security = [];
      $stmt = $db->connection->prepare('SELECT * FROM QUIK_Obligacii WHERE SecCode=? OR ISIN=? LIMIT 1');
      $stmt->bind_param("ss", $url_parts[2], $url_parts[2]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/security.php');
         $security = $result->fetch_object('Security');
      }
      
      $emitent = [];
      if(!empty($instrument) && !empty($instrument->Emitent_Id)){
         $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE Id=? LIMIT 1');
         $stmt->bind_param("i", $instrument->Emitent_Id);
         $stmt->execute();
         $result = $stmt->get_result();
         if ($result->num_rows > 0) {
            require_once('DB/classes/emitent.php');
            $emitent = $result->fetch_object('Emitent');
         }
      }
      
      return ['MORTGAGE_NOTE', $instrument, $security, $emitent];
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // ЕСЛИ НЕ ИНСТРУМЕНТ - ПРОВЕРЯЕМ ОСТАЛЬНЫЕ ТИПЫ СТРАНИЦ
   // ═══════════════════════════════════════════════════════════════════════
   // Если URL не соответствует ни одному финансовому инструменту,
   // проверяем стандартные типы страниц: MENU, SECTOR, EMITENT
   // ═══════════════════════════════════════════════════════════════════════
   else{
      
      
      // ─────────────────────────────────────────────────────────────────────
      // MENU - Страницы из таблицы MENU
      // ─────────────────────────────────────────────────────────────────────
      // Примеры: /bonds/, /about/, /contacts/
      // ─────────────────────────────────────────────────────────────────────
      $stmt = $db->connection->prepare('SELECT * FROM MENU WHERE MENU_url=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[1]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/menu.php');
         return ['MENU', $result->fetch_object('Menu')];
      }

      // ─────────────────────────────────────────────────────────────────────
      // SECTOR - Страница сектора экономики
      // ─────────────────────────────────────────────────────────────────────
      // Примеры: /agro-sector/, /it-sector/
      // ─────────────────────────────────────────────────────────────────────
      $stmt = $db->connection->prepare('SELECT * FROM SECTOR_ECONOMIKI WHERE SECTOR_ECONOMIKI_URL=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[1]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/sector.php');
         return ['SECTOR', $result->fetch_object('Sector')];
      }

      // ─────────────────────────────────────────────────────────────────────
      // EMITENT - Страница эмитента
      // ─────────────────────────────────────────────────────────────────────
      // Примеры: /gazprom/, /sberbank/
      // ─────────────────────────────────────────────────────────────────────
      $stmt = $db->connection->prepare('SELECT * FROM Emitent WHERE EMITENT_URL=? LIMIT 1');
      $stmt->bind_param("s", $url_parts[1]);
      $stmt->execute();
      $result = $stmt->get_result();
      if ($result->num_rows > 0) {
         require_once('DB/classes/emitent.php');
         return ['EMITENT', $result->fetch_object('Emitent')];
      }
   }

   // ═══════════════════════════════════════════════════════════════════════
   // СТРАНИЦА НЕ НАЙДЕНА
   // ═══════════════════════════════════════════════════════════════════════
   // Если ни один тип не подошёл, возвращаем [null, null]
   // Это приведёт к отображению 404 страницы
   // ═══════════════════════════════════════════════════════════════════════
   return [null, null];
}

/**
 * ───────────────────────────────────────────────────────────────────────────
 * ФУНКЦИЯ: setPageVariablesAndGetPageTypeAndMeta()
 * ───────────────────────────────────────────────────────────────────────────
 * 
 * Заполняет глобальные переменные мета-данными на основе типа страницы
 * 
 * @param array $url_parts - Массив частей URL
 * @return array - Данные страницы от getPageTypeAndMeta()
 * 
 * Заполняемые глобальные переменные:
 *   - $title - HTML title страницы
 *   - $og_title - Open Graph title
 *   - $og_description - Open Graph description
 *   - $og_image - Open Graph image URL
 *   - $og_image_alt - Open Graph image alt text
 *   - $og_url - Open Graph URL (относительный)
 *   - $description - HTML meta description
 *   - $keywords - HTML meta keywords
 * ───────────────────────────────────────────────────────────────────────────
 */
function setPageVariablesAndGetPageTypeAndMeta($url_parts){
   global $title, $og_title, $og_description, $og_image, $og_image_alt, $og_url, $description, $keywords;

   // Получаем данные страницы
   $data = getPageTypeAndMeta($url_parts);

   // Если страница найдена, заполняем мета-данные
   if($data[0] !== null){

      // ═══════════════════════════════════════════════════════════════════
      // MENU - Страницы меню
      // ═══════════════════════════════════════════════════════════════════
      if($data[0] == 'MENU'){
         $title = $data[1]->MENU_title;
         $og_title = $data[1]->MENU_og_title;
         $og_description = $data[1]->MENU_og_description;
         $og_image = $data[1]->MENU_og_image;
         $og_image_alt = $data[1]->MENU_og_image_alt;
         $og_url = $data[1]->MENU_url;
         $description = $data[1]->MENU_description;
         $keywords = $data[1]->MENU_keywords;

      // ═══════════════════════════════════════════════════════════════════
      // SECTOR - Сектор экономики
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'SECTOR'){
         $title = $data[1]->SECTOR_ECONOMIKI_TITLE;
         $og_title = $data[1]->SECTOR_ECONOMIKI_og_title;
         $og_description = $data[1]->SECTOR_ECONOMIKI_og_description;
         $og_image = $data[1]->SECTOR_ECONOMIKI_og_image;
         $og_image_alt = $data[1]->SECTOR_ECONOMIKI_og_image_alt;
         $og_url = $data[1]->SECTOR_ECONOMIKI_URL;
         $description = $data[1]->SECTOR_ECONOMIKI_DESCRIPTION;
         $keywords = $data[1]->SECTOR_ECONOMIKI_KEYWORDS;

      // ═══════════════════════════════════════════════════════════════════
      // EMITENT - Эмитент
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'EMITENT'){
         $title = $data[1]->EMITENT_TITLE;
         $og_title = $data[1]->EMITENT_og_title;
         $og_description = $data[1]->EMITENT_og_description;
         $og_image = $data[1]->EMITENT_og_image;
         $og_image_alt = $data[1]->EMITENT_og_image_alt;
         $og_url = $data[1]->EMITENT_URL;
         $description = $data[1]->EMITENT_DESCRIPTION;
         $keywords = $data[1]->EMITENT_KEYWORDS;

      // ═══════════════════════════════════════════════════════════════════
      // BOND - Облигация
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'BOND'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'bonds/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      
      // ═══════════════════════════════════════════════════════════════════
      // STOCK - Акция
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'STOCK'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'stocks/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      
      // ═══════════════════════════════════════════════════════════════════
      // RECEIPT - Депозитарная расписка
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'RECEIPT'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'receipts/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      
      // ═══════════════════════════════════════════════════════════════════
      // EUROBOND - Еврооблигация
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'EUROBOND'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'eurobonds/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      
      // ═══════════════════════════════════════════════════════════════════
      // FUND - Инвестиционный пай
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'FUND'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'funds/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      
      // ═══════════════════════════════════════════════════════════════════
      // MORTGAGE_NOTE - Ипотечный сертификат участия
      // ═══════════════════════════════════════════════════════════════════
      }else if($data[0] == 'MORTGAGE_NOTE'){
         $title = $data[1]->Instrument_TITLE;
         $og_title = $data[1]->Instrument_og_title;
         $og_description = $data[1]->Instrument_og_description;
         $og_image = $data[1]->Instrument_og_image;
         $og_image_alt = $data[1]->Instrument_og_image_alt;
         $og_url = 'mortgage-notes/' . $data[1]->Instrument_TRADE_CODE;
         $description = $data[1]->Instrument_DESCRIPTION;
         $keywords = $data[1]->Instrument_KEYWORDS;
      }

      // ─────────────────────────────────────────────────────────────────
      // Добавляем trailing slash к относительному URL для Open Graph
      // ─────────────────────────────────────────────────────────────────
      if($og_url != ''){
         $og_url .= '/';
      }

      return $data;
   }
   
   // ═══════════════════════════════════════════════════════════════════════
   // FIX v2.1.1: Возврат [null, null] если страница не найдена
   // ═══════════════════════════════════════════════════════════════════════
   return [null, null];
}

/**
 * ───────────────────────────────────────────────────────────────────────────
 * ФУНКЦИЯ: getMenuItems()
 * ───────────────────────────────────────────────────────────────────────────
 * 
 * Получает все элементы меню из таблицы MENU
 * 
 * @return array|null - Массив объектов Menu или null если записей нет
 * 
 * Использование:
 *   $menu_items = getMenuItems();
 *   foreach($menu_items as $item) {
 *       echo $item->MENU_title;
 *   }
 * ───────────────────────────────────────────────────────────────────────────
 */
function getMenuItems(){
   require_once('DB/classes/menu.php');
   global $db;
   
   $menu_items = [];
   
   // Получаем все пункты меню с непустым URL
   $query = 'SELECT * FROM MENU WHERE NOT MENU_url=""';
   $result = $db->connection->query($query);
   
   if ($result->num_rows > 0){
      // Преобразуем каждую строку в объект Menu
      while ($menu_item = $result->fetch_object('Menu')){
         array_push($menu_items, $menu_item);
      }    
   }else{
      return null;
   }

   return $menu_items;
}

/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КОНЕЦ ФАЙЛА
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * История изменений:
 *   v1.0 - Базовая версия с поддержкой BOND, MENU, SECTOR, EMITENT
 *   v2.0 - Добавлена поддержка всех 6 типов финансовых инструментов:
 *          STOCK, RECEIPT, EUROBOND, FUND, MORTGAGE_NOTE
 *   v2.1 - Добавлены подробные комментарии и документация
 *   v2.1.1 - FIX: Добавлен return [null, null] в конце setPageVariablesAndGetPageTypeAndMeta()
 *   v2.1.1 RESTORED - FIX: Восстановлена рабочая логика (2025-10-25)
 *            - ПРОБЛЕМА: v2.2.0/v2.2.1 добавили CATALOG_STOCKS с жёстко закодированными мета-данными
 *            - РЕЗУЛЬТАТ: /stocks/ не использовал данные из таблицы MENU (Id=3)
 *            - РЕШЕНИЕ: Удалены блоки проверки и мета-данных CATALOG_STOCKS
 *            - ВОССТАНОВЛЕНО: /stocks/ теперь проходит через проверку MENU → возвращает ['MENU', Menu Object]
 *            - МЕТА-ДАННЫЕ: Title, description, keywords берутся из MENU.MENU_title, MENU.MENU_description и т.д.
 * 
 * Авторы: DeepMax Development Team
 * Дата последнего обновления: 2025-10-25
 * ═══════════════════════════════════════════════════════════════════════════
 */
?>