<?php
/**
 * ═══════════════════════════════════════════════════════════════
 * DB/instruments.php v2.0 EMITENT
 * ═══════════════════════════════════════════════════════════════
 * 
 *
 * Изменения v2.1:
 *   📊 DETAILED: Добавлены поля для детальных карточек
 *   ✨ Для акций: FaceValue, FaceUnit, LotSize, IssueSize
 *   ✨ Для облигаций: MatDate, CouponValue, CouponPeriod, NextCoupon, DaysToMatDate
 *   🔗 Совместимость с emitent.php v8.5.0 DETAILED
 *
 * Версия: 2.0 (2025-10-19)
 * 
 * Изменения v2.0:
 *   ✨ NEW: getInstrumentsByEmitentId() - получение ВСЕХ инструментов
 *   ✨ LEFT JOIN с QUIK_Akcii и QUIK_Obligacii
 *   ✨ Возвращает инструменты с QUIK данными и без
 *   📝 Комментарии на русском (ПРАВИЛО 21)
 * 
 * Функции:
 *   - getInstrumentByISIN()
 *   - getInstrumentTxtShortByISIN()
 *   - getInstrumentsByEmitentId() v2.0 (НОВАЯ)
 * 
 * ═══════════════════════════════════════════════════════════════
 */

/**
 * Получить инструмент по ISIN
 * 
 * @param string $isin - ISIN код инструмента
 * @return object|null - Объект Instrument или null
 */
function getInstrumentByISIN($isin){
   require_once('DB/classes/instrument.php');
   global $db;
   $stmt = $db->connection->prepare('SELECT * FROM Instrument WHERE Instrument_ISIN=?');
   $stmt->bind_param('s', $isin);
   $stmt->execute();
   $result = $stmt->get_result();
   if ($result->num_rows > 0){
      return $result->fetch_object('Instrument');
   }else{
      return null;
   }
}

/**
 * Получить короткое описание инструмента по ISIN
 * 
 * @param string $isin - ISIN код инструмента
 * @return string|null - Короткое описание или null
 */
function getInstrumentTxtShortByISIN($isin){
   global $db;
   $stmt = $db->connection->prepare('SELECT Instrument_txt_short FROM Instrument WHERE Instrument_ISIN=?');
   $stmt->bind_param('s', $isin);
   $stmt->execute();
   $result = $stmt->get_result();
   if ($result->num_rows > 0){
      return $result->fetch_array()[0];
   }else{
      return null;
   }
}

/**
 * Получить ВСЕ инструменты эмитента с данными из QUIK
 * 
 * v2.0 (2025-10-19) - новая функция для emitent.php v7.1.0
 * 
 * Использует LEFT JOIN чтобы получить:
 *   - Все инструменты из таблицы Instrument
 *   - Торговые данные из QUIK_Akcii (для типов 1,2,4)
 *   - Торговые данные из QUIK_Obligacii (для типов 3,5,6)
 * 
 * Возвращает массив ассоциативных массивов с полями:
 *   - Все поля из Instrument (Instrument_ISIN, Instrument_TRADE_CODE и т.д.)
 *   - Поля из QUIK с префиксами:
 *     * Akcii_* - данные из QUIK_Akcii (SecCode, ShortName, SecStatus, SecSubtypeStatic)
 *     * Obligacii_* - данные из QUIK_Obligacii (SecCode, ShortName, SecStatus, SecSubtypeStatic)
 * 
 * Проверка наличия QUIK данных:
 *   - Если Akcii_SecCode IS NOT NULL → есть данные в QUIK_Akcii
 *   - Если Obligacii_SecCode IS NOT NULL → есть данные в QUIK_Obligacii
 *   - Если оба NULL → нет данных в QUIK (раздел "Нет данных")
 * 
 * @param int $emitent_id - ID эмитента
 * @return array - Массив инструментов
 * 
 * SQL безопасность: используется prepared statement
 */
function getInstrumentsByEmitentId($emitent_id) {
    global $db;
    
    $instruments = [];
    
    // LEFT JOIN к обеим QUIK таблицам
    // Условия JOIN учитывают тип инструмента:
    //   - QUIK_Akcii: только для типов 1 (акции), 2 (расписки), 4 (паи)
    //   - QUIK_Obligacii: только для типов 3 (евробонды), 5 (ипотечные), 6 (облигации)
    $stmt = $db->connection->prepare('
        SELECT 
            i.*,
            qa.SecCode as Akcii_SecCode, 
            qa.ShortName as Akcii_ShortName, 
            qa.SecName as Akcii_SecName,
            qa.SecStatus as Akcii_SecStatus, 
            qa.SecSubtypeStatic as Akcii_SecSubtypeStatic,
            qa.ISIN as Akcii_ISIN,
            qa.FaceValue as Akcii_FaceValue,
            qa.FaceUnit as Akcii_FaceUnit,
            qa.LotSize as Akcii_LotSize,
            qa.IssueSize as Akcii_IssueSize,
            qo.SecCode as Obligacii_SecCode, 
            qo.ShortName as Obligacii_ShortName,
            qo.SecName as Obligacii_SecName,
            qo.SecStatus as Obligacii_SecStatus, 
            qo.SecSubtypeStatic as Obligacii_SecSubtypeStatic,
            qo.ISIN as Obligacii_ISIN,
            qo.MatDate as Obligacii_MatDate,
            qo.CouponValue as Obligacii_CouponValue,
            qo.CouponPeriod as Obligacii_CouponPeriod,
            qo.NextCoupon as Obligacii_NextCoupon,
            qo.DaysToMatDate as Obligacii_DaysToMatDate
        FROM Instrument i
        LEFT JOIN QUIK_Akcii qa 
            ON i.Instrument_ISIN = qa.ISIN 
            AND i.Instrument_SUPERTYPE_ID IN (1, 2, 4)
        LEFT JOIN QUIK_Obligacii qo 
            ON i.Instrument_ISIN = qo.ISIN 
            AND i.Instrument_SUPERTYPE_ID IN (3, 5, 6)
        WHERE i.Emitent_Id = ?
        ORDER BY i.Instrument_SUPERTYPE_ID, i.Instrument_TRADE_CODE
    ');
    
    $stmt->bind_param('i', $emitent_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    // Возвращаем массив ассоциативных массивов
    while ($row = $result->fetch_assoc()) {
        $instruments[] = $row;
    }
    
    return $instruments;
}

/**
 * ═══════════════════════════════════════════════════════════════
 * КОНЕЦ DB/instruments.php v2.0 EMITENT
 * ═══════════════════════════════════════════════════════════════
 * 
 * Версия: 2.0 (2025-10-19)
 * Статус: PRODUCTION READY ✅
 * 
 * Всего функций: 3
 * 
 * Для работы с инструментами:
 *   - getInstrumentByISIN() - получить инструмент по ISIN
 *   - getInstrumentTxtShortByISIN() - получить описание
 *   - getInstrumentsByEmitentId() - получить ВСЕ инструменты эмитента
 * 
 * Безопасность:
 *   ✅ Все функции используют prepared statements
 *   ✅ Защита от SQL инъекций
 *   ✅ Корректная типизация параметров
 * 
 * Использование в emitent.php:
 *   $instruments = getInstrumentsByEmitentId($emitent->Id);
 *   
 *   foreach ($instruments as $inst) {
 *       // Проверка наличия QUIK данных
 *       $hasQuikData = !empty($inst['Akcii_SecCode']) || !empty($inst['Obligacii_SecCode']);
 *       
 *       if ($hasQuikData) {
 *           // Есть торговые данные → активная карточка с ссылкой
 *       } else {
 *           // Нет торговых данных → серая карточка без ссылки
 *       }
 *   }
 * 
 * ═══════════════════════════════════════════════════════════════
 */