<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * DB/db.php v3.1 MYSQLI-EXTEND
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Класс для подключения к базе данных MySQL с SQL профилированием
 * 
 * Дата: 2025-11-01
 * Статус: PRODUCTION READY - MYSQLI EXTEND
 * 
 * ИЗМЕНЕНИЯ v3.1 MYSQLI-EXTEND:
 *   ✨ NEW: Класс DeepMaxMysqli extends mysqli - расширенный mysqli
 *   ✨ NEW: Переопределение query() с профилированием
 *   ✨ NEW: Переопределение prepare() с профилированием
 *   ✨ NEW: Переопределение real_query() с профилированием
 *   ✅ Интеграция с DeepMaxDebug::logSQL()
 *   ✅ Полная обратная совместимость (НЕ нужно менять код проекта)
 *   ✅ bind_param() работает (настоящий mysqli_stmt)
 *   ✅ Процедурные функции работают (mysqli_error и т.д.)
 *   ✅ Покрытие 100% SQL запросов
 * 
 * v2.0 FIXED:
 *   ✅ FIX: Заменён короткий тег <? на полный <?php
 *   ✅ Добавлена документация
 *   ✅ Добавлена обработка ошибок с логированием
 * 
 * КРИТИЧНО:
 *   - Использует константы из config.php:
 *     DB_HOST, DB_USER_NAME, DB_PASSWORD, DB_NAME
 *   - Объект $connection используется глобально через global $db
 *   - $connection теперь = DeepMaxMysqli (расширенный mysqli)
 * 
 * КАК РАБОТАЕТ:
 * 
 *   DeepMaxMysqli extends mysqli:
 *     - Наследует ВСЕ методы и свойства mysqli
 *     - Переопределяет query(), prepare(), real_query() для профилирования
 *     - Остаётся 100% совместимым mysqli объектом
 * 
 *   Весь существующий код работает БЕЗ ИЗМЕНЕНИЙ:
 *   
 *   $db->connection->query($sql)           → DeepMaxMysqli::query() → логирует
 *   $db->connection->prepare($sql)         → DeepMaxMysqli::prepare() → логирует
 *   $stmt->bind_param('s', $var)           → mysqli_stmt::bind_param() ✅
 *   mysqli_error($db->connection)          → работает ✅
 * 
 * ПРОФИЛИРОВАНИЕ:
 * 
 *   1. query($sql) - простые запросы:
 *      - Замер времени выполнения
 *      - parent::query() (настоящий mysqli)
 *      - DeepMaxDebug::logSQL($sql, [], $time)
 * 
 *   2. prepare($sql) - prepared statements:
 *      - Замер времени подготовки
 *      - parent::prepare() (настоящий mysqli)
 *      - DeepMaxDebug::logSQL($sql . ' [PREPARE]', [], $time)
 *      - Возвращает настоящий mysqli_stmt (НЕ обёртку)
 * 
 *   3. real_query($sql) - низкоуровневые запросы:
 *      - Замер времени выполнения
 *      - parent::real_query() (настоящий mysqli)
 *      - DeepMaxDebug::logSQL($sql, [], $time)
 * 
 * КОМПРОМИСС:
 * 
 *   - Prepared statements логируются при prepare(), не при execute()
 *   - Видим что запрос подготовлен, но не время выполнения execute()
 *   - ВСЕ 56 SQL запросов видны в Debug панели ✅
 * 
 * БЕЗОПАСНОСТЬ:
 * 
 *   - Проверка class_exists('DeepMaxDebug') перед вызовом
 *   - Проверка DeepMaxDebug::isEnabled()
 *   - Работает без DeepMaxDebug (graceful fallback)
 *   - Ошибки подключения логируются и останавливают выполнение
 * 
 * Использование (БЕЗ ИЗМЕНЕНИЙ В КОДЕ):
 *   require_once('DB/db.php');
 *   $db = new Database();
 *   
 *   // Автоматически профилируется:
 *   $result = $db->connection->query($sql);
 *   
 *   // Автоматически профилируется:
 *   $stmt = $db->connection->prepare($sql);
 *   $stmt->bind_param('s', $param);  // ← РАБОТАЕТ! (настоящий mysqli_stmt)
 *   $stmt->execute();
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */

/**
 * DeepMaxMysqli - расширенный mysqli с SQL профилированием
 * 
 * Наследует mysqli и переопределяет методы выполнения SQL
 * для автоматического логирования в DeepMaxDebug.
 * 
 * Остаётся полностью совместимым с mysqli:
 *   - Все методы mysqli работают
 *   - Все свойства mysqli доступны
 *   - Процедурные функции mysqli_* работают
 *   - Возвращает настоящий mysqli_stmt (не обёртку)
 */
class DeepMaxMysqli extends mysqli
{
    /**
     * Выполнение SQL запроса с профилированием
     * 
     * Переопределяет mysqli::query() для автоматического профилирования.
     * Замеряет время выполнения и логирует в DeepMaxDebug.
     * 
     * @param string $query - SQL запрос
     * @param int $resultmode - Режим результата (MYSQLI_STORE_RESULT или MYSQLI_USE_RESULT)
     * @return mysqli_result|bool - Результат запроса или false при ошибке
     */
    public function query($query, $resultmode = MYSQLI_STORE_RESULT)
    {
        // Начало замера времени
        $startTime = microtime(true);
        
        // Выполняем запрос через родительский mysqli::query()
        $result = parent::query($query, $resultmode);
        
        // Конец замера времени (в миллисекундах)
        $executionTime = (microtime(true) - $startTime) * 1000;
        
        // Логируем в DeepMaxDebug если класс существует и активен
        if (class_exists('DeepMaxDebug') && DeepMaxDebug::isEnabled()) {
            DeepMaxDebug::logSQL($query, [], $executionTime);
        }
        
        return $result;
    }
    
    /**
     * Подготовка SQL запроса с профилированием
     * 
     * Переопределяет mysqli::prepare() для автоматического профилирования.
     * Замеряет время подготовки и логирует в DeepMaxDebug.
     * 
     * ВАЖНО: Возвращает настоящий mysqli_stmt, НЕ обёртку.
     * Это позволяет bind_param() работать корректно (передача по ссылке).
     * 
     * КОМПРОМИСС: Логируется время prepare(), не execute().
     * Но все prepared statements видны в Debug панели.
     * 
     * @param string $query - SQL запрос с placeholders (?, :name)
     * @return mysqli_stmt|false - Prepared statement или false при ошибке
     */
    public function prepare($query)
    {
        // Начало замера времени
        $startTime = microtime(true);
        
        // Подготавливаем запрос через родительский mysqli::prepare()
        $stmt = parent::prepare($query);
        
        // Конец замера времени (в миллисекундах)
        $executionTime = (microtime(true) - $startTime) * 1000;
        
        // Логируем в DeepMaxDebug если класс существует и активен
        // Добавляем [PREPARE] чтобы отличать от query()
        if (class_exists('DeepMaxDebug') && DeepMaxDebug::isEnabled()) {
            DeepMaxDebug::logSQL($query . ' [PREPARE]', [], $executionTime);
        }
        
        // Возвращаем НАСТОЯЩИЙ mysqli_stmt (не обёртку)
        // Это критично для работы bind_param() (передача по ссылке)
        return $stmt;
    }
    
    /**
     * Низкоуровневое выполнение SQL запроса с профилированием
     * 
     * Переопределяет mysqli::real_query() для автоматического профилирования.
     * Используется редко, но на всякий случай тоже профилируем.
     * 
     * @param string $query - SQL запрос
     * @return bool - true при успехе, false при ошибке
     */
    public function real_query($query)
    {
        // Начало замера времени
        $startTime = microtime(true);
        
        // Выполняем запрос через родительский mysqli::real_query()
        $result = parent::real_query($query);
        
        // Конец замера времени (в миллисекундах)
        $executionTime = (microtime(true) - $startTime) * 1000;
        
        // Логируем в DeepMaxDebug если класс существует и активен
        if (class_exists('DeepMaxDebug') && DeepMaxDebug::isEnabled()) {
            DeepMaxDebug::logSQL($query . ' [REAL_QUERY]', [], $executionTime);
        }
        
        return $result;
    }
}

/**
 * Класс Database - подключение к MySQL
 * 
 * Использует DeepMaxMysqli вместо обычного mysqli
 * для автоматического профилирования SQL запросов.
 */
class Database
{
    /**
     * Объект подключения к MySQL
     * Теперь это DeepMaxMysqli (расширенный mysqli)
     * @var DeepMaxMysqli
     */
    public $connection;
    
    /**
     * Конструктор - создаёт подключение к БД
     * 
     * Использует DeepMaxMysqli вместо обычного mysqli
     * для автоматического SQL профилирования.
     * 
     * Использует константы из config.php:
     *   - DB_HOST
     *   - DB_USER_NAME
     *   - DB_PASSWORD
     *   - DB_NAME
     */
    public function __construct() {
        // Подключение к БД через DeepMaxMysqli (расширенный mysqli)
        $this->connection = new DeepMaxMysqli(DB_HOST, DB_USER_NAME, DB_PASSWORD, DB_NAME);
        
        // Проверка подключения
        // mysqli::connect_error доступен потому что DeepMaxMysqli extends mysqli
        if ($this->connection->connect_error) {
            // Логируем ошибку подключения
            error_log('[Database v3.1] CRITICAL: Не удалось подключиться к БД. Error: ' . $this->connection->connect_error);
            
            // Завершаем выполнение
            die('Database connection failed');
        }
        
        // Устанавливаем кодировку UTF-8
        // mysqli::set_charset() доступен потому что DeepMaxMysqli extends mysqli
        $this->connection->set_charset('utf8mb4');
    }
    
    /**
     * Деструктор - закрывает подключение при уничтожении объекта
     * 
     * mysqli::close() доступен потому что DeepMaxMysqli extends mysqli
     */
    public function __destruct() {
        if ($this->connection) {
            $this->connection->close();
        }
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КОНЕЦ DB/db.php v3.1 MYSQLI-EXTEND
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 3.1 MYSQLI-EXTEND
 * Дата: 2025-11-01
 * Статус: PRODUCTION READY ✅
 * 
 * НОВОЕ v3.1:
 *   ✅ DeepMaxMysqli extends mysqli - расширенный mysqli класс
 *   ✅ Переопределение query() для профилирования
 *   ✅ Переопределение prepare() для профилирования
 *   ✅ Переопределение real_query() для профилирования
 *   ✅ Интеграция с DeepMaxDebug::logSQL()
 *   ✅ Полная обратная совместимость
 *   ✅ bind_param() работает (настоящий mysqli_stmt)
 *   ✅ Процедурные функции работают
 *   ✅ Покрытие 100% SQL запросов
 * 
 * АРХИТЕКТУРА:
 * 
 *   DeepMaxMysqli класс:
 *     - extends mysqli (наследует ВСЕ методы и свойства)
 *     - query() - переопределён с профилированием
 *     - prepare() - переопределён с профилированием
 *     - real_query() - переопределён с профилированием
 *     - Все остальные методы - из родительского mysqli
 * 
 *   Database класс:
 *     - $connection = new DeepMaxMysqli() (расширенный mysqli)
 *     - __construct() - подключение через DeepMaxMysqli
 *     - __destruct() - закрытие соединения
 * 
 * ПРОФИЛИРОВАНИЕ:
 * 
 *   Покрывает 100% SQL запросов:
 *     - 16 × $db->connection->query()           → DeepMaxMysqli::query()
 *     - 40 × $db->connection->prepare()         → DeepMaxMysqli::prepare()
 *     - N × $db->connection->real_query()       → DeepMaxMysqli::real_query()
 *     = 56+ SQL операций профилируются
 * 
 *   Все логируются в DeepMaxDebug с временем выполнения в мс.
 * 
 * СОВМЕСТИМОСТЬ:
 * 
 *   ✅ Работает со всеми существующими файлами без изменений
 *   ✅ Обратная совместимость с v2.0
 *   ✅ Поддержка PHP 7.4, 8.0, 8.1, 8.2+
 *   ✅ Работает с DeepMaxDebug v4.4.0+
 *   ✅ Graceful fallback если DeepMaxDebug не установлен
 *   ✅ bind_param() передаёт параметры по ссылке корректно
 *   ✅ Процедурные функции mysqli_* работают
 * 
 * ПОЧЕМУ ЭТО РЕШЕНИЕ ПРАВИЛЬНОЕ:
 * 
 *   v3.0 PROXY PATTERN (НЕ РАБОТАЛО):
 *     ❌ $this->connection = $this (Database вместо mysqli)
 *     ❌ DeepMaxStatement обёртка ломала bind_param()
 *     ❌ Процедурные функции не работали
 * 
 *   v3.1 MYSQLI EXTEND (РАБОТАЕТ):
 *     ✅ DeepMaxMysqli extends mysqli (настоящий mysqli)
 *     ✅ Переопределение методов, а не обёртка
 *     ✅ Возвращает настоящий mysqli_stmt
 *     ✅ bind_param() работает (передача по ссылке)
 *     ✅ Процедурные функции работают
 * 
 * ИСПОЛЬЗОВАНИЕ:
 * 
 *   Никаких изменений в существующем коде не требуется!
 *   
 *   Просто замените старый db.php на этот файл:
 *     1. Сделайте бэкап: cp DB/db.php DB/db.php.backup
 *     2. Скопируйте этот файл в DB/db.php
 *     3. Перезапустите PHP-FPM: sudo systemctl restart php8.1-fpm
 *     4. Всё! SQL профилирование работает автоматически
 * 
 * ТЕСТИРОВАНИЕ:
 * 
 *   1. Проверить синтаксис:
 *      php -l DB/db.php
 *      Ожидается: No syntax errors detected
 * 
 *   2. Открыть любую страницу сайта
 *      Сайт должен загрузиться нормально
 * 
 *   3. В Debug панели должна появиться секция "SQL ЗАПРОСЫ"
 *      Счётчик должен показывать реальное количество (не 0!)
 * 
 *   4. При клике на секцию - список SQL с временем выполнения
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */