<?php
/**
 * Каталог акций Московской биржи
 * Версия: 2.3.0 FIRE
 * Дата: 2025-10-21
 * 
 * ИЗМЕНЕНИЯ v2.3.0 FIRE:
 * - NEW: Отдельный wrapper для легенды (.legend-wrapper)
 * - NEW: Легенда и примечание вынесены из catalog-wrapper
 * - NEW: Подготовка к mobile order: каталог → метрики → легенда
 * 
 * ИСПРАВЛЕНИЯ v2.2.1 FIX:
 * - FIX: Правильная структура wrapper'ов для flexbox order
 * - FIX: Метрики действительно перемещаются вниз на mobile
 * - FIX: Маленькие квадратики 3 в ряд на mobile
 * - FIX: Все div правильно закрыты
 * 
 * ИЗМЕНЕНИЯ v2.2.0 MOBILE-METRICS:
 * - NEW: Flexbox wrapper для изменения порядка блоков на mobile
 * - NEW: Метрики перемещены вниз на mobile (flexbox order)
 * - NEW: Маленькие квадратики метрик на mobile
 * - NEW: Обёртки .metrics-wrapper и .catalog-wrapper
 */

if (!function_exists('esc')) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА:</strong> Функция esc() не найдена.</div>');
}

global $db;
global $page_data;

if (!isset($db)) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА:</strong> База данных не подключена.</div>');
}

// Маппинг секторов к иконкам
if (!defined('SECTOR_ICONS')) {
    define('SECTOR_ICONS', [
        1  => '💻', 2  => '🌾', 3  => '🏦', 4  => '🏛️', 5  => '🏥',
        6  => '💳', 7  => '⚙️', 8  => '🔩', 9  => '🏗️', 10 => '🛢️',
        11 => '🏭', 12 => '📡', 13 => '🔬', 14 => '🛒', 15 => '🚚',
        16 => '💰', 18 => '🧪', 19 => '⚡'
    ]);
}

// Маппинг уровней листинга
$listingLevels = [
    1 => ['name' => 'Первый уровень листинга', 'icon' => '🏆'],
    2 => ['name' => 'Второй уровень листинга', 'icon' => '🥈'],
    3 => ['name' => 'Третий уровень листинга', 'icon' => '🥉']
];

// ============================================================================
// ПОЛУЧЕНИЕ ДАННЫХ ИЗ БД
// ============================================================================

$stmt = $db->connection->prepare("SELECT 
    i.Id,
    i.Instrument_TRADE_CODE,
    i.Instrument_ISIN,
    i.Instrument_TYPE,
    i.Instrument_LIST_SECTION_ID,
    i.Emitent_Id,
    i.Instrument_EMITENT_FULL_NAME,
    e.EMITENT_SHORT_NAME,
    e.EMITENT_FULL_NAME,
    e.INN,
    e.EMITENT_URL,
    e.ID_SECTOR_ECONOMIKI,
    e.SECTOR_ECONOMIKI,
    qa.SecCode as Akcii_SecCode
FROM Instrument i
LEFT JOIN Emitent e ON i.Emitent_Id = e.Id
LEFT JOIN QUIK_Akcii qa ON i.Instrument_ISIN = qa.ISIN
WHERE i.Instrument_SUPERTYPE_ID = ?
ORDER BY i.Instrument_LIST_SECTION_ID, e.ID_SECTOR_ECONOMIKI, e.EMITENT_SHORT_NAME, i.Instrument_TRADE_CODE");

$supertype_id = 1;
$stmt->bind_param("i", $supertype_id);
$stmt->execute();

if ($stmt->errno) {
    die('<div class="alert alert-danger" role="alert"><strong>ОШИБКА БД:</strong> ' . esc($stmt->error) . '</div>');
}

$result = $stmt->get_result();

// Группируем данные: Листинг -> Сектор -> Эмитент -> Акции
$data = [];
$totalStocks = 0;
$tradingStocks = 0;

while ($row = $result->fetch_assoc()) {
    $listingId = (int)$row['Instrument_LIST_SECTION_ID'];
    $sectorId = (int)$row['ID_SECTOR_ECONOMIKI'];
    $emitentId = (int)$row['Emitent_Id'];
    
    // Пропускаем записи без эмитента или сектора
    if (!$emitentId || !$sectorId || !$listingId) continue;
    
    // Инициализация структуры
    if (!isset($data[$listingId])) {
        $data[$listingId] = ['sectors' => []];
    }
    if (!isset($data[$listingId]['sectors'][$sectorId])) {
        $data[$listingId]['sectors'][$sectorId] = [
            'name' => $row['SECTOR_ECONOMIKI'] ?: 'Без сектора',
            'emitents' => []
        ];
    }
    if (!isset($data[$listingId]['sectors'][$sectorId]['emitents'][$emitentId])) {
        $data[$listingId]['sectors'][$sectorId]['emitents'][$emitentId] = [
            'name' => $row['EMITENT_SHORT_NAME'] ?: $row['EMITENT_FULL_NAME'],
            'inn' => $row['INN'] ?: '',
            'url' => $row['EMITENT_URL'] ?: '',
            'stocks' => []
        ];
    }
    
    // Определяем статус акции
    $status = !empty($row['Akcii_SecCode']) ? 'торгуется' : 'нет данных';
    if ($status === 'торгуется') $tradingStocks++;
    
    // Добавляем акцию
    $data[$listingId]['sectors'][$sectorId]['emitents'][$emitentId]['stocks'][] = [
        'code' => $row['Instrument_TRADE_CODE'],
        'name' => $row['Instrument_EMITENT_FULL_NAME'] ?: $row['EMITENT_SHORT_NAME'],
        'type' => $row['Instrument_TYPE'] ?: 'Акция обыкновенная',
        'status' => $status
    ];
    
    $totalStocks++;
}

// Закрываем statement
$stmt->close();

// Подсчёт статистики
$totalEmitents = 0;

// Подсчёт общего количества секторов из константы
$totalSectors = count(SECTOR_ICONS);

// Подсчёт эмитентов
foreach ($data as $listing) {
    foreach ($listing['sectors'] as $sector) {
        $totalEmitents += count($sector['emitents']);
    }
}

// Подсчёт акций по уровням листинга
$listingStocksCount = [];
foreach ($data as $listingId => $listing) {
    $listingStocksCount[$listingId] = 0;
    foreach ($listing['sectors'] as $sector) {
        foreach ($sector['emitents'] as $emitent) {
            $listingStocksCount[$listingId] += count($emitent['stocks']);
        }
    }
}

?>

<div class="container">

<!-- HERO СЕКЦИЯ -->
<section class="hero">
    <div class="hero-header">
        <span class="sector-icon-large">📈</span>
        <div class="hero-content-sector">
            <h1><?php echo esc($page_data[1]->MENU_ancor); ?></h1>
            <p class="hero-description">
                <?php echo esc($page_data[1]->MENU_description); ?>
            </p>
        </div>
    </div>
</section>

<!-- FLEXBOX WRAPPER ДЛЯ ИЗМЕНЕНИЯ ПОРЯДКА НА MOBILE -->
<div class="page-content-wrapper">

    <!-- МЕТРИКИ WRAPPER -->
    <div class="metrics-wrapper">
        <!-- КОМПАКТНАЯ СТАТИСТИКА -->
        <div class="metrics-grid metrics-grid-compact">
            
            <!-- УРОВНИ ЛИСТИНГА -->
            <?php foreach ($listingLevels as $listingId => $listingInfo): ?>
            <?php if (isset($listingStocksCount[$listingId])): ?>
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-listing-<?php echo esc($listingId); ?>-desc">
                <div class="metric-icon metric-icon-compact"><?php echo $listingInfo['icon']; ?></div>
                <div class="metric-label"><?php echo esc($listingInfo['name']); ?></div>
                <div class="metric-value"><?php echo esc($listingStocksCount[$listingId]); ?></div>
                <span id="metric-listing-<?php echo esc($listingId); ?>-desc" class="sr-only">
                    <?php echo esc($listingInfo['name']); ?>: <?php echo esc($listingStocksCount[$listingId]); ?> акций
                </span>
            </div>
            <?php endif; ?>
            <?php endforeach; ?>
            
            <!-- СЕКТОРОВ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-sectors-desc">
                <div class="metric-icon metric-icon-compact">🗂️</div>
                <div class="metric-label">Секторов</div>
                <div class="metric-value"><?php echo esc($totalSectors); ?></div>
                <span id="metric-sectors-desc" class="sr-only">
                    Количество секторов экономики: <?php echo esc($totalSectors); ?>
                </span>
            </div>
            
            <!-- ЭМИТЕНТОВ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-emitents-desc">
                <div class="metric-icon metric-icon-compact">🏢</div>
                <div class="metric-label">Эмитентов</div>
                <div class="metric-value"><?php echo esc($totalEmitents); ?></div>
                <span id="metric-emitents-desc" class="sr-only">
                    Количество эмитентов: <?php echo esc($totalEmitents); ?>
                </span>
            </div>
            
            <!-- АКЦИЙ ВСЕГО -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-stocks-desc">
                <div class="metric-icon metric-icon-compact">📊</div>
                <div class="metric-label">Акций всего</div>
                <div class="metric-value"><?php echo esc($totalStocks); ?></div>
                <span id="metric-stocks-desc" class="sr-only">
                    Общее количество акций: <?php echo esc($totalStocks); ?>
                </span>
            </div>
            
            <!-- ТОРГУЕТСЯ -->
            <div class="metric-card metric-card-compact" role="article" aria-describedby="metric-trading-desc">
                <div class="metric-icon metric-icon-compact">🟢</div>
                <div class="metric-label">Торгуется</div>
                <div class="metric-value"><?php echo esc($tradingStocks); ?></div>
                <span id="metric-trading-desc" class="sr-only">
                    Количество активно торгуемых акций: <?php echo esc($tradingStocks); ?>
                </span>
            </div>
        </div>
    </div>
    <!-- КОНЕЦ МЕТРИКИ WRAPPER -->

    <!-- КАТАЛОГ WRAPPER -->
    <div class="catalog-wrapper">

        <?php if (empty($data)): ?>
        <div class="alert alert-info" role="alert">
            <strong>Нет данных</strong><br>
            В базе данных не найдено акций для отображения.
        </div>
        <?php else: ?>

        <!-- 3 РАВНЫЕ КОЛОНКИ ДЛЯ УРОВНЕЙ ЛИСТИНГА -->
        <div class="sectors-grid">
            <?php foreach ($data as $listingId => $listing): ?>
            <?php if (!isset($listingLevels[$listingId])) continue; ?>
            
            <div class="listing-column">
                <h2 class="listing-column-title" id="listing-<?php echo esc($listingId); ?>">
                    <span><?php echo $listingLevels[$listingId]['icon']; ?></span>
                    <?php echo esc($listingLevels[$listingId]['name']); ?>
                </h2>
                
                <!-- ГРУППИРОВКА ПО СЕКТОРАМ -->
                <?php foreach ($listing['sectors'] as $sectorId => $sector): ?>
                
                <details class="v6-sector" open>
                    <summary>
                        <h3>
                            <span class="v6-icon"><?php echo SECTOR_ICONS[$sectorId] ?? '📁'; ?></span>
                            <span><?php echo esc($sector['name']); ?></span>
                            <span class="v1-count">(<?php echo esc(count($sector['emitents'])); ?>)</span>
                        </h3>
                    </summary>
                    <div class="v6-content">
                        
                        <!-- СПИСОК ЭМИТЕНТОВ С АКЦИЯМИ -->
                        <?php foreach ($sector['emitents'] as $emitentId => $emitent): ?>
                        
                        <div class="v6-emitent">
                            <?php if (!empty($emitent['url'])): ?>
                            <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/') . esc($emitent['url']); ?>/" 
                               class="v6-emitent-name"
                               title="<?php echo esc($emitent['name']); ?> — полная информация об эмитенте">
                                <?php echo esc($emitent['name']); ?>
                            </a>
                            <?php else: ?>
                            <span class="v6-emitent-name"><?php echo esc($emitent['name']); ?></span>
                            <?php endif; ?>
                            
                            <div class="v6-stocks">
                                <?php foreach ($emitent['stocks'] as $stock): ?>
                                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>stocks/<?php echo esc($stock['code']); ?>/" 
                                   class="v6-stock<?php echo $stock['status'] !== 'торгуется' ? ' paused' : ''; ?>"
                                   title="<?php echo esc($stock['type']); ?> — <?php echo esc($stock['code']); ?>">
                                    <?php echo esc($stock['code']); ?>
                                </a>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <?php endforeach; ?>
                        
                    </div>
                </details>
                
                <?php endforeach; ?>
                
            </div>
            
            <?php endforeach; ?>
        </div>

        <?php endif; ?>

    </div>
    <!-- КОНЕЦ КАТАЛОГ WRAPPER -->

    <!-- ЛЕГЕНДА WRAPPER -->
    <div class="legend-wrapper">

        <!-- ОБОЗНАЧЕНИЯ -->
        <div class="legend-block">
            <h3 class="legend-title">
                <span>🔍</span>
                Обозначения
            </h3>
            <div class="legend-content">
                <div class="legend-item">
                    <span class="legend-icon">🟢</span>
                    <strong>Торгуется</strong> — активная торговля на бирже
                </div>
                <div class="legend-item">
                    <span class="legend-icon">🟠</span>
                    <strong>Нет данных</strong> — торговая информация отсутствует
                </div>
                <div class="legend-item">
                    <span class="legend-icon">🏆</span>
                    <strong>Первый уровень</strong> — наиболее ликвидные акции
                </div>
                <div class="legend-item">
                    <span class="legend-icon">🥈</span>
                    <strong>Второй уровень</strong> — средняя ликвидность
                </div>
                <div class="legend-item">
                    <span class="legend-icon">🥉</span>
                    <strong>Третий уровень</strong> — низкая ликвидность
                </div>
            </div>
        </div>

        <div class="info-notice">
            <strong>Источник данных:</strong> Справочная информация из базы данных DeepMax.ru.<br>
            Данные обновляются периодически и могут отличаться от текущих биржевых.<br><br>
            <strong>Структура каталога:</strong> Акции сгруппированы по уровням листинга Московской биржи, 
            затем по секторам экономики и эмитентам.
        </div>

    </div>
    <!-- КОНЕЦ ЛЕГЕНДА WRAPPER -->

</div>
<!-- КОНЕЦ PAGE-CONTENT-WRAPPER -->

</div>