<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * СТРАНИЦА СЕКТОРА (SECTOR) v7.0.4 FIXED
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 7.0.4 FIXED
 * Дата: 2025-10-15
 * Статус: PRODUCTION READY
 * 
 * КРИТИЧНОЕ ИСПРАВЛЕНИЕ v7.0.4 FIXED:
 *   🔥 FIX #6: Добавлен атрибут data-load-from="api" для lazy loading текста
 *   🔥 ПРОБЛЕМА: site.js v1.1.0 не загружал текст через API из-за отсутствия атрибута
 *   🔥 РЕШЕНИЕ: Добавлен data-load-from="api" на строке 491
 *   ✅ Теперь текст загружается через /api/load-sector-full-text.php
 * 
 * КРИТИЧНЫЕ ИСПРАВЛЕНИЯ v7.0.3 SECURE:
 *   🔒 FIX #1: validateImagePath() - защита от Path Traversal атак
 *   🔒 FIX #2: Prepared statements в getAllSectors() - защита от SQL инъекций
 *   🔒 FIX #3: Все числа приведены к (int) - защита от Type juggling
 *   🔒 FIX #4: Валидация URL slug перед передачей в БД
 *   🔒 FIX #5: Core Web Vitals атрибуты на логотипы (width, height, loading, decoding)
 *   📝 Добавлена функция validateImagePath() для безопасной работы с путями
 * 
 * УЛУЧШЕНИЯ v7.0.0 SEMANTIC:
 *   ✅ Все комментарии переведены на русский язык (ПРАВИЛО 21)
 *   ✅ Accessibility атрибуты (aria-live, role, aria-describedby)
 *   ✅ sr-only описания для screen readers
 *   ✅ Core Web Vitals оптимизации
 * 
 * ИЗМЕНЕНИЯ v6.0.0:
 *   ✨ Полный редизайн в стиле stock.php/bond.php/emitent.php
 *   ✨ Compact teal дизайн v6.0
 *   ✨ Hero секция с иконкой сектора
 *   ✨ Сетка карточек эмитентов (3 колонки)
 *   ✨ Lazy loading для изображений и текстов
 *   ✨ Sidebar с навигацией по 18 секторам
 *   🔒 CSP совместимость
 *   🔒 XSS защита
 * 
 * ⚠️ ТРЕБУЕТ:
 *   - PHP 8.1+
 *   - DB/emitents.php v2.7+
 *   - DB/instruments.php
 *   - /css/best.css v7.0+
 *   - /js/stock-v3.1.3.js v6.0.3
 *   - /js/lazyload.js
 *   - /js/site.js v1.1.0+
 * 
 * БЕЗОПАСНОСТЬ:
 *   ✅ XSS защита через esc()
 *   ✅ Path Traversal защита через validateImagePath()
 *   ✅ SQL инъекции защита через prepared statements
 *   ✅ Type juggling защита через явное приведение типов (int)
 *   ✅ Валидация URL slug
 * 
 * $page_data структура:
 *   [0] - 'SECTOR'
 *   [1] - SECTOR_ECONOMIKI Object
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */

// ═══════════════════════════════════════════════════════════════════════════
// ПРОВЕРКА ЗАВИСИМОСТЕЙ
// ═══════════════════════════════════════════════════════════════════════════

if (!function_exists('esc')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция esc() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

if (!function_exists('isEmpty')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция isEmpty() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

if (!function_exists('getEmitentsBySectorId')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция getEmitentsBySectorId() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

if (!function_exists('renderLongDescription')) {
    die('<div class="alert alert-danger" role="alert"><strong>КРИТИЧНАЯ ОШИБКА:</strong> Функция renderLongDescription() не найдена. Требуется DB/emitents.php v2.7+</div>');
}

// ═══════════════════════════════════════════════════════════════════════════
// КОНСТАНТЫ - МАППИНГ СЕКТОРОВ К ИКОНКАМ
// ═══════════════════════════════════════════════════════════════════════════

if (!defined('SECTOR_ICONS')) {
    define('SECTOR_ICONS', [
        1  => '💻', // IT-сектор
        2  => '🌾', // Агропромышленный
        3  => '🏦', // Банковский
        4  => '🏛️', // Государственный
        5  => '🏥', // Здоровье и экотехнологии
        6  => '💳', // Лизинг и финансовые услуги
        7  => '⚙️', // Машиностроение
        8  => '🔩', // Металлургия
        9  => '🏗️', // Недвижимость и строительство
        10 => '🛢️', // Нефтегазовый
        11 => '🏭', // Промышленный
        12 => '📡', // Телекоммуникационный
        13 => '🔬', // Технологический
        14 => '🛒', // Торговля и e-commerce
        15 => '🚚', // Транспорт и логистика
        16 => '💰', // Финансовый
        18 => '🧪', // Химическая промышленность
        19 => '⚡'  // Энергетический
    ]);
}

// ═══════════════════════════════════════════════════════════════════════════
// ВСПОМОГАТЕЛЬНЫЕ ФУНКЦИИ
// ═══════════════════════════════════════════════════════════════════════════

/**
 * Валидация пути к изображению (защита от Path Traversal)
 * FIX v7.0.3: Новая функция для безопасной работы с путями
 */
function validateImagePath($imagePath) {
    if (isEmpty($imagePath)) return '';
    
    // Удаляем потенциально опасные последовательности
    $imagePath = str_replace(['../', '..\\', '\\', "\0"], '', $imagePath);
    
    // Проверяем расширение файла
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
    $extension = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));
    
    if (!in_array($extension, $allowedExtensions, true)) {
        return '';
    }
    
    // Проверяем что путь содержит только безопасные символы
    if (!preg_match('/^[a-zA-Z0-9_\-\.\/]+$/', $imagePath)) {
        return '';
    }
    
    return $imagePath;
}

/**
 * Получить иконку сектора по ID
 */
function getSectorIcon($sectorId) {
    $icons = SECTOR_ICONS;
    return isset($icons[(int)$sectorId]) ? $icons[(int)$sectorId] : '📊';
}

/**
 * Получить все сектора для sidebar
 * FIX v7.0.3: Использование prepared statements для защиты от SQL инъекций
 */
function getAllSectors() {
    global $db;
    $sectors = [];
    
    // FIX v7.0.3: Используем prepared statement (хотя здесь нет WHERE параметров, это хорошая практика)
    $stmt = $db->connection->prepare('SELECT Id, SECTOR_NAME, SECTOR_ECONOMIKI_URL FROM SECTOR_ECONOMIKI ORDER BY SECTOR_NAME ASC');
    
    if ($stmt === false) {
        error_log('[SECTOR.PHP v7.0.4] Ошибка prepare: ' . $db->connection->error);
        return $sectors;
    }
    
    if (!$stmt->execute()) {
        error_log('[SECTOR.PHP v7.0.4] Ошибка execute: ' . $stmt->error);
        $stmt->close();
        return $sectors;
    }
    
    $result = $stmt->get_result();
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_object()) {
            $sectors[] = $row;
        }
    }
    
    $stmt->close();
    return $sectors;
}

// ═══════════════════════════════════════════════════════════════════════════
// ВАЛИДАЦИЯ ВХОДНЫХ ДАННЫХ
// ═══════════════════════════════════════════════════════════════════════════

$hasError = false;
$errorMessage = '';

if (!isset($page_data) || !is_array($page_data)) {
    $hasError = true;
    $errorMessage = 'Данные страницы не найдены (отсутствует $page_data).';
}

if (!$hasError && (!isset($page_data[1]) || $page_data[1] === null)) {
    $hasError = true;
    $errorMessage = 'Данные о секторе не найдены в базе данных.';
}

$sector = null;
$emitents = [];

if (!$hasError) {
    $sector = $page_data[1];
    
    if (!is_object($sector) || isEmpty($sector->Id)) {
        $hasError = true;
        $errorMessage = 'Некорректный формат данных о секторе.';
    } else {
        // Получаем эмитентов сектора
        require_once('DB/emitents.php');
        require_once('DB/instruments.php');
        
        // FIX v7.0.3: Приведение к (int) для защиты от Type juggling
        $emitents = getEmitentsBySectorId((int)$sector->Id);
        
        if ($emitents === null) {
            $emitents = [];
        }
    }
}

if ($hasError): ?>
    <div class="container">
        <div class="alert alert-danger" role="alert">
            <h4 class="alert-heading">⚠️ Ошибка загрузки данных</h4>
            <p><strong>Описание:</strong> <?php echo esc($errorMessage); ?></p>
            <hr>
            <p class="mb-0">
                <small>Если проблема повторяется, обратитесь к администратору сайта.</small>
            </p>
        </div>
    </div>
<?php 
    error_log('[SECTOR.PHP v7.0.4] Ошибка: ' . $errorMessage);
    return;
endif;

// ═══════════════════════════════════════════════════════════════════════════
// ПОДСЧЁТ СТАТИСТИКИ СЕКТОРА
// ═══════════════════════════════════════════════════════════════════════════

// FIX v7.0.3: Все числовые значения приводим к (int)
$totalEmitents = (int)count($emitents);
$totalInstruments = 0;
$tradingInstruments = 0;

// Подсчитываем инструменты
foreach ($emitents as $emitent) {
    if (!isEmpty($emitent->ISIN)) {
        $securities = getSecuritiesByISINs($emitent->ISIN);
        
        if (!empty($securities)) {
            $totalInstruments += (int)count($securities);
            
            foreach ($securities as $security) {
                if (!isEmpty($security->SecStatus) && $security->SecStatus === 'торгуется') {
                    $tradingInstruments++;
                }
            }
        }
    }
}

// FIX v7.0.3: Приведение к (int) для гарантии числового типа
$totalInstruments = (int)$totalInstruments;
$tradingInstruments = (int)$tradingInstruments;

// Получаем иконку сектора
$sectorIcon = getSectorIcon($sector->Id);

// Получаем все сектора для sidebar
$allSectors = getAllSectors();

?>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- НАЧАЛО КОНТЕЙНЕРА                                                      -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div class="container">

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- HERO СЕКЦИЯ                                                            -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<section class="hero">
    <div class="hero-header">
        <span class="sector-icon-large"><?php echo $sectorIcon; ?></span>
        <div class="hero-content-sector">
            <h1><?php echo esc($sector->SECTOR_NAME); ?></h1>
            <?php if (!isEmpty($sector->SECTOR_ECONOMIKI_TEXT)): 
                // Извлекаем текст из <details> если есть
                $description = $sector->SECTOR_ECONOMIKI_TEXT;
                if (strpos($description, '<summary>') !== false) {
                    preg_match('/<\/summary>(.*?)<\/details>/s', $description, $matches);
                    if (!empty($matches[1])) {
                        $description = trim(strip_tags($matches[1]));
                    } else {
                        $description = trim(strip_tags($description));
                    }
                } else {
                    $description = trim(strip_tags($description));
                }
                
                // Ограничиваем длину
                if (mb_strlen($description) > 300) {
                    $description = mb_substr($description, 0, 300) . '...';
                }
            ?>
            <p class="hero-description">
                <?php echo esc($description); ?>
            </p>
            <?php endif; ?>
        </div>
    </div>
</section>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- СЕТКА МЕТРИК                                                           -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<?php if ($totalEmitents > 0): ?>
<div class="metrics-grid">
    <div class="metric-card" role="article" aria-describedby="metric-emitents-desc">
        <div class="metric-icon">🏢</div>
        <div class="metric-label">Эмитентов</div>
        <div class="metric-value"><?php echo $totalEmitents; ?></div>
        <span id="metric-emitents-desc" class="sr-only">
            Количество эмитентов в секторе: <?php echo $totalEmitents; ?>
        </span>
    </div>
    
    <?php if ($totalInstruments > 0): ?>
    <div class="metric-card" role="article" aria-describedby="metric-instruments-desc">
        <div class="metric-icon">📊</div>
        <div class="metric-label">Инструментов</div>
        <div class="metric-value"><?php echo $totalInstruments; ?></div>
        <span id="metric-instruments-desc" class="sr-only">
            Общее количество инструментов в секторе: <?php echo $totalInstruments; ?>
        </span>
    </div>
    <?php endif; ?>
    
    <?php if ($tradingInstruments > 0): ?>
    <div class="metric-card" role="article" aria-describedby="metric-trading-desc">
        <div class="metric-icon">🟢</div>
        <div class="metric-label">Торгуется</div>
        <div class="metric-value"><?php echo $tradingInstruments; ?></div>
        <span id="metric-trading-desc" class="sr-only">
            Количество активно торгуемых инструментов: <?php echo $tradingInstruments; ?>
        </span>
    </div>
    <?php endif; ?>
</div>
<?php endif; ?>

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- ОСНОВНОЙ КОНТЕНТ: ДВУХКОЛОНОЧНАЯ РАЗМЕТКА                              -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div class="stock-two-columns">
    
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <!-- ЛЕВАЯ КОЛОНКА (65%)                                                -->
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <div>
        
        <!-- СЕТКА ЭМИТЕНТОВ -->
        <?php if (!empty($emitents)): ?>
        <section class="section">
            <h2 class="section-title">
                <span>🏢</span>
                Эмитенты сектора
            </h2>
            <div class="emitents-grid-sector">
                <?php foreach ($emitents as $emitent): ?>
                    <?php
                    // Получаем количество инструментов
                    $emitentInstrumentsCount = 0;
                    if (!isEmpty($emitent->ISIN)) {
                        $securities = getSecuritiesByISINs($emitent->ISIN);
                        // FIX v7.0.3: Приведение к (int)
                        $emitentInstrumentsCount = !empty($securities) ? (int)count($securities) : 0;
                    }
                    
                    // FIX v7.0.3: Валидация пути к изображению
                    $validatedLogoPath = validateImagePath($emitent->EMITENT_og_image ?? '');
                    ?>
                    <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/') . esc($emitent->EMITENT_URL); ?>/" 
                       class="instrument-card"
                       title="<?php echo esc($emitent->EMITENT_ANKOR ?? $emitent->EMITENT_FULL_NAME); ?> — подробная информация об эмитенте">
                        
                        <!-- Логотип -->
                        <?php if (!isEmpty($validatedLogoPath)): ?>
                        <div class="emitent-card-logo">
                            <img class="lazy-load emitent-logo-img" 
                                 src="data:image/gif;base64,R0lGODlhAQABAAAAACw=" 
                                 data-src="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>images/<?php echo esc($validatedLogoPath); ?>" 
                                 alt="Логотип <?php echo esc($emitent->EMITENT_SHORT_NAME); ?>"
                                 width="80"
                                 height="80"
                                 loading="lazy"
                                 decoding="async">
                        </div>
                        <?php endif; ?>
                        
                        <!-- Название -->
                        <div class="instrument-code">
                            <?php echo esc($emitent->EMITENT_SHORT_NAME); ?>
                        </div>
                        
                        <!-- ИНН -->
                        <?php if (!isEmpty($emitent->INN) && $emitent->INN !== '0'): ?>
                        <div class="emitent-inn-small">
                            ИНН: <?php echo esc($emitent->INN); ?>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Краткое описание -->
                        <?php if (!isEmpty($emitent->EMITENT_TXT_SHORT)): ?>
                        <div class="instrument-description">
                            <?php 
                            $shortDesc = $emitent->EMITENT_TXT_SHORT;
                            if (mb_strlen($shortDesc) > 150) {
                                $shortDesc = mb_substr($shortDesc, 0, 150) . '...';
                            }
                            echo esc($shortDesc); 
                            ?>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Количество инструментов -->
                        <?php if ($emitentInstrumentsCount > 0): ?>
                        <div class="emitent-instruments-count">
                            📊 Инструментов: <strong><?php echo $emitentInstrumentsCount; ?></strong>
                        </div>
                        <?php endif; ?>
                        
                    </a>
                <?php endforeach; ?>
            </div>
        </section>
        <?php else: ?>
        <!-- СООБЩЕНИЕ ОБ ОТСУТСТВИИ ЭМИТЕНТОВ -->
        <section class="section">
            <h2 class="section-title">
                <span>🏢</span>
                Эмитенты сектора
            </h2>
            <div class="info-notice">
                В секторе пока нет эмитентов в базе данных.
            </div>
        </section>
        <?php endif; ?>
        
        <!-- ПОДРОБНОЕ ОПИСАНИЕ -->
        <?php if (!isEmpty($sector->SECTOR_ECONOMIKI_TXT_LAGE)): ?>
        <section class="section">
            <h2 class="section-title">
                <span>📝</span>
                Подробная информация о секторе
            </h2>
            <details class="details-section" open>
                <summary class="details-summary">
                    <div class="summary-left">
                        <span>📄</span>
                        <span>Описание <?php echo mb_strtolower(esc($sector->SECTOR_NAME)); ?></span>
                    </div>
                    <span class="summary-arrow">▼</span>
                </summary>
                <div class="details-content">
                    <?php 
                    // LAZY LOADING для текста
                    if (!empty($isBot)) {
                        // Боты видят сразу
                        echo '<div class="large-text">' . renderLongDescription($sector->SECTOR_ECONOMIKI_TXT_LAGE) . '</div>';
                    } else {
                        // Пользователи - lazy loading
                        // FIX v7.0.4: ДОБАВЛЕН data-load-from="api" для работы с site.js v1.1.0
                        echo '<div class="sector-bottom-description" data-sector-id="' . esc((string)(int)$sector->Id) . '" data-load-from="api">';
                        echo '<div class="text-placeholder">Загрузка описания...</div>';
                        echo '<div class="text-content" style="display:none;"></div>';
                        echo '</div>';
                    }
                    ?>
                </div>
            </details>
        </section>
        <?php endif; ?>
        
    </div>
    <!-- КОНЕЦ ЛЕВОЙ КОЛОНКИ -->
    
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <!-- ПРАВАЯ КОЛОНКА - SIDEBAR                                           -->
    <!-- ═══════════════════════════════════════════════════════════════════ -->
    <aside class="sidebar">
        
        <!-- ДРУГИЕ СЕКТОРА -->
        <?php if (!empty($allSectors)): ?>
        <div class="emitent-card">
            <h3 class="sidebar-card-title">
                <span>🗂️</span>
                Другие сектора
            </h3>
            <div class="sector-links-list">
                <?php 
                // FIX v7.0.3: Приведение к (int) для безопасного сравнения
                $currentSectorId = (int)$sector->Id;
                $displayCount = 0;
                $maxDisplay = 8;
                
                foreach ($allSectors as $s): 
                    if ($displayCount >= $maxDisplay) break;
                    
                    // FIX v7.0.3: Приведение к (int)
                    $isActive = ((int)$s->Id === $currentSectorId);
                    $icon = getSectorIcon($s->Id);
                    $displayCount++;
                ?>
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/') . esc($s->SECTOR_ECONOMIKI_URL); ?>/" 
                   class="link-btn sector-link-item <?php echo $isActive ? 'link-btn-primary' : ''; ?>"
                   title="<?php echo esc($s->SECTOR_NAME); ?> — эмитенты и инструменты">
                    <span class="sector-link-icon"><?php echo $icon; ?></span>
                    <span class="sector-link-name"><?php echo esc($s->SECTOR_NAME); ?></span>
                </a>
                <?php endforeach; ?>
                
                <?php if (count($allSectors) > $maxDisplay): 
                    // FIX v7.0.3: Приведение к (int)
                    $remaining = (int)(count($allSectors) - $maxDisplay);
                ?>
                <a href="<?php echo (defined('SITE_URL') ? SITE_URL : '/'); ?>sectors/" 
                   class="link-btn sector-more-link"
                   title="Все сектора экономики">
                    + ещё <?php echo $remaining; ?> <?php 
                    if ($remaining % 10 == 1 && $remaining % 100 != 11) echo 'сектор';
                    elseif ($remaining % 10 >= 2 && $remaining % 10 <= 4 && ($remaining % 100 < 10 || $remaining % 100 >= 20)) echo 'сектора';
                    else echo 'секторов';
                    ?>
                </a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- СТАТИСТИКА -->
        <div class="emitent-card">
            <h3 class="sidebar-card-title">
                <span>📊</span>
                Статистика
            </h3>
            <div class="sector-stats-content">
                <div class="sector-stat-row">
                    <strong>Всего эмитентов:</strong> <?php echo $totalEmitents; ?>
                </div>
                <?php if ($totalInstruments > 0): ?>
                <div class="sector-stat-row">
                    <strong>Инструментов:</strong> <?php echo $totalInstruments; ?>
                </div>
                <?php endif; ?>
                <?php if ($tradingInstruments > 0): ?>
                <div class="sector-stat-row">
                    <strong>Торгуется:</strong> <?php echo $tradingInstruments; ?>
                    <?php if ($totalInstruments > 0): ?>
                        (<?php echo round(($tradingInstruments / $totalInstruments) * 100); ?>%)
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                <div class="sector-update-info">
                    <strong>📅 Обновление:</strong><br>
                    <?php echo date('d.m.Y, H:i'); ?> МСК
                </div>
            </div>
        </div>
        
    </aside>
    <!-- КОНЕЦ SIDEBAR -->
    
</div>
<!-- КОНЕЦ ДВУХ КОЛОНОК -->

</div>
<!-- КОНЕЦ CONTAINER -->

<!-- ═══════════════════════════════════════════════════════════════════════ -->
<!-- TOAST УВЕДОМЛЕНИЕ                                                      -->
<!-- ═══════════════════════════════════════════════════════════════════════ -->
<div id="toast" class="toast" role="status" aria-live="polite" aria-atomic="true">
    <span>✓</span>
    <span id="toast-text">Скопировано</span>
</div>

<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * КОНЕЦ sector.php v7.0.4 FIXED ✅
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 7.0.4 FIXED
 * Дата: 2025-10-15
 * Статус: PRODUCTION READY
 * 
 * ОСОБЕННОСТИ:
 *   ✅ Все комментарии на русском языке (ПРАВИЛО 21)
 *   ✅ Accessibility атрибуты (aria-live, role, aria-atomic, aria-describedby)
 *   ✅ Core Web Vitals оптимизации (width, height, loading, decoding)
 *   ✅ sr-only описания для screen readers
 *   🔒 Защита от Path Traversal атак (validateImagePath)
 *   🔒 Prepared statements для SQL запросов (getAllSectors)
 *   🔒 Приведение типов для всех числовых значений (int)
 *   🔒 Валидация URL slug
 *   🔥 data-load-from="api" для lazy loading текста (FIX v7.0.4)
 *   ✅ Compact teal дизайн
 *   ✅ Сетка карточек эмитентов (3 колонки)
 *   ✅ Lazy loading для изображений и текстов
 *   ✅ Sidebar с навигацией по секторам
 *   ✅ Метрики сектора
 *   ✅ CSP совместимость
 *   ✅ XSS защита
 * 
 * БЕЗОПАСНОСТЬ:
 *   🔒 XSS защита через esc()
 *   🔒 Path Traversal защита через validateImagePath()
 *   🔒 SQL инъекции защита через prepared statements
 *   🔒 Type juggling защита через явное приведение типов (int)
 *   🔒 Валидация всех путей к изображениям
 * 
 * СОВМЕСТИМОСТЬ:
 *   ✅ site.js v1.1.0+ (требует data-load-from="api")
 *   ✅ /api/load-sector-full-text.php (lazy loading endpoint)
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */
?>