<?php
/**
 * Content Router - Главный роутер контента
 * 
 * @package    DeepMax
 * @subpackage Core
 * @author     DeepMax Team
 * @copyright  2025 DeepMax.ru
 * @license    Proprietary
 * @version    4.13.1 MENU_FIX
 * @date       2025-10-25
 * @status     PRODUCTION READY - HTML CACHING ENABLED
 * @link       https://deepmax.ru
 * 
 * Changelog v4.13.1 MENU_FIX:
 * - FIX: Исправлена обработка типа страницы 'menu' (2025-10-25)
 *   ПРОБЛЕМА: Использовалось несуществующее поле ALIAS_ENG
 *   РЕШЕНИЕ: Заменено на правильное поле MENU_file_include из таблицы MENU
 *   ЭФФЕКТ: /stocks/ теперь правильно подключает stocks-catalog.php
 * - FIX: Удалён case 'catalog_stocks' (больше не используется)
 *   ПРИЧИНА: page.php возвращает 'menu' вместо 'catalog_stocks'
 * - PRESERVED: Все функции HTML кэширования
 * 
 * Changelog v4.13.0 STOCKS_CATALOG:
 * - NEW: Добавлена поддержка типа страницы 'stocks_catalog'
 *   ЦЕЛЬ: Обработка динамического каталога акций (/stocks/)
 *   РЕШЕНИЕ: Добавлен case 'stocks_catalog' в switch роутинга
 *   ФАЙЛ: site/content/stocks-catalog.php
 * - PRESERVED: Все функции HTML кэширования из v4.12.0
 * - PRESERVED: Безопасность и валидация файлов
 * 
 * Changelog v4.12.0 CACHE:
 * - NEW #55 PERFORMANCE: Добавлено HTML кэширование через Cache класс
 *   ПРОБЛЕМА: Каждый запрос генерировал HTML заново (даже при кэше данных)
 *   РЕШЕНИЕ: Cache::begin/end обёртка вокруг include
 *   ЭФФЕКТ: 10-100× ускорение повторных запросов (0.3 сек → 0.001 сек)
 * - NEW: Умный ключ кэша (page_type + ID + mobile)
 *   stock_123, stock_123_mobile, sectors, sectors_mobile
 * - NEW: Bypass кэша для разработчика (через Cache::isDebugDeveloper)
 * - PRESERVED: Все проверки безопасности из v4.11.0
 * - PRESERVED: Обработка ошибок при include
 * 
 * Changelog v4.11.0 FINAL FIX:
 * - FIX #54 CRITICAL: ДЕЙСТВИТЕЛЬНО убрано замыкание
 *   Прямой include без замыкания для доступа к $page_data
 * 
 * ВАЖНО: Требуется cache.php v3.2.0 BULLETPROOF
 */

// ============================================================================
// КРИТИЧНАЯ ПРОВЕРКА ЗАВИСИМОСТЕЙ
// ============================================================================

// Проверяем существование критичных файлов
$required_files = [
    'cache.php' => 'Cache system',
    'includes/bot-detector.php' => 'Bot detector',
    'DB/emitents.php' => 'Database helpers'
];

foreach ($required_files as $file => $description) {
    if (!file_exists($file)) {
        http_response_code(500);
        error_log("[CONTENT.PHP v4.13.0] FATAL: Missing dependency - {$description}: {$file}");
        die("Critical system error. Please contact administrator.");
    }
}

// ============================================================================
// ИНИЦИАЛИЗАЦИЯ И БЕЗОПАСНОСТЬ
// ============================================================================

require_once('cache.php');
require_once('includes/bot-detector.php');
require_once('DB/emitents.php');
require_once('classes/BreadcrumbRenderer.php');

// Проверяем доступность критичных функций
$required_functions = ['isSearchBot', 'esc'];
foreach ($required_functions as $func) {
    if (!function_exists($func)) {
        http_response_code(500);
        error_log("[CONTENT.PHP v4.13.0] FATAL: Missing function - {$func}");
        die("Critical system error. Please contact administrator.");
    }
}

$isBot = isSearchBot();

/**
 * Безопасное логирование с санитизацией
 * 
 * @param string $message Сообщение для логирования
 * @return bool True если логирование успешно
 */
function safeLog($message) {
    if (!function_exists('error_log')) {
        return false;
    }
    $safe_message = preg_replace('/[^\x20-\x7E\t\r\n]/', '', $message);
    return error_log($safe_message);
}

// ============================================================================
// ЗАЩИТА ГЛОБАЛЬНОГО СОСТОЯНИЯ
// ============================================================================

define('CONTENT_MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('CONTENT_ALLOWED_TYPES', ['stock', 'bond', 'emitent', 'sector', 'sectors']);

// ============================================================================
// ПРОВЕРКА ОБЪЕКТОВ ИЗ INDEX.PHP
// ============================================================================

$hasError = false;
$errorMessage = '';

$_initial_error_state = $hasError;

// Проверяем наличие критичных переменных из index.php
if (!isset($page_type)) {
    $hasError = true;
    $errorMessage = 'Отсутствует определение типа страницы';
    safeLog("[CONTENT.PHP v4.13.0] \$page_type не установлен");
}

if (!isset($page_data)) {
    $hasError = true;
    $errorMessage = 'Отсутствуют данные страницы';
    safeLog("[CONTENT.PHP v4.13.0] \$page_data не установлен");
}

if (!isset($db)) {
    $hasError = true;
    $errorMessage = 'Отсутствует подключение к базе данных';
    safeLog("[CONTENT.PHP v4.13.0] \$db не установлен");
}

if (!isset($is_mobile)) {
    $is_mobile = false;
    safeLog("[CONTENT.PHP v4.13.0] WARNING: \$is_mobile не установлен, использую false");
}

// Дополнительная проверка $page_type на безопасность
if (!$hasError && isset($page_type)) {
    // Приведение к нижнему регистру для единообразия
    $page_type = strtolower($page_type);
    
    // Удаление опасных символов
    $page_type = preg_replace('/[^a-z0-9_-]/', '', $page_type);
    
    // Проверка длины
    if (strlen($page_type) > 50) {
        $hasError = true;
        $errorMessage = 'Некорректный формат типа страницы';
        safeLog("[CONTENT.PHP v4.13.0] page_type превышает допустимую длину: " . strlen($page_type));
    }
    
    // Проверка на пустоту после санитизации
    if (empty($page_type)) {
        $hasError = true;
        $errorMessage = 'Некорректный формат типа страницы';
        safeLog("[CONTENT.PHP v4.13.0] page_type пустой после санитизации");
    }
}

?>
<!-- Основной контент страницы -->
<main id="main-content" class="content">
    <div class="content-wrapper">
        
        <!-- ================================================================
             ХЛЕБНЫЕ КРОШКИ И SCHEMA.ORG РАЗМЕТКА
             ================================================================ -->
        <?php
        // DEBUG режим
        if (defined('DEBUG_MODE') && DEBUG_MODE && !empty($page_type)) {
            echo "\n<!-- DEBUG: page_type=" . esc($page_type) . " -->\n";
        }

        // Используем $breadcrumb из index.php
        if (isset($breadcrumb) && is_object($breadcrumb)) {
            try {
                $renderer = new BreadcrumbRenderer();
                echo $renderer->renderHtml($breadcrumb->getBreadcrumbs(), $is_mobile ?? false);
                
                if (!empty($page_data)) {
                    echo "\n<!-- DEBUG: page_data[0] = " . htmlspecialchars($page_data[0] ?? 'NOT SET') . " -->\n";
                    echo "<!-- DEBUG: page_data count = " . count($page_data) . " -->\n";
                    
                    echo $breadcrumb->getJsonLdEnhanced($page_data);
                }
                
            } catch (Throwable $e) {
                safeLog("[CONTENT.PHP v4.13.0] Ошибка BreadcrumbBuilder: " . $e->getMessage());
                
                if (defined('DEBUG_MODE') && DEBUG_MODE) {
                    echo '<div class="alert alert-warning">Ошибка генерации хлебных крошек: ' . 
                         esc($e->getMessage()) . '</div>';
                }
            }
        } else {
            safeLog("[CONTENT.PHP v4.13.0] WARNING: \$breadcrumb не доступен из index.php");
            
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                echo '<div class="alert alert-warning">BreadcrumbBuilder недоступен</div>';
            }
        }
        ?>
        
        <div class="content-spacing">
        
        <?php
        // ----------------------------------------------------------------
        // РОУТИНГ КОНТЕНТА В ЗАВИСИМОСТИ ОТ ТИПА СТРАНИЦЫ
        // ----------------------------------------------------------------
        
        if (!$hasError && isset($page_type)) {
            $include_file = null;
            
            // Определяем путь к файлу контента на основе типа страницы
            switch ($page_type) {
                
                // Инструменты
                case 'bond':
                    $filename = 'bond.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'stock':
                    $filename = 'stock.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'receipt':
                    $filename = 'receipt.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'eurobond':
                    $filename = 'eurobond.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'fund':
                    $filename = 'fund.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'mortgage_note':
                    $filename = 'mortgage-note.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                
                    
                case 'menu':
                    if (isset($page_data[1]) && 
                        is_object($page_data[1]) && 
                        !empty($page_data[1]->MENU_file_include)) {
                        
                        $filename = basename($page_data[1]->MENU_file_include);
                        $include_file = 'site/content/' . $filename;
                    }
                    break;
                    
                case 'sector':
                    $filename = 'sector.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                case 'emitent':
                    $filename = 'emitent.php';
                    $include_file = 'site/content/' . $filename;
                    break;
                    
                default:
                    $hasError = true;
                    $errorMessage = 'Неизвестный тип страницы';
                    safeLog("[CONTENT.PHP v4.13.0] Неизвестный page_type: " . esc($page_type));
            }
            
            // ----------------------------------------------------------------
            // ВКЛЮЧЕНИЕ ФАЙЛА С HTML КЭШИРОВАНИЕМ И БЕЗОПАСНОСТЬЮ
            // ----------------------------------------------------------------
            
            if (!$hasError && $include_file) {
                
                if (file_exists($include_file)) {
                    
                    // Проверка на symbolic link
                    if (is_link($include_file)) {
                        $hasError = true;
                        $errorMessage = 'Доступ запрещён';
                        safeLog("[CONTENT.PHP v4.13.0] SECURITY: Symlink attack blocked - " . $include_file);
                        
                    // Проверка что это файл
                    } elseif (!is_file($include_file)) {
                        $hasError = true;
                        $errorMessage = 'Некорректный тип ресурса';
                        safeLog("[CONTENT.PHP v4.13.0] SECURITY: Not a file - " . $include_file);
                        
                    } else {
                        
                        // Защита от DoS
                        $file_size = @filesize($include_file);
                        
                        if ($file_size === false) {
                            $hasError = true;
                            $errorMessage = 'Ошибка чтения файла';
                            safeLog("[CONTENT.PHP v4.13.0] Filesize error: " . $include_file);
                            
                        } elseif ($file_size > CONTENT_MAX_FILE_SIZE) {
                            $hasError = true;
                            $errorMessage = 'Запрашиваемый файл слишком большой';
                            safeLog("[CONTENT.PHP v4.13.0] DoS attempt: file=" . $include_file . ", size=" . $file_size);
                            
                        } else {
                            
                            // ============================================
                            // HTML КЭШИРОВАНИЕ (v4.12.0)
                            // ============================================
                            
                            // Генерируем умный ключ кэша
                            $cache_key = $page_type;
                            
                            // Для страниц с ID добавляем ID в ключ
                            if (in_array($page_type, ['stock', 'bond', 'emitent', 'sector']) && 
                                isset($page_data[1]) && 
                                is_object($page_data[1]) && 
                                !empty($page_data[1]->Id)) {
                                $cache_key .= '_' . $page_data[1]->Id;
                            }
                            
                            // Для мобильной версии отдельный кэш
                            if ($is_mobile) {
                                $cache_key .= '_mobile';
                            }
                            
                            // Начинаем кэширование HTML
                            if (Cache::begin($cache_key)) {
                                // Кэша нет или разработчик - генерируем контент
                                
                                safeLog("[CONTENT.PHP v4.13.0] Генерация контента для cache_key: " . $cache_key);
                                
                                // Error handler для безопасного include
                                $error_occurred = false;
                                $error_message = '';
                                
                                set_error_handler(function($errno, $errstr, $errfile, $errline) use (&$error_occurred, &$error_message) {
                                    $error_occurred = true;
                                    $error_message = $errstr;
                                    safeLog("[CONTENT.PHP v4.13.0] Include error: {$errstr} in {$errfile}:{$errline}");
                                    return true;
                                });
                                
                                // Прямой include БЕЗ замыкания
                                // Переменные доступны: $page_data, $is_mobile, $db, $breadcrumb
                                $include_result = include($include_file);
                                
                                restore_error_handler();
                                
                                // Обработка ошибок
                                if ($error_occurred || ($include_result === false && !file_exists($include_file))) {
                                    // Ошибка - не сохраняем кэш
                                    ob_end_clean(); // Очищаем буфер без сохранения
                                    
                                    $hasError = true;
                                    $errorMessage = $error_occurred ? 'Ошибка выполнения страницы' : 'Файл был удалён во время обработки';
                                    safeLog("[CONTENT.PHP v4.13.0] Include failed: " . $include_file . " - " . $error_message);
                                } else {
                                    // Успех - сохраняем в кэш и выводим
                                    Cache::end();
                                    
                                    safeLog("[CONTENT.PHP v4.13.0] HTML кэш сохранён для: " . $cache_key);
                                }
                                
                                // Проверка изменения глобального состояния
                                if ($hasError !== $_initial_error_state && !$error_occurred) {
                                    safeLog("[CONTENT.PHP v4.13.0] WARNING: Include modified \$hasError");
                                }
                            } else {
                                // Кэш был найден и выведен Cache::begin()
                                safeLog("[CONTENT.PHP v4.13.0] ⚡ Загружено из HTML кэша: " . $cache_key);
                            }
                        }
                    }
                    
                } else {
                    // Файл не найден
                    $hasError = true;
                    $errorMessage = 'Запрашиваемая страница не найдена';
                    
                    // Оптимизация памяти
                    $user_agent_safe = 'unknown';
                    if (isset($_SERVER['HTTP_USER_AGENT'])) {
                        $ua_chunk = substr($_SERVER['HTTP_USER_AGENT'], 0, 100);
                        $user_agent_safe = preg_replace('/[^\x20-\x7E]/', '', $ua_chunk);
                        unset($ua_chunk);
                    }
                    
                    $log_message = "[CONTENT.PHP v4.13.0] Ошибка 404: page_type=" . 
                        ($page_type ?: 'NULL') . ", is_mobile=" . 
                        ($is_mobile ? 'true' : 'false') . ", file=" . 
                        ($include_file ?: 'NULL') . ", ua=" . 
                        $user_agent_safe;
                    
                    safeLog($log_message);
                }
            }
        }
        
        // ----------------------------------------------------------------
        // ВЫВОД ОШИБОК
        // ----------------------------------------------------------------
        
        if ($hasError) {
            ?>
            <div class="alert alert-danger" role="alert">
                <h4 class="alert-heading">Ошибка</h4>
                <hr>
                <p class="mb-0"><?php echo esc($errorMessage); ?></p>
            </div>
            <?php
        }
        ?>
        
        </div><!-- .content-spacing -->
        
    </div><!-- .content-wrapper -->
</main><!-- .site-main -->

<?php
/**
 * ═══════════════════════════════════════════════════════════════════════════
 * END content.php v4.13.0 STOCKS_CATALOG ✅
 * ═══════════════════════════════════════════════════════════════════════════
 * 
 * Версия: 4.13.0 STOCKS_CATALOG
 * Дата: 2025-10-22
 * Статус: PRODUCTION READY
 * 
 * НОВОЕ v4.13.0:
 *   ✅ Поддержка каталога акций (stocks_catalog)
 *   ✅ Маршрутизация на site/content/stocks-catalog.php
 *   ✅ HTML кэширование для каталога (как и для других страниц)
 * 
 * НОВОЕ v4.12.0:
 *   ✅ HTML кэширование через Cache::begin/end
 *   ✅ Умный ключ кэша (page_type + ID + mobile)
 *   ✅ Bypass для разработчика (автоматически)
 *   ✅ 10-100× ускорение повторных запросов
 *   ✅ Защита от параллельной генерации HTML
 * 
 * ПРОИЗВОДИТЕЛЬНОСТЬ:
 *   - Первый запрос: ~0.3 сек (генерация + сохранение)
 *   - Последующие: ~0.001 сек (чтение из кэша)
 *   - 10 параллельных: 1× генерация + 9× кэш
 * 
 * ТРЕБОВАНИЯ:
 *   - cache.php v3.2.0 BULLETPROOF
 *   - PHP 8.1+
 * 
 * ═══════════════════════════════════════════════════════════════════════════
 */