/**
 * DEEPMAX CORE v1.0 - Universal features (always loaded)
 * deepmax.ru project
 * @version 1.0 | @date 2025-11-03 | @status PRODUCTION READY ✅
 */

(function() {
    'use strict';
    console.log('[DeepMax Core] v1.0 - Initializing...');
    

  // [SECTION: Lazy Load Images - from lazyload.js]
  // (IIFE wrapper removed for merging)

  // Wait for full DOM load AND CSS to be ready
  function initLazyLoadWhenReady() {
    // v7.2: Wait for CSS to load before initializing
    if (!document.body.classList.contains('css-ready')) {
      setTimeout(initLazyLoadWhenReady, 50);
      return;
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // INITIALIZATION: Find all lazy-load images
    // ═══════════════════════════════════════════════════════════════════════
    const lazyImages = document.querySelectorAll('.lazy-load');
    
    console.log("[LazyLoad] Images found:", lazyImages.length);
    
    // Early exit if no images to load
    if (lazyImages.length === 0) {
      console.warn("[LazyLoad] No images with .lazy-load class found");
      return;
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // INTERSECTION OBSERVER: Detect when images enter viewport
    // ═══════════════════════════════════════════════════════════════════════
    const imageObserver = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        
        // Check if image is visible in viewport
        if (entry.isIntersecting) {
          const img = entry.target;
          
          console.log("[LazyLoad] Loading image:", img.dataset.src);
          
          // ═════════════════════════════════════════════════════════════════
          // BLUR-UP EFFECT: Initial styling before image loads
          // ═════════════════════════════════════════════════════════════════
          img.style.filter = "blur(8px)";           // Start with blur
          img.style.transition = "filter 0.6s ease, opacity 0.6s ease";
          img.style.opacity = "0";                  // Start invisible
          
          // ═════════════════════════════════════════════════════════════════
          // IMAGE LOADING: Set source and wait for load
          // ═════════════════════════════════════════════════════════════════
          img.src = img.dataset.src;
          
          // ═════════════════════════════════════════════════════════════════
          // ONLOAD HANDLER: Apply fade-in and remove blur when loaded
          // ═════════════════════════════════════════════════════════════════
          img.onload = () => {
            img.classList.add('loaded');            // Trigger CSS fade-in
            img.style.filter = "blur(0)";           // Remove blur effect
            img.style.opacity = "1";                // Make fully visible
            img.classList.remove('lazy-load');      // No longer lazy
            
            console.log("[LazyLoad] Successfully loaded:", img.src);
          };
          
          // ═════════════════════════════════════════════════════════════════
          // CLEANUP: Stop observing this image (performance optimization)
          // ═════════════════════════════════════════════════════════════════
          imageObserver.unobserve(img);
        }
      });
    });
    
    // ═══════════════════════════════════════════════════════════════════════
    // START OBSERVING: Attach observer to all lazy-load images
    // ═══════════════════════════════════════════════════════════════════════
    lazyImages.forEach(img => {
      console.log("[LazyLoad] Observing:", img.dataset.src);
      imageObserver.observe(img);
    });
    
  } // END initLazyLoadWhenReady
  
  // Start when DOM is ready
  document.addEventListener('DOMContentLoaded', initLazyLoadWhenReady);
  // [END SECTION: Lazy Load Images]



    // [SECTION: Breadcrumb Animation - from breadcrumb-animator.js]
    // (IIFE wrapper removed for merging)
    
    // Wait for full DOM load
    document.addEventListener('DOMContentLoaded', function() {
        
        // ═══════════════════════════════════════════════════════════════════════
        // CHECK: Works only on desktop
        // ═══════════════════════════════════════════════════════════════════════
        const isMobile = window.innerWidth <= 768;
        if (isMobile) {
            console.log('📱 Breadcrumb animator: Disabled on mobile device');
            return; // Exit, don't initialize animation
        }
        
        // ═══════════════════════════════════════════════════════════════════════
        // INITIALIZATION
        // ═══════════════════════════════════════════════════════════════════════
        const breadcrumbNav = document.querySelector('.breadcrumbs');
        const indicator = document.querySelector('.breadcrumb-indicator');
        
        // Check elements exist
        if (!breadcrumbNav || !indicator) {
            console.warn('⚠️ Breadcrumb animator: Elements not found');
            return;
        }
        
        const links = breadcrumbNav.querySelectorAll('a');
        
        if (links.length === 0) {
            console.warn('⚠️ Breadcrumb animator: No links in breadcrumbs');
            return;
        }
        
        console.log('✅ Breadcrumb animator: Initialized (' + links.length + ' links)');
        
        // ═══════════════════════════════════════════════════════════════════════
        // FUNCTION: Animate indicator
        // ═══════════════════════════════════════════════════════════════════════
        function animateIndicator(link) {
            const linkRect = link.getBoundingClientRect();
            const navRect = breadcrumbNav.getBoundingClientRect();
            
            // Calculate position and width
            const left = linkRect.left - navRect.left;
            const width = linkRect.width;
            
            // Apply styles
            indicator.style.left = left + 'px';
            indicator.style.width = width + 'px';
            indicator.classList.add('active');
        }
        
        // ═══════════════════════════════════════════════════════════════════════
        // FUNCTION: Hide indicator
        // ═══════════════════════════════════════════════════════════════════════
        function hideIndicator() {
            indicator.style.width = '0';
            indicator.classList.remove('active');
        }
        
        // ═══════════════════════════════════════════════════════════════════════
        // EVENT DELEGATION: Optimization (instead of addEventListener on each link)
        // ═══════════════════════════════════════════════════════════════════════
        breadcrumbNav.addEventListener('mouseover', function(e) {
            // Check that this is a link
            const link = e.target.closest('a');
            if (link && breadcrumbNav.contains(link)) {
                animateIndicator(link);
            }
        });
        
        breadcrumbNav.addEventListener('mouseout', function(e) {
            const link = e.target.closest('a');
            if (link && breadcrumbNav.contains(link)) {
                hideIndicator();
            }
        });
        
        // ═══════════════════════════════════════════════════════════════════════
        // ACCESSIBILITY: Tab navigation support
        // ═══════════════════════════════════════════════════════════════════════
        breadcrumbNav.addEventListener('focusin', function(e) {
            if (e.target.tagName === 'A') {
                animateIndicator(e.target);
            }
        });
        
        breadcrumbNav.addEventListener('focusout', function(e) {
            if (e.target.tagName === 'A') {
                hideIndicator();
            }
        });
        
        // ═══════════════════════════════════════════════════════════════════════
        // RESIZE HANDLER: Recalculate on window resize
        // ═══════════════════════════════════════════════════════════════════════
        let resizeTimer;
        window.addEventListener('resize', function() {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(function() {
                // If switched to mobile size - hide indicator
                if (window.innerWidth <= 768) {
                    hideIndicator();
                }
            }, 250);
        });
        
    }); // END DOMContentLoaded
    // [END SECTION: Breadcrumb Animation]



    // [SECTION: Mobile Menu - from menu-v7.js]
    // (IIFE wrapper removed for merging)
    
    // Check for duplicate initialization
    if (window.MenuV7 && window.MenuV7.initialized) {
        console.warn('[MenuV7] Already initialized');
        return;
    }
    
    // Array for tracking event handlers
    var eventHandlers = [];
    
    // ═══════════════════════════════════════════════════════════════════════
    // UTILITY: Add event listener with tracking
    // ═══════════════════════════════════════════════════════════════════════
    function addTrackedEventListener(element, event, handler) {
        if (!element) return;
        
        element.addEventListener(event, handler);
        eventHandlers.push({
            element: element,
            event: event,
            handler: handler
        });
    }
    
    // ═══════════════════════════════════════════════════════════════════════
    // MAIN INITIALIZATION
    // ═══════════════════════════════════════════════════════════════════════
    function init() {
        console.log('[MenuV7] Initializing navigation menu v7.1...');
        
        // ═══════════════════════════════════════════════════════════════════
        // GET ELEMENTS
        // ═══════════════════════════════════════════════════════════════════
        var navToggle = document.querySelector('.nav-toggle');
        var navMenu = document.querySelector('.nav-menu');
        var navOverlay = document.querySelector('.nav-overlay');
        
        // Check for required elements
        if (!navToggle) {
            console.warn('[MenuV7] ⚠️  Toggle button not found (.nav-toggle)');
            return;
        }
        
        if (!navMenu) {
            console.warn('[MenuV7] ⚠️  Menu not found (.nav-menu)');
            return;
        }
        
        if (!navOverlay) {
            console.warn('[MenuV7] ⚠️  Overlay not found (.nav-overlay) - continuing without it');
        }
        
        console.log('[MenuV7] ✅ Elements found');
        
        // ═══════════════════════════════════════════════════════════════════
        // FUNCTION: Open menu
        // ═══════════════════════════════════════════════════════════════════
        function openMenu() {
            navToggle.setAttribute('aria-expanded', 'true');
            navMenu.classList.add('active');
            
            if (navOverlay) {
                navOverlay.classList.add('active');
            }
            
            // Block body scroll on mobile
            document.body.style.overflow = 'hidden';
            
            console.log('[MenuV7] Menu opened');
        }
        
        // ═══════════════════════════════════════════════════════════════════
        // FUNCTION: Close menu
        // ═══════════════════════════════════════════════════════════════════
        function closeMenu() {
            navToggle.setAttribute('aria-expanded', 'false');
            navMenu.classList.remove('active');
            
            if (navOverlay) {
                navOverlay.classList.remove('active');
            }
            
            // Restore body scroll
            document.body.style.overflow = '';
            
            console.log('[MenuV7] Menu closed');
        }
        
        // ═══════════════════════════════════════════════════════════════════
        // FUNCTION: Toggle menu
        // ═══════════════════════════════════════════════════════════════════
        function toggleMenu() {
            var isExpanded = navToggle.getAttribute('aria-expanded') === 'true';
            
            if (isExpanded) {
                closeMenu();
            } else {
                openMenu();
            }
        }
        
        // ═══════════════════════════════════════════════════════════════════
        // HANDLER: Click on toggle button
        // ═══════════════════════════════════════════════════════════════════
        addTrackedEventListener(navToggle, 'click', function(e) {
            e.preventDefault();
            toggleMenu();
        });
        
        // ═══════════════════════════════════════════════════════════════════
        // HANDLER: Click on overlay (close menu)
        // ═══════════════════════════════════════════════════════════════════
        if (navOverlay) {
            addTrackedEventListener(navOverlay, 'click', function() {
                closeMenu();
            });
        }
        
        // ═══════════════════════════════════════════════════════════════════
        // HANDLER: ESC key (close menu)
        // ═══════════════════════════════════════════════════════════════════
        function handleEscKey(e) {
            // ESC key (support for old browsers)
            if (e.key === 'Escape' || e.keyCode === 27) {
                var isExpanded = navToggle.getAttribute('aria-expanded') === 'true';
                
                if (isExpanded) {
                    closeMenu();
                }
            }
        }
        
        addTrackedEventListener(document, 'keydown', handleEscKey);
        
        // ═══════════════════════════════════════════════════════════════════
        // HANDLER: Click on menu link (close menu on mobile)
        // ═══════════════════════════════════════════════════════════════════
        var navLinks = navMenu.querySelectorAll('.nav-link');
        
        if (navLinks.length > 0) {
            for (var i = 0; i < navLinks.length; i++) {
                (function(link) {
                    addTrackedEventListener(link, 'click', function() {
                        // Close menu only on mobile (when overlay is visible)
                        if (navOverlay && navOverlay.classList.contains('active')) {
                            closeMenu();
                        }
                    });
                })(navLinks[i]);
            }
            
            console.log('[MenuV7] Event handlers added for ' + navLinks.length + ' links');
        }
        
        // ═══════════════════════════════════════════════════════════════════
        // HANDLER: Resize (close menu when switching to desktop)
        // ═══════════════════════════════════════════════════════════════════
        var resizeTimer;
        
        function handleResize() {
            clearTimeout(resizeTimer);
            
            resizeTimer = setTimeout(function() {
                // If switched to desktop size (> 768px) - close menu
                if (window.innerWidth > 768) {
                    var isExpanded = navToggle.getAttribute('aria-expanded') === 'true';
                    
                    if (isExpanded) {
                        closeMenu();
                    }
                }
            }, 250);
        }
        
        addTrackedEventListener(window, 'resize', handleResize);
        
        // ═══════════════════════════════════════════════════════════════════
        // CLEANUP: Remove handlers on page unload
        // ═══════════════════════════════════════════════════════════════════
        function cleanup() {
            console.log('[MenuV7] Cleanup: removing ' + eventHandlers.length + ' handlers');
            
            // Restore overflow if menu was open
            document.body.style.overflow = '';
            
            // Remove all handlers
            for (var i = 0; i < eventHandlers.length; i++) {
                var handler = eventHandlers[i];
                try {
                    handler.element.removeEventListener(handler.event, handler.handler);
                } catch (e) {
                    // Ignore errors during removal
                }
            }
            
            // Clear array
            eventHandlers.length = 0;
            
            // Clear timer
            if (resizeTimer) {
                clearTimeout(resizeTimer);
            }
        }
        
        addTrackedEventListener(window, 'beforeunload', cleanup);
        
        // ═══════════════════════════════════════════════════════════════════
        // EXPORT API
        // ═══════════════════════════════════════════════════════════════════
        window.MenuV7 = {
            version: '7.1',
            initialized: true,
            openMenu: openMenu,
            closeMenu: closeMenu,
            toggleMenu: toggleMenu,
            cleanup: cleanup
        };
        
        console.log('[MenuV7] ✅ Initialization completed successfully');
        
    } // END init()
    
    // ═══════════════════════════════════════════════════════════════════════
    // STARTUP - v7.2: Wait for CSS before initializing menu
    // ═══════════════════════════════════════════════════════════════════════
    function initMenuWhenReady() {
        // Wait for CSS to load
        if (!document.body.classList.contains('css-ready')) {
            setTimeout(initMenuWhenReady, 50);
            return;
        }
        init();
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initMenuWhenReady);
    } else {
        initMenuWhenReady();
    }
    // [END SECTION: Mobile Menu]


    // [SECTION: Site.js Functions - Back to Top + Lazy Load Sectors]
    // (Selected functions extracted for merging)

function initSectorLazyLoading() {
  const cache = new Map();
  const states = new Map();
  const elements = new Map();
  const queue = [];
  let activeRequests = 0;
  let queueTimer = null;
  let textObserver = null;
  const MAX_CONCURRENT = 3;
  
  // Check IntersectionObserver support
  if (!('IntersectionObserver' in window)) {
    console.warn('IntersectionObserver is not supported');
    return;
  }
  
  // ═══════════════════════════════════════════════════════════════════════
  // LOAD QUEUE MANAGEMENT
  // ═══════════════════════════════════════════════════════════════════════
  
  function processQueue() {
    if (queueTimer) {
      clearTimeout(queueTimer);
      queueTimer = null;
    }
    
    while (queue.length > 0 && activeRequests < MAX_CONCURRENT) {
      const sectorId = queue.shift();
      
      if (states.get(sectorId) !== 'queued') continue;
      
      const data = elements.get(sectorId);
      if (!data) {
        states.delete(sectorId);
        continue;
      }
      
      executeLoad(sectorId);
    }
  }
  
  function scheduleQueueProcessing() {
    if (queueTimer) return;
    queueTimer = setTimeout(processQueue, 50);
  }
  
  function enqueue(sectorId) {
    const state = states.get(sectorId);
    if (state === 'queued' || state === 'loading' || state === 'loaded') {
      return false;
    }
    
    states.set(sectorId, 'queued');
    queue.push(sectorId);
    return true;
  }
  
  // ═══════════════════════════════════════════════════════════════════════
  // LOAD SECTOR DESCRIPTION VIA API
  // ═══════════════════════════════════════════════════════════════════════
  
  function executeLoad(sectorId) {
    const data = elements.get(sectorId);
    if (!data) return;
    
    const { container, contentDiv, btn } = data;
    
    // Check cache
    if (cache.has(sectorId)) {
      contentDiv.innerHTML = cache.get(sectorId);
      contentDiv.style.display = 'block';
      if (btn) btn.style.display = 'none';
      states.set(sectorId, 'loaded');
      console.log('✔ Sector #' + sectorId + ' loaded from cache');
      return;
    }
    
    states.set(sectorId, 'loading');
    activeRequests++;
    
    if (btn) {
      btn.textContent = 'Loading...';
      btn.disabled = true;
    }
    
    console.log('⟳ Loading sector #' + sectorId + ' description via API...');
    
    fetch('/api/load-sector-description.php?id=' + sectorId)
      .then(function(response) {
        if (!response.ok) throw new Error('HTTP ' + response.status);
        return response.text();
      })
      .then(function(html) {
        if (!html || html.trim().length === 0) throw new Error('Empty response');
        
        cache.set(sectorId, html);
        
        if (container.isConnected) {
          contentDiv.innerHTML = html;
          contentDiv.style.display = 'block';
          if (btn) btn.style.display = 'none';
          console.log('✔ Sector #' + sectorId + ' loaded and displayed successfully');
        }
        
        states.set(sectorId, 'loaded');
      })
      .catch(function(error) {
        console.error('✘ Error loading sector #' + sectorId + ':', error);
        states.set(sectorId, 'error');
        
        if (container.isConnected && btn) {
          btn.textContent = 'Error. Click to retry';
          btn.disabled = false;
        }
      })
      .finally(function() {
        activeRequests--;
        scheduleQueueProcessing();
      });
  }
  
  function loadDescription(sectorId) {
    if (!sectorId) return;
    
    const state = states.get(sectorId);
    
    if (state === 'loaded') {
      const data = elements.get(sectorId);
      if (data) {
        data.contentDiv.style.display = 'block';
        if (data.btn) data.btn.style.display = 'none';
      }
      return;
    }
    
    if (enqueue(sectorId)) {
      processQueue();
    }
  }
  
  // ═══════════════════════════════════════════════════════════════════════
  // INTERSECTION OBSERVER FOR SECTOR CARDS
  // ═══════════════════════════════════════════════════════════════════════
  
  const observer = new IntersectionObserver(function(entries) {
    entries.forEach(function(entry) {
      if (entry.isIntersecting) {
        const sectorId = entry.target.dataset.sectorId;
        if (sectorId) {
          console.log('▶ Sector #' + sectorId + ' entered viewport');
          loadDescription(sectorId);
          observer.unobserve(entry.target);
        }
      }
    });
  }, {
    rootMargin: '200px',
    threshold: 0.01
  });
  
  // ═══════════════════════════════════════════════════════════════════════
  // INITIALIZE SECTOR CARDS
  // ═══════════════════════════════════════════════════════════════════════
  
  const containers = document.querySelectorAll('.sector-description');
  const priorityIds = [];
  
  if (containers.length > 0) {
    console.log('✔ Found ' + containers.length + ' sector descriptions for lazy loading');
  }
  
  containers.forEach(function(container, index) {
    const sectorId = container.dataset.sectorId;
    if (!sectorId) return;
    
    const contentDiv = container.querySelector('.description-content');
    const btn = container.querySelector('.btn-load-description');
    
    if (!contentDiv) return;
    
    elements.set(sectorId, { container: container, contentDiv: contentDiv, btn: btn });
    states.set(sectorId, null);
    
    // Load button handler
    if (btn) {
      btn.addEventListener('click', function() {
        console.log('👆 Click on sector #' + sectorId + ' button');
        const currentState = states.get(sectorId);
        if (currentState === 'error') {
          states.set(sectorId, null);
        }
        loadDescription(sectorId);
      });
    }
    
    // Priority load for first 2 sectors
    if (index < 2) {
      priorityIds.push(sectorId);
    } else {
      observer.observe(container);
    }
  });
  
  // Load first 2 sectors with delay
  if (priorityIds.length > 0) {
    console.log('🚀 Priority loading first ' + priorityIds.length + ' sectors');
  }
  
  priorityIds.forEach(function(sectorId, index) {
    setTimeout(function() {
      loadDescription(sectorId);
    }, index * 100);
  });
  
  // ═══════════════════════════════════════════════════════════════════════
  // LAZY LOADING FOR BOTTOM PAGE TEXT
  // FIX v1.1.0: Support for <details open>
  // ═══════════════════════════════════════════════════════════════════════
  
  const bottomText = document.querySelector('.sector-bottom-description');
  if (bottomText) {
    console.log('✔ Found bottom page text block');
    
    const sectorId = bottomText.dataset.sectorId;
    const loadFromApi = bottomText.dataset.loadFrom === 'api';
    
    // ========================================================================
    // NEW: Check parent <details> - is it open by default?
    // ========================================================================
    const parentDetails = bottomText.closest('details');
    const isDetailsOpen = parentDetails && parentDetails.hasAttribute('open');
    
    // Text loading function (extracted for reusability)
    function loadBottomText() {
      console.log('▶ Bottom page text - starting load');
      
      const placeholder = bottomText.querySelector('.text-placeholder');
      const content = bottomText.querySelector('.text-content');
      
      if (!content) {
        console.warn('⚠ .text-content not found');
        return;
      }
      
      // If text needs to be loaded via API
      if (loadFromApi && content.innerHTML.trim() === '') {
        console.log('⟳ Loading text via API (sector #' + sectorId + ')...');
        
        fetch('/api/load-sector-full-text.php?id=' + sectorId)
          .then(function(response) {
            if (!response.ok) throw new Error('HTTP ' + response.status);
            return response.text();
          })
          .then(function(html) {
            if (!html || html.trim().length === 0) throw new Error('Empty response');
            
            content.innerHTML = html;
            content.style.display = 'block';
            if (placeholder) placeholder.style.display = 'none';
            console.log('✔ Text loaded and displayed successfully (' + html.length + ' characters)');
          })
          .catch(function(error) {
            console.error('✘ Error loading text:', error);
            content.innerHTML = '<p class="text-muted">Error loading text</p>';
            content.style.display = 'block';
            if (placeholder) placeholder.style.display = 'none';
          });
      } else {
        // Text already in HTML, just show it
        content.style.display = 'block';
        if (placeholder) placeholder.style.display = 'none';
        console.log('✔ Text displayed (from HTML, for bot)');
      }
    }
    
    // ========================================================================
    // NEW: If <details> is OPEN by default - load IMMEDIATELY
    // ========================================================================
    if (isDetailsOpen) {
      console.log('🚀 <details> open by default - loading text immediately');
      // Small delay to let page render
      setTimeout(function() {
        loadBottomText();
      }, 100);
    } 
    // Otherwise use IntersectionObserver (when <details> is closed)
    else {
      console.log('👁 <details> closed - using IntersectionObserver');
      
      textObserver = new IntersectionObserver(function(entries) {
        entries.forEach(function(entry) {
          if (entry.isIntersecting) {
            loadBottomText();
            textObserver.unobserve(entry.target);
          }
        });
      }, {
        rootMargin: '0px',
        threshold: 0.01
      });
      
      textObserver.observe(bottomText);
    }
  } else {
    console.log('ℹ Bottom page text block not found (/sectors/ page)');
  }
  
  // ═══════════════════════════════════════════════════════════════════════
  // CLEANUP
  // ═══════════════════════════════════════════════════════════════════════
  
  function cleanup() {
    observer.disconnect();
    if (textObserver) textObserver.disconnect();
    if (queueTimer) clearTimeout(queueTimer);
    queue.length = 0;
    cache.clear();
    states.clear();
    elements.clear();
    console.log('✔ Lazy loading cleanup completed');
  }
  
  window.addEventListener('beforeunload', cleanup);
}

function initBackToTop() {
  const button = document.getElementById('back-to-top');
  
  if (!button) {
    console.log('ℹ Back to top button not found in DOM');
    return;
  }
  
  console.log('✔ Initializing back to top button');
  
  // Show/hide button based on scroll position
  function updateButtonVisibility() {
    if (window.pageYOffset > 500) {
      button.classList.add('visible');
      button.setAttribute('aria-hidden', 'false');
    } else {
      button.classList.remove('visible');
      button.setAttribute('aria-hidden', 'true');
    }
  }
  
  // Smooth scroll to top
  function scrollToTop(e) {
    e.preventDefault();
    
    // Use smooth scrolling
    window.scrollTo({
      top: 0,
      behavior: 'smooth'
    });
    
    console.log('⬆️ Scrolling to top');
    
    // Focus on skip-link after scroll (accessibility)
    setTimeout(function() {
      const skipLink = document.querySelector('.skip-link');
      if (skipLink) {
        skipLink.focus();
      }
    }, 500);
  }
  
  // Event listeners
  window.addEventListener('scroll', updateButtonVisibility);
  button.addEventListener('click', scrollToTop);
  
  // Initial check
  updateButtonVisibility();
  
  // Cleanup on page unload
  window.addEventListener('beforeunload', function() {
    window.removeEventListener('scroll', updateButtonVisibility);
    button.removeEventListener('click', scrollToTop);
  });
}
    // [END SECTION: Site.js Functions]
    
    // Init on load
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            initBackToTop();
            initSectorLazyLoading();
        });
    } else {
        initBackToTop();
        initSectorLazyLoading();
    }
    
    console.log('[DeepMax Core] ✅ Initialized');
})();